"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Port = exports.Protocol = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Protocol for use in Connection Rules
 */
var Protocol;
(function (Protocol) {
    Protocol["ALL"] = "-1";
    Protocol["TCP"] = "tcp";
    Protocol["UDP"] = "udp";
    Protocol["ICMP"] = "icmp";
    Protocol["ICMPV6"] = "58";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Interface for classes that provide the connection-specification parts of a security group rule
 */
class Port {
    constructor(props) {
        this.props = props;
        this.canInlineRule = !core_1.Token.isUnresolved(props.fromPort) && !core_1.Token.isUnresolved(props.toPort);
    }
    /**
     * A single TCP port
     */
    static tcp(port) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: port,
            toPort: port,
            stringRepresentation: renderPort(port),
        });
    }
    /**
     * A TCP port range
     */
    static tcpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any TCP traffic
     */
    static allTcp() {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'ALL PORTS',
        });
    }
    /**
     * A single UDP port
     */
    static udp(port) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: port,
            toPort: port,
            stringRepresentation: `UDP ${renderPort(port)}`,
        });
    }
    /**
     * A UDP port range
     */
    static udpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `UDP ${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any UDP traffic
     */
    static allUdp() {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'UDP ALL PORTS',
        });
    }
    /**
     * A specific combination of ICMP type and code
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     */
    static icmpTypeAndCode(type, code) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: code,
            stringRepresentation: `ICMP Type ${type} Code ${code}`,
        });
    }
    /**
     * All codes for a single ICMP type
     */
    static icmpType(type) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: -1,
            stringRepresentation: `ICMP Type ${type}`,
        });
    }
    /**
     * ICMP ping (echo) traffic
     */
    static icmpPing() {
        return Port.icmpType(8);
    }
    /**
     * All ICMP traffic
     */
    static allIcmp() {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: -1,
            toPort: -1,
            stringRepresentation: 'ALL ICMP',
        });
    }
    /**
     * All traffic
     */
    static allTraffic() {
        return new Port({
            protocol: Protocol.ALL,
            stringRepresentation: 'ALL TRAFFIC',
        });
    }
    /**
     * Produce the ingress/egress rule JSON for the given connection
     */
    toRuleJson() {
        return {
            ipProtocol: this.props.protocol,
            fromPort: this.props.fromPort,
            toPort: this.props.toPort,
        };
    }
    toString() {
        return this.props.stringRepresentation;
    }
}
exports.Port = Port;
function renderPort(port) {
    return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
}
//# sourceMappingURL=data:application/json;base64,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