"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchemaTransformer = void 0;
const fs = require("fs");
const path_1 = require("path");
const graphql_auth_transformer_1 = require("graphql-auth-transformer");
const graphql_connection_transformer_1 = require("graphql-connection-transformer");
const graphql_dynamodb_transformer_1 = require("graphql-dynamodb-transformer");
const graphql_http_transformer_1 = require("graphql-http-transformer");
const graphql_key_transformer_1 = require("graphql-key-transformer");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_versioned_transformer_1 = require("graphql-versioned-transformer");
const cdk_transformer_1 = require("./cdk-transformer");
// Import this way because FunctionTransformer.d.ts types were throwing an eror. And we didn't write this package so hope for the best :P
// eslint-disable-next-line
const { FunctionTransformer } = require('graphql-function-transformer');
class SchemaTransformer {
    constructor(props) {
        this.schemaPath = props.schemaPath || './schema.graphql';
        this.outputPath = props.outputPath || './appsync';
        this.isSyncEnabled = props.syncEnabled || false;
        this.outputs = {};
        this.resolvers = {};
        // TODO: Make this better?
        this.authTransformerConfig = {
            authConfig: {
                defaultAuthentication: {
                    authenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    userPoolConfig: {
                        userPoolId: '12345xyz',
                    },
                },
                additionalAuthenticationProviders: [
                    {
                        authenticationType: 'API_KEY',
                        apiKeyConfig: {
                            description: 'Testing',
                            apiKeyExpirationDays: 100,
                        },
                    },
                    {
                        authenticationType: 'AWS_IAM',
                    },
                ],
            },
        };
    }
    transform() {
        var _a, _b;
        const transformConfig = this.isSyncEnabled ? this.loadConfigSync() : {};
        // Note: This is not exact as we are omitting the @searchable transformer as well as some others.
        const transformer = new graphql_transformer_core_1.GraphQLTransform({
            transformConfig: transformConfig,
            transformers: [
                new graphql_dynamodb_transformer_1.DynamoDBModelTransformer(),
                new graphql_versioned_transformer_1.VersionedModelTransformer(),
                new FunctionTransformer(),
                new graphql_key_transformer_1.KeyTransformer(),
                new graphql_connection_transformer_1.ModelConnectionTransformer(),
                new graphql_auth_transformer_1.ModelAuthTransformer(this.authTransformerConfig),
                new graphql_http_transformer_1.HttpTransformer(),
                new cdk_transformer_1.CdkTransformer(),
            ],
        });
        const schema = fs.readFileSync(this.schemaPath);
        const cfdoc = transformer.transform(schema.toString());
        // TODO: Get Unauth Role and Auth Role policies for authorization stuff
        this.unauthRolePolicy = ((_a = cfdoc.rootStack.Resources) === null || _a === void 0 ? void 0 : _a.UnauthRolePolicy01) || undefined;
        this.writeSchema(cfdoc.schema);
        this.writeResolversToFile(cfdoc.resolvers);
        // Outputs shouldn't be null but default to empty map
        this.outputs = (_b = cfdoc.rootStack.Outputs) !== null && _b !== void 0 ? _b : {};
        return this.outputs;
    }
    /**
       *
       */
    getResolvers() {
        const statements = ['Query', 'Mutation', 'Subscription'];
        const resolversDirPath = path_1.normalize('./appsync/resolvers');
        if (fs.existsSync(resolversDirPath)) {
            const files = fs.readdirSync(resolversDirPath);
            files.forEach(file => {
                // Example: Mutation.createChannel.response
                let args = file.split('.');
                let typeName = args[0];
                let name = args[1];
                let templateType = args[2]; // request or response
                let filepath = path_1.normalize(`${resolversDirPath}/${file}`);
                if (statements.indexOf(typeName) >= 0 || (this.outputs.noneResolvers && this.outputs.noneResolvers[name])) {
                    if (!this.resolvers[name]) {
                        this.resolvers[name] = {
                            typeName: typeName,
                            fieldName: name,
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers[name].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers[name].responseMappingTemplate = filepath;
                    }
                }
                else { // This is a GSI
                    if (!this.resolvers.gsi) {
                        this.resolvers.gsi = {};
                    }
                    let mapName = `${typeName}${name}`;
                    if (!this.resolvers.gsi[mapName]) {
                        this.resolvers.gsi[mapName] = {
                            typeName: typeName,
                            fieldName: name,
                            tableName: name.charAt(0).toUpperCase() + name.slice(1),
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers.gsi[mapName].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers.gsi[mapName].responseMappingTemplate = filepath;
                    }
                }
            });
        }
        return this.resolvers;
    }
    /**
       * Writes the schema to the output directory for use with @aws-cdk/aws-appsync
       * @param schema
       */
    writeSchema(schema) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        fs.writeFileSync(`${this.outputPath}/schema.graphql`, schema);
    }
    /**
       * Writes all the resolvers to the output directory for loading into the datasources later
       * @param resolvers
       */
    writeResolversToFile(resolvers) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        const resolverFolderPath = path_1.normalize(this.outputPath + '/resolvers');
        if (fs.existsSync(resolverFolderPath)) {
            const files = fs.readdirSync(resolverFolderPath);
            files.forEach(file => fs.unlinkSync(resolverFolderPath + '/' + file));
            fs.rmdirSync(resolverFolderPath);
        }
        if (!fs.existsSync(resolverFolderPath)) {
            fs.mkdirSync(resolverFolderPath);
        }
        Object.keys(resolvers).forEach((key) => {
            const resolver = resolvers[key];
            const fileName = key.replace('.vtl', '');
            const resolverFilePath = path_1.normalize(`${resolverFolderPath}/${fileName}`);
            fs.writeFileSync(resolverFilePath, resolver);
        });
    }
    /**
       * @returns {@link TransformConfig}
      */
    loadConfigSync(projectDir = 'resources') {
        // Initialize the config always with the latest version, other members are optional for now.
        let config = {
            Version: graphql_transformer_core_1.TRANSFORM_CURRENT_VERSION,
            ResolverConfig: {
                project: {
                    ConflictHandler: "OPTIMISTIC_CONCURRENCY" /* OPTIMISTIC */,
                    ConflictDetection: 'VERSION',
                },
            },
        };
        const configDir = path_1.join(__dirname, '..', '..', projectDir);
        try {
            const configPath = path_1.join(configDir, graphql_transformer_core_1.TRANSFORM_CONFIG_FILE_NAME);
            const configExists = fs.existsSync(configPath);
            if (configExists) {
                const configStr = fs.readFileSync(configPath);
                config = JSON.parse(configStr.toString());
            }
            return config;
        }
        catch (err) {
            return config;
        }
    }
}
exports.SchemaTransformer = SchemaTransformer;
//# sourceMappingURL=data:application/json;base64,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