"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppSyncTransformer = void 0;
const aws_appsync_1 = require("@aws-cdk/aws-appsync");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const schema_transformer_1 = require("./transformer/schema-transformer");
const defaultAuthorizationConfig = {
    defaultAuthorization: {
        authorizationType: aws_appsync_1.AuthorizationType.API_KEY,
        apiKeyConfig: {
            description: 'Auto generated API Key from construct',
            name: 'dev',
        },
    },
};
/**
 * (experimental) AppSyncTransformer Construct.
 *
 * @experimental
 */
class AppSyncTransformer extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.isSyncEnabled = props.syncEnabled ? props.syncEnabled : false;
        const transformerConfiguration = {
            schemaPath: props.schemaPath,
            syncEnabled: (_a = props.syncEnabled) !== null && _a !== void 0 ? _a : false,
        };
        const transformer = new schema_transformer_1.SchemaTransformer(transformerConfiguration);
        this.outputs = transformer.transform();
        const resolvers = transformer.getResolvers();
        this.functionResolvers = (_b = this.outputs.functionResolvers) !== null && _b !== void 0 ? _b : {};
        // Remove any function resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, functionResolvers] of Object.entries(this.functionResolvers)) {
            functionResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.httpResolvers = (_c = this.outputs.httpResolvers) !== null && _c !== void 0 ? _c : {};
        // Remove any http resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, httpResolvers] of Object.entries(this.httpResolvers)) {
            httpResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.resolvers = resolvers;
        this.nestedAppsyncStack = new core_1.NestedStack(this, 'appsync-nested-stack');
        // AppSync
        this.appsyncAPI = new aws_appsync_1.GraphqlApi(this.nestedAppsyncStack, `${id}-api`, {
            name: props.apiName ? props.apiName : `${id}-api`,
            authorizationConfig: props.authorizationConfig ? props.authorizationConfig : defaultAuthorizationConfig,
            logConfig: {
                fieldLogLevel: props.fieldLogLevel ? props.fieldLogLevel : aws_appsync_1.FieldLogLevel.NONE,
            },
            schema: aws_appsync_1.Schema.fromAsset('./appsync/schema.graphql'),
        });
        let tableData = (_d = this.outputs.cdkTables) !== null && _d !== void 0 ? _d : {};
        // Check to see if sync is enabled
        if (tableData.DataStore) {
            this.isSyncEnabled = true;
            this.syncTable = this.createSyncTable(tableData.DataStore);
            delete tableData.DataStore; // We don't want to create this again below so remove it from the tableData map
        }
        this.tableNameMap = this.createTablesAndResolvers(tableData, resolvers);
        if (this.outputs.noneResolvers)
            this.createNoneDataSourceAndResolvers(this.outputs.noneResolvers, resolvers);
        this.createHttpResolvers();
        // Outputs so we can generate exports
        new core_1.CfnOutput(scope, 'appsyncGraphQLEndpointOutput', {
            value: this.appsyncAPI.graphqlUrl,
            description: 'Output for aws_appsync_graphqlEndpoint',
        });
    }
    /**
     * Creates NONE data source and associated resolvers
     * @param noneResolvers The resolvers that belong to the none data source
     * @param resolvers The resolver map minus function resolvers
     */
    createNoneDataSourceAndResolvers(noneResolvers, resolvers) {
        const noneDataSource = this.appsyncAPI.addNoneDataSource('NONE');
        Object.keys(noneResolvers).forEach((resolverKey) => {
            const resolver = resolvers[resolverKey];
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: noneDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
            });
        });
    }
    /**
     * Creates each dynamodb table, gsis, dynamodb datasource, and associated resolvers
     * If sync is enabled then TTL configuration is added
     * Returns tableName: table map in case it is needed for lambda functions, etc
     * @param tableData The CdkTransformer table information
     * @param resolvers The resolver map minus function resolvers
     */
    createTablesAndResolvers(tableData, resolvers) {
        const tableNameMap = {};
        Object.keys(tableData).forEach((tableKey) => {
            const table = this.createTable(tableData[tableKey]);
            const dataSource = this.appsyncAPI.addDynamoDbDataSource(tableKey, table);
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appsync-datasource-deltasyncconfig.html
            if (this.isSyncEnabled && this.syncTable) {
                //@ts-ignore - ds is the base CfnDataSource and the db config needs to be versioned - see CfnDataSource
                dataSource.ds.dynamoDbConfig.versioned = true;
                //@ts-ignore - ds is the base CfnDataSource - see CfnDataSource
                dataSource.ds.dynamoDbConfig.deltaSyncConfig = {
                    baseTableTtl: '43200',
                    deltaSyncTableName: this.syncTable.tableName,
                    deltaSyncTableTtl: '30',
                };
                // Need to add permission for our datasource service role to access the sync table
                dataSource.grantPrincipal.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'dynamodb:*',
                    ],
                    resources: [
                        this.syncTable.tableArn,
                    ],
                }));
            }
            const dynamoDbConfig = dataSource.ds.dynamoDbConfig;
            tableNameMap[tableKey] = dynamoDbConfig.tableName;
            // Loop the basic resolvers
            tableData[tableKey].resolvers.forEach((resolverKey) => {
                let resolver = resolvers[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
            // Loop the gsi resolvers
            tableData[tableKey].gsiResolvers.forEach((resolverKey) => {
                let resolver = resolvers.gsi[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
        });
        return tableNameMap;
    }
    createTable(tableData) {
        let tableProps = {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
        };
        if (tableData.sortKey && tableData.sortKey.name) {
            tableProps.sortKey = {
                name: tableData.sortKey.name,
                type: this.convertAttributeType(tableData.sortKey.type),
            };
        }
        ;
        if (tableData.ttl && tableData.ttl.Enabled) {
            tableProps.timeToLiveAttribute = tableData.ttl.AttributeName;
        }
        const table = new aws_dynamodb_1.Table(this.nestedAppsyncStack, tableData.tableName, tableProps);
        if (tableData.globalSecondaryIndexes && tableData.globalSecondaryIndexes.length > 0) {
            tableData.globalSecondaryIndexes.forEach((gsi) => {
                table.addGlobalSecondaryIndex({
                    indexName: gsi.indexName,
                    partitionKey: {
                        name: gsi.partitionKey.name,
                        type: this.convertAttributeType(gsi.partitionKey.type),
                    },
                    projectionType: this.convertProjectionType(gsi.projection.ProjectionType),
                });
            });
        }
        return table;
    }
    /**
     * Creates the sync table for Amplify DataStore
     * https://docs.aws.amazon.com/appsync/latest/devguide/conflict-detection-and-sync.html
     * @param tableData The CdkTransformer table information
     */
    createSyncTable(tableData) {
        var _a;
        return new aws_dynamodb_1.Table(this, 'appsync-api-sync-table', {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            sortKey: {
                name: tableData.sortKey.name,
                type: this.convertAttributeType(tableData.sortKey.type),
            },
            timeToLiveAttribute: ((_a = tableData.ttl) === null || _a === void 0 ? void 0 : _a.AttributeName) || '_ttl',
        });
    }
    convertAttributeType(type) {
        switch (type) {
            case 'N':
                return aws_dynamodb_1.AttributeType.NUMBER;
            case 'B':
                return aws_dynamodb_1.AttributeType.BINARY;
            case 'S': // Same as default
            default:
                return aws_dynamodb_1.AttributeType.STRING;
        }
    }
    convertProjectionType(type) {
        switch (type) {
            case 'INCLUDE':
                return aws_dynamodb_1.ProjectionType.INCLUDE;
            case 'KEYS_ONLY':
                return aws_dynamodb_1.ProjectionType.KEYS_ONLY;
            case 'ALL': // Same as default
            default:
                return aws_dynamodb_1.ProjectionType.ALL;
        }
    }
    createHttpResolvers() {
        for (const [endpoint, httpResolvers] of Object.entries(this.httpResolvers)) {
            const strippedEndpoint = endpoint.replace(/[^_0-9A-Za-z]/g, '');
            const httpDataSource = this.appsyncAPI.addHttpDataSource(`${strippedEndpoint}`, endpoint);
            httpResolvers.forEach((resolver) => {
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: httpDataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
                });
            });
        }
    }
    /**
     * (experimental) Adds the function as a lambdaDataSource to the AppSync api Adds all of the functions resolvers to the AppSync api.
     *
     * @param functionName The function name specified in the.
     * @param id The id to give.
     * @param lambdaFunction The lambda function to attach.
     * @experimental
     * @function directive of the schema
     */
    addLambdaDataSourceAndResolvers(functionName, id, lambdaFunction, options) {
        const functionDataSource = this.appsyncAPI.addLambdaDataSource(id, lambdaFunction, options);
        for (const resolver of this.functionResolvers[functionName]) {
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: functionDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
            });
        }
        return functionDataSource;
    }
}
exports.AppSyncTransformer = AppSyncTransformer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwc3luYy10cmFuc2Zvcm1lci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9hcHBzeW5jLXRyYW5zZm9ybWVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNEQVc4QjtBQUU5Qix3REFBMEY7QUFDMUYsOENBQTJEO0FBRTNELHdDQUFrRTtBQVVsRSx5RUFBNkY7QUFpQzdGLE1BQU0sMEJBQTBCLEdBQXdCO0lBQ3RELG9CQUFvQixFQUFFO1FBQ3BCLGlCQUFpQixFQUFFLCtCQUFpQixDQUFDLE9BQU87UUFDNUMsWUFBWSxFQUFFO1lBQ1osV0FBVyxFQUFFLHVDQUF1QztZQUNwRCxJQUFJLEVBQUUsS0FBSztTQUNaO0tBQ0Y7Q0FDRixDQUFDOzs7Ozs7QUFLRixNQUFhLGtCQUFtQixTQUFRLGdCQUFTOzs7O0lBcUMvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQThCOztRQUN0RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1FBRW5FLE1BQU0sd0JBQXdCLEdBQTJCO1lBQ3ZELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixXQUFXLFFBQUUsS0FBSyxDQUFDLFdBQVcsbUNBQUksS0FBSztTQUN4QyxDQUFDO1FBRUYsTUFBTSxXQUFXLEdBQUcsSUFBSSxzQ0FBaUIsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ3BFLElBQUksQ0FBQyxPQUFPLEdBQUcsV0FBVyxDQUFDLFNBQVMsRUFBRSxDQUFDO1FBQ3ZDLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUU3QyxJQUFJLENBQUMsaUJBQWlCLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsbUNBQUksRUFBRSxDQUFDO1FBRTlELGlFQUFpRTtRQUNqRSxtQ0FBbUM7UUFDbkMsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLGlCQUFpQixDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsRUFBRTtZQUMzRSxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxRQUFhLEVBQUUsRUFBRTtnQkFDMUMsUUFBUSxRQUFRLENBQUMsUUFBUSxFQUFFO29CQUN6QixLQUFLLE9BQU8sQ0FBQztvQkFDYixLQUFLLFVBQVUsQ0FBQztvQkFDaEIsS0FBSyxjQUFjO3dCQUNqQixPQUFPLFNBQVMsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUM7d0JBQ3JDLE1BQU07aUJBQ1Q7WUFDSCxDQUFDLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxDQUFDLGFBQWEsU0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsbUNBQUksRUFBRSxDQUFDO1FBRXRELDZEQUE2RDtRQUM3RCxtQ0FBbUM7UUFDbkMsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLGFBQWEsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQ25FLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxRQUFhLEVBQUUsRUFBRTtnQkFDdEMsUUFBUSxRQUFRLENBQUMsUUFBUSxFQUFFO29CQUN6QixLQUFLLE9BQU8sQ0FBQztvQkFDYixLQUFLLFVBQVUsQ0FBQztvQkFDaEIsS0FBSyxjQUFjO3dCQUNqQixPQUFPLFNBQVMsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUM7d0JBQ3JDLE1BQU07aUJBQ1Q7WUFDSCxDQUFDLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxDQUFDLFNBQVMsR0FBRyxTQUFTLENBQUM7UUFFM0IsSUFBSSxDQUFDLGtCQUFrQixHQUFHLElBQUksa0JBQVcsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztRQUV4RSxVQUFVO1FBQ1YsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLHdCQUFVLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRSxNQUFNLEVBQUU7WUFDckUsSUFBSSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxNQUFNO1lBQ2pELG1CQUFtQixFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQywwQkFBMEI7WUFDdkcsU0FBUyxFQUFFO2dCQUNULGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQywyQkFBYSxDQUFDLElBQUk7YUFDOUU7WUFDRCxNQUFNLEVBQUUsb0JBQU0sQ0FBQyxTQUFTLENBQUMsMEJBQTBCLENBQUM7U0FDckQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxTQUFTLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLG1DQUFJLEVBQUUsQ0FBQztRQUU3QyxrQ0FBa0M7UUFDbEMsSUFBSSxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3ZCLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDO1lBQzFCLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDM0QsT0FBTyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsK0VBQStFO1NBQzVHO1FBRUQsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hFLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhO1lBQUUsSUFBSSxDQUFDLGdDQUFnQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzdHLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO1FBRTNCLHFDQUFxQztRQUNyQyxJQUFJLGdCQUFTLENBQUMsS0FBSyxFQUFFLDhCQUE4QixFQUFFO1lBQ25ELEtBQUssRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFVBQVU7WUFDakMsV0FBVyxFQUFFLHdDQUF3QztTQUN0RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGdDQUFnQyxDQUFDLGFBQXlELEVBQUUsU0FBYztRQUNoSCxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRWpFLE1BQU0sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBZ0IsRUFBRSxFQUFFO1lBQ3RELE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUV4QyxJQUFJLHNCQUFRLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsUUFBUSxDQUFDLFFBQVEsSUFBSSxRQUFRLENBQUMsU0FBUyxXQUFXLEVBQUU7Z0JBQzNGLEdBQUcsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDcEIsUUFBUSxFQUFFLFFBQVEsQ0FBQyxRQUFRO2dCQUMzQixTQUFTLEVBQUUsUUFBUSxDQUFDLFNBQVM7Z0JBQzdCLFVBQVUsRUFBRSxjQUFjO2dCQUMxQixzQkFBc0IsRUFBRSw2QkFBZSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsc0JBQXNCLENBQUM7Z0JBQ2pGLHVCQUF1QixFQUFFLDZCQUFlLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyx1QkFBdUIsQ0FBQzthQUNwRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSyx3QkFBd0IsQ0FBQyxTQUFrRCxFQUFFLFNBQWM7UUFDakcsTUFBTSxZQUFZLEdBQVEsRUFBRSxDQUFDO1FBRTdCLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBYSxFQUFFLEVBQUU7WUFDL0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztZQUNwRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLHFCQUFxQixDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUUxRSx3SEFBd0g7WUFFeEgsSUFBSSxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7Z0JBQ3hDLHVHQUF1RztnQkFDdkcsVUFBVSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQztnQkFFOUMsK0RBQStEO2dCQUMvRCxVQUFVLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxlQUFlLEdBQUc7b0JBQzdDLFlBQVksRUFBRSxPQUFPO29CQUNyQixrQkFBa0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVM7b0JBQzVDLGlCQUFpQixFQUFFLElBQUk7aUJBQ3hCLENBQUM7Z0JBRUYsa0ZBQWtGO2dCQUNsRixVQUFVLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxJQUFJLHlCQUFlLENBQUM7b0JBQ3hELE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7b0JBQ3BCLE9BQU8sRUFBRTt3QkFDUCxZQUFZO3FCQUNiO29CQUNELFNBQVMsRUFBRTt3QkFDVCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVE7cUJBQ3hCO2lCQUNGLENBQUMsQ0FBQyxDQUFDO2FBQ0w7WUFFRCxNQUFNLGNBQWMsR0FBRyxVQUFVLENBQUMsRUFBRSxDQUFDLGNBQXNELENBQUM7WUFDNUYsWUFBWSxDQUFDLFFBQVEsQ0FBQyxHQUFHLGNBQWMsQ0FBQyxTQUFTLENBQUM7WUFFbEQsMkJBQTJCO1lBQzNCLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBZ0IsRUFBRSxFQUFFO2dCQUN6RCxJQUFJLFFBQVEsR0FBRyxTQUFTLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQ3RDLElBQUksc0JBQVEsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxRQUFRLENBQUMsUUFBUSxJQUFJLFFBQVEsQ0FBQyxTQUFTLFdBQVcsRUFBRTtvQkFDM0YsR0FBRyxFQUFFLElBQUksQ0FBQyxVQUFVO29CQUNwQixRQUFRLEVBQUUsUUFBUSxDQUFDLFFBQVE7b0JBQzNCLFNBQVMsRUFBRSxRQUFRLENBQUMsU0FBUztvQkFDN0IsVUFBVSxFQUFFLFVBQVU7b0JBQ3RCLHNCQUFzQixFQUFFLDZCQUFlLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQztvQkFDakYsdUJBQXVCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLHVCQUF1QixDQUFDO2lCQUNwRixDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztZQUVILHlCQUF5QjtZQUN6QixTQUFTLENBQUMsUUFBUSxDQUFDLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDLFdBQWdCLEVBQUUsRUFBRTtnQkFDNUQsSUFBSSxRQUFRLEdBQUcsU0FBUyxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDMUMsSUFBSSxzQkFBUSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLFFBQVEsQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLFNBQVMsV0FBVyxFQUFFO29CQUMzRixHQUFHLEVBQUUsSUFBSSxDQUFDLFVBQVU7b0JBQ3BCLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtvQkFDM0IsU0FBUyxFQUFFLFFBQVEsQ0FBQyxTQUFTO29CQUM3QixVQUFVLEVBQUUsVUFBVTtvQkFDdEIsc0JBQXNCLEVBQUUsNkJBQWUsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLHNCQUFzQixDQUFDO29CQUNqRix1QkFBdUIsRUFBRSw2QkFBZSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsdUJBQXVCLENBQUM7aUJBQ3BGLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLFlBQVksQ0FBQztJQUN0QixDQUFDO0lBRU8sV0FBVyxDQUFDLFNBQThCO1FBQ2hELElBQUksVUFBVSxHQUFRO1lBQ3BCLFdBQVcsRUFBRSwwQkFBVyxDQUFDLGVBQWU7WUFDeEMsWUFBWSxFQUFFO2dCQUNaLElBQUksRUFBRSxTQUFTLENBQUMsWUFBWSxDQUFDLElBQUk7Z0JBQ2pDLElBQUksRUFBRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUM7YUFDN0Q7U0FDRixDQUFDO1FBRUYsSUFBSSxTQUFTLENBQUMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFO1lBQy9DLFVBQVUsQ0FBQyxPQUFPLEdBQUc7Z0JBQ25CLElBQUksRUFBRSxTQUFTLENBQUMsT0FBTyxDQUFDLElBQUk7Z0JBQzVCLElBQUksRUFBRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUM7YUFDeEQsQ0FBQztTQUNIO1FBQUEsQ0FBQztRQUVGLElBQUksU0FBUyxDQUFDLEdBQUcsSUFBSSxTQUFTLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRTtZQUMxQyxVQUFVLENBQUMsbUJBQW1CLEdBQUcsU0FBUyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUM7U0FDOUQ7UUFFRCxNQUFNLEtBQUssR0FBRyxJQUFJLG9CQUFLLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLFNBQVMsQ0FBQyxTQUFTLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFbEYsSUFBSSxTQUFTLENBQUMsc0JBQXNCLElBQUksU0FBUyxDQUFDLHNCQUFzQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDbkYsU0FBUyxDQUFDLHNCQUFzQixDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQVEsRUFBRSxFQUFFO2dCQUNwRCxLQUFLLENBQUMsdUJBQXVCLENBQUM7b0JBQzVCLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUztvQkFDeEIsWUFBWSxFQUFFO3dCQUNaLElBQUksRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLElBQUk7d0JBQzNCLElBQUksRUFBRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUM7cUJBQ3ZEO29CQUNELGNBQWMsRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQUM7aUJBQzFFLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZUFBZSxDQUFDLFNBQThCOztRQUNwRCxPQUFPLElBQUksb0JBQUssQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUU7WUFDL0MsV0FBVyxFQUFFLDBCQUFXLENBQUMsZUFBZTtZQUN4QyxZQUFZLEVBQUU7Z0JBQ1osSUFBSSxFQUFFLFNBQVMsQ0FBQyxZQUFZLENBQUMsSUFBSTtnQkFDakMsSUFBSSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQzthQUM3RDtZQUNELE9BQU8sRUFBRTtnQkFDUCxJQUFJLEVBQUUsU0FBUyxDQUFDLE9BQVEsQ0FBQyxJQUFJO2dCQUM3QixJQUFJLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxPQUFRLENBQUMsSUFBSSxDQUFDO2FBQ3pEO1lBQ0QsbUJBQW1CLEVBQUUsT0FBQSxTQUFTLENBQUMsR0FBRywwQ0FBRSxhQUFhLEtBQUksTUFBTTtTQUM1RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sb0JBQW9CLENBQUMsSUFBWTtRQUN2QyxRQUFRLElBQUksRUFBRTtZQUNaLEtBQUssR0FBRztnQkFDTixPQUFPLDRCQUFhLENBQUMsTUFBTSxDQUFDO1lBQzlCLEtBQUssR0FBRztnQkFDTixPQUFPLDRCQUFhLENBQUMsTUFBTSxDQUFDO1lBQzlCLEtBQUssR0FBRyxDQUFDLENBQUMsa0JBQWtCO1lBQzVCO2dCQUNFLE9BQU8sNEJBQWEsQ0FBQyxNQUFNLENBQUM7U0FDL0I7SUFDSCxDQUFDO0lBRU8scUJBQXFCLENBQUMsSUFBWTtRQUN4QyxRQUFRLElBQUksRUFBRTtZQUNaLEtBQUssU0FBUztnQkFDWixPQUFPLDZCQUFjLENBQUMsT0FBTyxDQUFDO1lBQ2hDLEtBQUssV0FBVztnQkFDZCxPQUFPLDZCQUFjLENBQUMsU0FBUyxDQUFDO1lBQ2xDLEtBQUssS0FBSyxDQUFDLENBQUMsa0JBQWtCO1lBQzlCO2dCQUNFLE9BQU8sNkJBQWMsQ0FBQyxHQUFHLENBQUM7U0FDN0I7SUFDSCxDQUFDO0lBRU8sbUJBQW1CO1FBQ3pCLEtBQUssTUFBTSxDQUFDLFFBQVEsRUFBRSxhQUFhLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUMxRSxNQUFNLGdCQUFnQixHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDaEUsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLGdCQUFnQixFQUFFLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFFMUYsYUFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQW9DLEVBQUUsRUFBRTtnQkFDN0QsSUFBSSxzQkFBUSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLFFBQVEsQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLFNBQVMsV0FBVyxFQUFFO29CQUMzRixHQUFHLEVBQUUsSUFBSSxDQUFDLFVBQVU7b0JBQ3BCLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtvQkFDM0IsU0FBUyxFQUFFLFFBQVEsQ0FBQyxTQUFTO29CQUM3QixVQUFVLEVBQUUsY0FBYztvQkFDMUIsc0JBQXNCLEVBQUUsNkJBQWUsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLDZCQUE2QixDQUFDO29CQUMxRix1QkFBdUIsRUFBRSw2QkFBZSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsOEJBQThCLENBQUM7aUJBQzdGLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDOzs7Ozs7Ozs7O0lBVU0sK0JBQStCLENBQUMsWUFBb0IsRUFBRSxFQUFVLEVBQUUsY0FBeUIsRUFBRSxPQUEyQjtRQUM3SCxNQUFNLGtCQUFrQixHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsbUJBQW1CLENBQUMsRUFBRSxFQUFFLGNBQWMsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUU1RixLQUFLLE1BQU0sUUFBUSxJQUFJLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLENBQUMsRUFBRTtZQUMzRCxJQUFJLHNCQUFRLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsUUFBUSxDQUFDLFFBQVEsSUFBSSxRQUFRLENBQUMsU0FBUyxXQUFXLEVBQUU7Z0JBQzNGLEdBQUcsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDcEIsUUFBUSxFQUFFLFFBQVEsQ0FBQyxRQUFRO2dCQUMzQixTQUFTLEVBQUUsUUFBUSxDQUFDLFNBQVM7Z0JBQzdCLFVBQVUsRUFBRSxrQkFBa0I7Z0JBQzlCLHNCQUFzQixFQUFFLDZCQUFlLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyw2QkFBNkIsQ0FBQztnQkFDMUYsdUJBQXVCLEVBQUUsNkJBQWUsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLDhCQUE4QixDQUFDO2FBQzdGLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTyxrQkFBa0IsQ0FBQztJQUM1QixDQUFDO0NBQ0Y7QUEvVUQsZ0RBK1VDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtcbiAgR3JhcGhxbEFwaSxcbiAgQXV0aG9yaXphdGlvblR5cGUsXG4gIEZpZWxkTG9nTGV2ZWwsXG4gIE1hcHBpbmdUZW1wbGF0ZSxcbiAgQ2ZuRGF0YVNvdXJjZSxcbiAgUmVzb2x2ZXIsXG4gIEF1dGhvcml6YXRpb25Db25maWcsXG4gIFNjaGVtYSxcbiAgRGF0YVNvdXJjZU9wdGlvbnMsXG4gIExhbWJkYURhdGFTb3VyY2UsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1hcHBzeW5jJztcblxuaW1wb3J0IHsgVGFibGUsIEF0dHJpYnV0ZVR5cGUsIFByb2plY3Rpb25UeXBlLCBCaWxsaW5nTW9kZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1keW5hbW9kYic7XG5pbXBvcnQgeyBFZmZlY3QsIFBvbGljeVN0YXRlbWVudCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgSUZ1bmN0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIE5lc3RlZFN0YWNrLCBDZm5PdXRwdXQgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHtcbiAgQ2RrVHJhbnNmb3JtZXJSZXNvbHZlcixcbiAgQ2RrVHJhbnNmb3JtZXJGdW5jdGlvblJlc29sdmVyLFxuICBDZGtUcmFuc2Zvcm1lckh0dHBSZXNvbHZlcixcbiAgQ2RrVHJhbnNmb3JtZXJUYWJsZSxcbiAgU2NoZW1hVHJhbnNmb3JtZXJPdXRwdXRzLFxufSBmcm9tICcuL3RyYW5zZm9ybWVyJztcblxuaW1wb3J0IHsgU2NoZW1hVHJhbnNmb3JtZXIsIFNjaGVtYVRyYW5zZm9ybWVyUHJvcHMgfSBmcm9tICcuL3RyYW5zZm9ybWVyL3NjaGVtYS10cmFuc2Zvcm1lcic7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQXBwU3luY1RyYW5zZm9ybWVyUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzY2hlbWFQYXRoOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXV0aG9yaXphdGlvbkNvbmZpZz86IEF1dGhvcml6YXRpb25Db25maWc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhcGlOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHN5bmNFbmFibGVkPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZmllbGRMb2dMZXZlbD86IEZpZWxkTG9nTGV2ZWw7XG59XG5cbmNvbnN0IGRlZmF1bHRBdXRob3JpemF0aW9uQ29uZmlnOiBBdXRob3JpemF0aW9uQ29uZmlnID0ge1xuICBkZWZhdWx0QXV0aG9yaXphdGlvbjoge1xuICAgIGF1dGhvcml6YXRpb25UeXBlOiBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZLFxuICAgIGFwaUtleUNvbmZpZzoge1xuICAgICAgZGVzY3JpcHRpb246ICdBdXRvIGdlbmVyYXRlZCBBUEkgS2V5IGZyb20gY29uc3RydWN0JyxcbiAgICAgIG5hbWU6ICdkZXYnLFxuICAgIH0sXG4gIH0sXG59O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBcHBTeW5jVHJhbnNmb3JtZXIgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBhcHBzeW5jQVBJOiBHcmFwaHFsQXBpXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBuZXN0ZWRBcHBzeW5jU3RhY2s6IE5lc3RlZFN0YWNrO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB0YWJsZU5hbWVNYXA6IHsgW25hbWU6IHN0cmluZ106IGFueSB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBvdXRwdXRzOiBTY2hlbWFUcmFuc2Zvcm1lck91dHB1dHM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHJlc29sdmVyczogYW55O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGZ1bmN0aW9uUmVzb2x2ZXJzOiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lckZ1bmN0aW9uUmVzb2x2ZXJbXSB9O1xuXG4gIHB1YmxpYyByZWFkb25seSBodHRwUmVzb2x2ZXJzOiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lckh0dHBSZXNvbHZlcltdIH07XG5cbiAgcHJpdmF0ZSBpc1N5bmNFbmFibGVkOiBib29sZWFuXG4gIHByaXZhdGUgc3luY1RhYmxlOiBUYWJsZSB8IHVuZGVmaW5lZFxuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcHBTeW5jVHJhbnNmb3JtZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmlzU3luY0VuYWJsZWQgPSBwcm9wcy5zeW5jRW5hYmxlZCA/IHByb3BzLnN5bmNFbmFibGVkIDogZmFsc2U7XG5cbiAgICBjb25zdCB0cmFuc2Zvcm1lckNvbmZpZ3VyYXRpb246IFNjaGVtYVRyYW5zZm9ybWVyUHJvcHMgPSB7XG4gICAgICBzY2hlbWFQYXRoOiBwcm9wcy5zY2hlbWFQYXRoLFxuICAgICAgc3luY0VuYWJsZWQ6IHByb3BzLnN5bmNFbmFibGVkID8/IGZhbHNlLFxuICAgIH07XG5cbiAgICBjb25zdCB0cmFuc2Zvcm1lciA9IG5ldyBTY2hlbWFUcmFuc2Zvcm1lcih0cmFuc2Zvcm1lckNvbmZpZ3VyYXRpb24pO1xuICAgIHRoaXMub3V0cHV0cyA9IHRyYW5zZm9ybWVyLnRyYW5zZm9ybSgpO1xuICAgIGNvbnN0IHJlc29sdmVycyA9IHRyYW5zZm9ybWVyLmdldFJlc29sdmVycygpO1xuXG4gICAgdGhpcy5mdW5jdGlvblJlc29sdmVycyA9IHRoaXMub3V0cHV0cy5mdW5jdGlvblJlc29sdmVycyA/PyB7fTtcblxuICAgIC8vIFJlbW92ZSBhbnkgZnVuY3Rpb24gcmVzb2x2ZXJzIGZyb20gdGhlIHRvdGFsIGxpc3Qgb2YgcmVzb2x2ZXJzXG4gICAgLy8gT3RoZXJ3aXNlIGl0IHdpbGwgYWRkIHRoZW0gdHdpY2VcbiAgICBmb3IgKGNvbnN0IFtfLCBmdW5jdGlvblJlc29sdmVyc10gb2YgT2JqZWN0LmVudHJpZXModGhpcy5mdW5jdGlvblJlc29sdmVycykpIHtcbiAgICAgIGZ1bmN0aW9uUmVzb2x2ZXJzLmZvckVhY2goKHJlc29sdmVyOiBhbnkpID0+IHtcbiAgICAgICAgc3dpdGNoIChyZXNvbHZlci50eXBlTmFtZSkge1xuICAgICAgICAgIGNhc2UgJ1F1ZXJ5JzpcbiAgICAgICAgICBjYXNlICdNdXRhdGlvbic6XG4gICAgICAgICAgY2FzZSAnU3Vic2NyaXB0aW9uJzpcbiAgICAgICAgICAgIGRlbGV0ZSByZXNvbHZlcnNbcmVzb2x2ZXIuZmllbGROYW1lXTtcbiAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICB0aGlzLmh0dHBSZXNvbHZlcnMgPSB0aGlzLm91dHB1dHMuaHR0cFJlc29sdmVycyA/PyB7fTtcblxuICAgIC8vIFJlbW92ZSBhbnkgaHR0cCByZXNvbHZlcnMgZnJvbSB0aGUgdG90YWwgbGlzdCBvZiByZXNvbHZlcnNcbiAgICAvLyBPdGhlcndpc2UgaXQgd2lsbCBhZGQgdGhlbSB0d2ljZVxuICAgIGZvciAoY29uc3QgW18sIGh0dHBSZXNvbHZlcnNdIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMuaHR0cFJlc29sdmVycykpIHtcbiAgICAgIGh0dHBSZXNvbHZlcnMuZm9yRWFjaCgocmVzb2x2ZXI6IGFueSkgPT4ge1xuICAgICAgICBzd2l0Y2ggKHJlc29sdmVyLnR5cGVOYW1lKSB7XG4gICAgICAgICAgY2FzZSAnUXVlcnknOlxuICAgICAgICAgIGNhc2UgJ011dGF0aW9uJzpcbiAgICAgICAgICBjYXNlICdTdWJzY3JpcHRpb24nOlxuICAgICAgICAgICAgZGVsZXRlIHJlc29sdmVyc1tyZXNvbHZlci5maWVsZE5hbWVdO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMucmVzb2x2ZXJzID0gcmVzb2x2ZXJzO1xuXG4gICAgdGhpcy5uZXN0ZWRBcHBzeW5jU3RhY2sgPSBuZXcgTmVzdGVkU3RhY2sodGhpcywgJ2FwcHN5bmMtbmVzdGVkLXN0YWNrJyk7XG5cbiAgICAvLyBBcHBTeW5jXG4gICAgdGhpcy5hcHBzeW5jQVBJID0gbmV3IEdyYXBocWxBcGkodGhpcy5uZXN0ZWRBcHBzeW5jU3RhY2ssIGAke2lkfS1hcGlgLCB7XG4gICAgICBuYW1lOiBwcm9wcy5hcGlOYW1lID8gcHJvcHMuYXBpTmFtZSA6IGAke2lkfS1hcGlgLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzogcHJvcHMuYXV0aG9yaXphdGlvbkNvbmZpZyA/IHByb3BzLmF1dGhvcml6YXRpb25Db25maWcgOiBkZWZhdWx0QXV0aG9yaXphdGlvbkNvbmZpZyxcbiAgICAgIGxvZ0NvbmZpZzoge1xuICAgICAgICBmaWVsZExvZ0xldmVsOiBwcm9wcy5maWVsZExvZ0xldmVsID8gcHJvcHMuZmllbGRMb2dMZXZlbCA6IEZpZWxkTG9nTGV2ZWwuTk9ORSxcbiAgICAgIH0sXG4gICAgICBzY2hlbWE6IFNjaGVtYS5mcm9tQXNzZXQoJy4vYXBwc3luYy9zY2hlbWEuZ3JhcGhxbCcpLFxuICAgIH0pO1xuXG4gICAgbGV0IHRhYmxlRGF0YSA9IHRoaXMub3V0cHV0cy5jZGtUYWJsZXMgPz8ge307XG5cbiAgICAvLyBDaGVjayB0byBzZWUgaWYgc3luYyBpcyBlbmFibGVkXG4gICAgaWYgKHRhYmxlRGF0YS5EYXRhU3RvcmUpIHtcbiAgICAgIHRoaXMuaXNTeW5jRW5hYmxlZCA9IHRydWU7XG4gICAgICB0aGlzLnN5bmNUYWJsZSA9IHRoaXMuY3JlYXRlU3luY1RhYmxlKHRhYmxlRGF0YS5EYXRhU3RvcmUpO1xuICAgICAgZGVsZXRlIHRhYmxlRGF0YS5EYXRhU3RvcmU7IC8vIFdlIGRvbid0IHdhbnQgdG8gY3JlYXRlIHRoaXMgYWdhaW4gYmVsb3cgc28gcmVtb3ZlIGl0IGZyb20gdGhlIHRhYmxlRGF0YSBtYXBcbiAgICB9XG5cbiAgICB0aGlzLnRhYmxlTmFtZU1hcCA9IHRoaXMuY3JlYXRlVGFibGVzQW5kUmVzb2x2ZXJzKHRhYmxlRGF0YSwgcmVzb2x2ZXJzKTtcbiAgICBpZiAodGhpcy5vdXRwdXRzLm5vbmVSZXNvbHZlcnMpIHRoaXMuY3JlYXRlTm9uZURhdGFTb3VyY2VBbmRSZXNvbHZlcnModGhpcy5vdXRwdXRzLm5vbmVSZXNvbHZlcnMsIHJlc29sdmVycyk7XG4gICAgdGhpcy5jcmVhdGVIdHRwUmVzb2x2ZXJzKCk7XG5cbiAgICAvLyBPdXRwdXRzIHNvIHdlIGNhbiBnZW5lcmF0ZSBleHBvcnRzXG4gICAgbmV3IENmbk91dHB1dChzY29wZSwgJ2FwcHN5bmNHcmFwaFFMRW5kcG9pbnRPdXRwdXQnLCB7XG4gICAgICB2YWx1ZTogdGhpcy5hcHBzeW5jQVBJLmdyYXBocWxVcmwsXG4gICAgICBkZXNjcmlwdGlvbjogJ091dHB1dCBmb3IgYXdzX2FwcHN5bmNfZ3JhcGhxbEVuZHBvaW50JyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIE5PTkUgZGF0YSBzb3VyY2UgYW5kIGFzc29jaWF0ZWQgcmVzb2x2ZXJzXG4gICAqIEBwYXJhbSBub25lUmVzb2x2ZXJzIFRoZSByZXNvbHZlcnMgdGhhdCBiZWxvbmcgdG8gdGhlIG5vbmUgZGF0YSBzb3VyY2VcbiAgICogQHBhcmFtIHJlc29sdmVycyBUaGUgcmVzb2x2ZXIgbWFwIG1pbnVzIGZ1bmN0aW9uIHJlc29sdmVyc1xuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVOb25lRGF0YVNvdXJjZUFuZFJlc29sdmVycyhub25lUmVzb2x2ZXJzOiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclJlc29sdmVyIH0sIHJlc29sdmVyczogYW55KSB7XG4gICAgY29uc3Qgbm9uZURhdGFTb3VyY2UgPSB0aGlzLmFwcHN5bmNBUEkuYWRkTm9uZURhdGFTb3VyY2UoJ05PTkUnKTtcblxuICAgIE9iamVjdC5rZXlzKG5vbmVSZXNvbHZlcnMpLmZvckVhY2goKHJlc29sdmVyS2V5OiBhbnkpID0+IHtcbiAgICAgIGNvbnN0IHJlc29sdmVyID0gcmVzb2x2ZXJzW3Jlc29sdmVyS2V5XTtcblxuICAgICAgbmV3IFJlc29sdmVyKHRoaXMubmVzdGVkQXBwc3luY1N0YWNrLCBgJHtyZXNvbHZlci50eXBlTmFtZX0tJHtyZXNvbHZlci5maWVsZE5hbWV9LXJlc29sdmVyYCwge1xuICAgICAgICBhcGk6IHRoaXMuYXBwc3luY0FQSSxcbiAgICAgICAgdHlwZU5hbWU6IHJlc29sdmVyLnR5cGVOYW1lLFxuICAgICAgICBmaWVsZE5hbWU6IHJlc29sdmVyLmZpZWxkTmFtZSxcbiAgICAgICAgZGF0YVNvdXJjZTogbm9uZURhdGFTb3VyY2UsXG4gICAgICAgIHJlcXVlc3RNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tRmlsZShyZXNvbHZlci5yZXF1ZXN0TWFwcGluZ1RlbXBsYXRlKSxcbiAgICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tRmlsZShyZXNvbHZlci5yZXNwb25zZU1hcHBpbmdUZW1wbGF0ZSksXG4gICAgICB9KTtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGVhY2ggZHluYW1vZGIgdGFibGUsIGdzaXMsIGR5bmFtb2RiIGRhdGFzb3VyY2UsIGFuZCBhc3NvY2lhdGVkIHJlc29sdmVyc1xuICAgKiBJZiBzeW5jIGlzIGVuYWJsZWQgdGhlbiBUVEwgY29uZmlndXJhdGlvbiBpcyBhZGRlZFxuICAgKiBSZXR1cm5zIHRhYmxlTmFtZTogdGFibGUgbWFwIGluIGNhc2UgaXQgaXMgbmVlZGVkIGZvciBsYW1iZGEgZnVuY3Rpb25zLCBldGNcbiAgICogQHBhcmFtIHRhYmxlRGF0YSBUaGUgQ2RrVHJhbnNmb3JtZXIgdGFibGUgaW5mb3JtYXRpb25cbiAgICogQHBhcmFtIHJlc29sdmVycyBUaGUgcmVzb2x2ZXIgbWFwIG1pbnVzIGZ1bmN0aW9uIHJlc29sdmVyc1xuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVUYWJsZXNBbmRSZXNvbHZlcnModGFibGVEYXRhOiB7IFtuYW1lOiBzdHJpbmddOiBDZGtUcmFuc2Zvcm1lclRhYmxlIH0sIHJlc29sdmVyczogYW55KTogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH0ge1xuICAgIGNvbnN0IHRhYmxlTmFtZU1hcDogYW55ID0ge307XG5cbiAgICBPYmplY3Qua2V5cyh0YWJsZURhdGEpLmZvckVhY2goKHRhYmxlS2V5OiBhbnkpID0+IHtcbiAgICAgIGNvbnN0IHRhYmxlID0gdGhpcy5jcmVhdGVUYWJsZSh0YWJsZURhdGFbdGFibGVLZXldKTtcbiAgICAgIGNvbnN0IGRhdGFTb3VyY2UgPSB0aGlzLmFwcHN5bmNBUEkuYWRkRHluYW1vRGJEYXRhU291cmNlKHRhYmxlS2V5LCB0YWJsZSk7XG5cbiAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1wcm9wZXJ0aWVzLWFwcHN5bmMtZGF0YXNvdXJjZS1kZWx0YXN5bmNjb25maWcuaHRtbFxuXG4gICAgICBpZiAodGhpcy5pc1N5bmNFbmFibGVkICYmIHRoaXMuc3luY1RhYmxlKSB7XG4gICAgICAgIC8vQHRzLWlnbm9yZSAtIGRzIGlzIHRoZSBiYXNlIENmbkRhdGFTb3VyY2UgYW5kIHRoZSBkYiBjb25maWcgbmVlZHMgdG8gYmUgdmVyc2lvbmVkIC0gc2VlIENmbkRhdGFTb3VyY2VcbiAgICAgICAgZGF0YVNvdXJjZS5kcy5keW5hbW9EYkNvbmZpZy52ZXJzaW9uZWQgPSB0cnVlO1xuXG4gICAgICAgIC8vQHRzLWlnbm9yZSAtIGRzIGlzIHRoZSBiYXNlIENmbkRhdGFTb3VyY2UgLSBzZWUgQ2ZuRGF0YVNvdXJjZVxuICAgICAgICBkYXRhU291cmNlLmRzLmR5bmFtb0RiQ29uZmlnLmRlbHRhU3luY0NvbmZpZyA9IHtcbiAgICAgICAgICBiYXNlVGFibGVUdGw6ICc0MzIwMCcsIC8vIEdvdCB0aGlzIHZhbHVlIGZyb20gYW1wbGlmeSAtIDMwIGRheXMgaW4gbWludXRlc1xuICAgICAgICAgIGRlbHRhU3luY1RhYmxlTmFtZTogdGhpcy5zeW5jVGFibGUudGFibGVOYW1lLFxuICAgICAgICAgIGRlbHRhU3luY1RhYmxlVHRsOiAnMzAnLCAvLyBHb3QgdGhpcyB2YWx1ZSBmcm9tIGFtcGxpZnkgLSAzMCBtaW51dGVzXG4gICAgICAgIH07XG5cbiAgICAgICAgLy8gTmVlZCB0byBhZGQgcGVybWlzc2lvbiBmb3Igb3VyIGRhdGFzb3VyY2Ugc2VydmljZSByb2xlIHRvIGFjY2VzcyB0aGUgc3luYyB0YWJsZVxuICAgICAgICBkYXRhU291cmNlLmdyYW50UHJpbmNpcGFsLmFkZFRvUG9saWN5KG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICdkeW5hbW9kYjoqJywgLy8gVE9ETzogVGhpcyBtYXkgYmUgdG9vIHBlcm1pc3NpdmVcbiAgICAgICAgICBdLFxuICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgdGhpcy5zeW5jVGFibGUudGFibGVBcm4sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSkpO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBkeW5hbW9EYkNvbmZpZyA9IGRhdGFTb3VyY2UuZHMuZHluYW1vRGJDb25maWcgYXMgQ2ZuRGF0YVNvdXJjZS5EeW5hbW9EQkNvbmZpZ1Byb3BlcnR5O1xuICAgICAgdGFibGVOYW1lTWFwW3RhYmxlS2V5XSA9IGR5bmFtb0RiQ29uZmlnLnRhYmxlTmFtZTtcblxuICAgICAgLy8gTG9vcCB0aGUgYmFzaWMgcmVzb2x2ZXJzXG4gICAgICB0YWJsZURhdGFbdGFibGVLZXldLnJlc29sdmVycy5mb3JFYWNoKChyZXNvbHZlcktleTogYW55KSA9PiB7XG4gICAgICAgIGxldCByZXNvbHZlciA9IHJlc29sdmVyc1tyZXNvbHZlcktleV07XG4gICAgICAgIG5ldyBSZXNvbHZlcih0aGlzLm5lc3RlZEFwcHN5bmNTdGFjaywgYCR7cmVzb2x2ZXIudHlwZU5hbWV9LSR7cmVzb2x2ZXIuZmllbGROYW1lfS1yZXNvbHZlcmAsIHtcbiAgICAgICAgICBhcGk6IHRoaXMuYXBwc3luY0FQSSxcbiAgICAgICAgICB0eXBlTmFtZTogcmVzb2x2ZXIudHlwZU5hbWUsXG4gICAgICAgICAgZmllbGROYW1lOiByZXNvbHZlci5maWVsZE5hbWUsXG4gICAgICAgICAgZGF0YVNvdXJjZTogZGF0YVNvdXJjZSxcbiAgICAgICAgICByZXF1ZXN0TWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbUZpbGUocmVzb2x2ZXIucmVxdWVzdE1hcHBpbmdUZW1wbGF0ZSksXG4gICAgICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tRmlsZShyZXNvbHZlci5yZXNwb25zZU1hcHBpbmdUZW1wbGF0ZSksXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG5cbiAgICAgIC8vIExvb3AgdGhlIGdzaSByZXNvbHZlcnNcbiAgICAgIHRhYmxlRGF0YVt0YWJsZUtleV0uZ3NpUmVzb2x2ZXJzLmZvckVhY2goKHJlc29sdmVyS2V5OiBhbnkpID0+IHtcbiAgICAgICAgbGV0IHJlc29sdmVyID0gcmVzb2x2ZXJzLmdzaVtyZXNvbHZlcktleV07XG4gICAgICAgIG5ldyBSZXNvbHZlcih0aGlzLm5lc3RlZEFwcHN5bmNTdGFjaywgYCR7cmVzb2x2ZXIudHlwZU5hbWV9LSR7cmVzb2x2ZXIuZmllbGROYW1lfS1yZXNvbHZlcmAsIHtcbiAgICAgICAgICBhcGk6IHRoaXMuYXBwc3luY0FQSSxcbiAgICAgICAgICB0eXBlTmFtZTogcmVzb2x2ZXIudHlwZU5hbWUsXG4gICAgICAgICAgZmllbGROYW1lOiByZXNvbHZlci5maWVsZE5hbWUsXG4gICAgICAgICAgZGF0YVNvdXJjZTogZGF0YVNvdXJjZSxcbiAgICAgICAgICByZXF1ZXN0TWFwcGluZ1RlbXBsYXRlOiBNYXBwaW5nVGVtcGxhdGUuZnJvbUZpbGUocmVzb2x2ZXIucmVxdWVzdE1hcHBpbmdUZW1wbGF0ZSksXG4gICAgICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tRmlsZShyZXNvbHZlci5yZXNwb25zZU1hcHBpbmdUZW1wbGF0ZSksXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICByZXR1cm4gdGFibGVOYW1lTWFwO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVUYWJsZSh0YWJsZURhdGE6IENka1RyYW5zZm9ybWVyVGFibGUpIHtcbiAgICBsZXQgdGFibGVQcm9wczogYW55ID0ge1xuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCxcbiAgICAgIHBhcnRpdGlvbktleToge1xuICAgICAgICBuYW1lOiB0YWJsZURhdGEucGFydGl0aW9uS2V5Lm5hbWUsXG4gICAgICAgIHR5cGU6IHRoaXMuY29udmVydEF0dHJpYnV0ZVR5cGUodGFibGVEYXRhLnBhcnRpdGlvbktleS50eXBlKSxcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGlmICh0YWJsZURhdGEuc29ydEtleSAmJiB0YWJsZURhdGEuc29ydEtleS5uYW1lKSB7XG4gICAgICB0YWJsZVByb3BzLnNvcnRLZXkgPSB7XG4gICAgICAgIG5hbWU6IHRhYmxlRGF0YS5zb3J0S2V5Lm5hbWUsXG4gICAgICAgIHR5cGU6IHRoaXMuY29udmVydEF0dHJpYnV0ZVR5cGUodGFibGVEYXRhLnNvcnRLZXkudHlwZSksXG4gICAgICB9O1xuICAgIH07XG5cbiAgICBpZiAodGFibGVEYXRhLnR0bCAmJiB0YWJsZURhdGEudHRsLkVuYWJsZWQpIHtcbiAgICAgIHRhYmxlUHJvcHMudGltZVRvTGl2ZUF0dHJpYnV0ZSA9IHRhYmxlRGF0YS50dGwuQXR0cmlidXRlTmFtZTtcbiAgICB9XG5cbiAgICBjb25zdCB0YWJsZSA9IG5ldyBUYWJsZSh0aGlzLm5lc3RlZEFwcHN5bmNTdGFjaywgdGFibGVEYXRhLnRhYmxlTmFtZSwgdGFibGVQcm9wcyk7XG5cbiAgICBpZiAodGFibGVEYXRhLmdsb2JhbFNlY29uZGFyeUluZGV4ZXMgJiYgdGFibGVEYXRhLmdsb2JhbFNlY29uZGFyeUluZGV4ZXMubGVuZ3RoID4gMCkge1xuICAgICAgdGFibGVEYXRhLmdsb2JhbFNlY29uZGFyeUluZGV4ZXMuZm9yRWFjaCgoZ3NpOiBhbnkpID0+IHtcbiAgICAgICAgdGFibGUuYWRkR2xvYmFsU2Vjb25kYXJ5SW5kZXgoe1xuICAgICAgICAgIGluZGV4TmFtZTogZ3NpLmluZGV4TmFtZSxcbiAgICAgICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgICAgIG5hbWU6IGdzaS5wYXJ0aXRpb25LZXkubmFtZSxcbiAgICAgICAgICAgIHR5cGU6IHRoaXMuY29udmVydEF0dHJpYnV0ZVR5cGUoZ3NpLnBhcnRpdGlvbktleS50eXBlKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHByb2plY3Rpb25UeXBlOiB0aGlzLmNvbnZlcnRQcm9qZWN0aW9uVHlwZShnc2kucHJvamVjdGlvbi5Qcm9qZWN0aW9uVHlwZSksXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRhYmxlO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgdGhlIHN5bmMgdGFibGUgZm9yIEFtcGxpZnkgRGF0YVN0b3JlXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9jb25mbGljdC1kZXRlY3Rpb24tYW5kLXN5bmMuaHRtbFxuICAgKiBAcGFyYW0gdGFibGVEYXRhIFRoZSBDZGtUcmFuc2Zvcm1lciB0YWJsZSBpbmZvcm1hdGlvblxuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVTeW5jVGFibGUodGFibGVEYXRhOiBDZGtUcmFuc2Zvcm1lclRhYmxlKTogVGFibGUge1xuICAgIHJldHVybiBuZXcgVGFibGUodGhpcywgJ2FwcHN5bmMtYXBpLXN5bmMtdGFibGUnLCB7XG4gICAgICBiaWxsaW5nTW9kZTogQmlsbGluZ01vZGUuUEFZX1BFUl9SRVFVRVNULFxuICAgICAgcGFydGl0aW9uS2V5OiB7XG4gICAgICAgIG5hbWU6IHRhYmxlRGF0YS5wYXJ0aXRpb25LZXkubmFtZSxcbiAgICAgICAgdHlwZTogdGhpcy5jb252ZXJ0QXR0cmlidXRlVHlwZSh0YWJsZURhdGEucGFydGl0aW9uS2V5LnR5cGUpLFxuICAgICAgfSxcbiAgICAgIHNvcnRLZXk6IHtcbiAgICAgICAgbmFtZTogdGFibGVEYXRhLnNvcnRLZXkhLm5hbWUsIC8vIFdlIGtub3cgaXQgaGFzIGEgc29ydGtleSBiZWNhdXNlIHdlIGZvcmNlZCBpdCB0b1xuICAgICAgICB0eXBlOiB0aGlzLmNvbnZlcnRBdHRyaWJ1dGVUeXBlKHRhYmxlRGF0YS5zb3J0S2V5IS50eXBlKSwgLy8gV2Uga25vdyBpdCBoYXMgYSBzb3J0a2V5IGJlY2F1c2Ugd2UgZm9yY2VkIGl0IHRvXG4gICAgICB9LFxuICAgICAgdGltZVRvTGl2ZUF0dHJpYnV0ZTogdGFibGVEYXRhLnR0bD8uQXR0cmlidXRlTmFtZSB8fCAnX3R0bCcsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGNvbnZlcnRBdHRyaWJ1dGVUeXBlKHR5cGU6IHN0cmluZyk6IEF0dHJpYnV0ZVR5cGUge1xuICAgIHN3aXRjaCAodHlwZSkge1xuICAgICAgY2FzZSAnTic6XG4gICAgICAgIHJldHVybiBBdHRyaWJ1dGVUeXBlLk5VTUJFUjtcbiAgICAgIGNhc2UgJ0InOlxuICAgICAgICByZXR1cm4gQXR0cmlidXRlVHlwZS5CSU5BUlk7XG4gICAgICBjYXNlICdTJzogLy8gU2FtZSBhcyBkZWZhdWx0XG4gICAgICBkZWZhdWx0OlxuICAgICAgICByZXR1cm4gQXR0cmlidXRlVHlwZS5TVFJJTkc7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjb252ZXJ0UHJvamVjdGlvblR5cGUodHlwZTogc3RyaW5nKTogUHJvamVjdGlvblR5cGUge1xuICAgIHN3aXRjaCAodHlwZSkge1xuICAgICAgY2FzZSAnSU5DTFVERSc6XG4gICAgICAgIHJldHVybiBQcm9qZWN0aW9uVHlwZS5JTkNMVURFO1xuICAgICAgY2FzZSAnS0VZU19PTkxZJzpcbiAgICAgICAgcmV0dXJuIFByb2plY3Rpb25UeXBlLktFWVNfT05MWTtcbiAgICAgIGNhc2UgJ0FMTCc6IC8vIFNhbWUgYXMgZGVmYXVsdFxuICAgICAgZGVmYXVsdDpcbiAgICAgICAgcmV0dXJuIFByb2plY3Rpb25UeXBlLkFMTDtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZUh0dHBSZXNvbHZlcnMoKSB7XG4gICAgZm9yIChjb25zdCBbZW5kcG9pbnQsIGh0dHBSZXNvbHZlcnNdIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMuaHR0cFJlc29sdmVycykpIHtcbiAgICAgIGNvbnN0IHN0cmlwcGVkRW5kcG9pbnQgPSBlbmRwb2ludC5yZXBsYWNlKC9bXl8wLTlBLVphLXpdL2csICcnKTtcbiAgICAgIGNvbnN0IGh0dHBEYXRhU291cmNlID0gdGhpcy5hcHBzeW5jQVBJLmFkZEh0dHBEYXRhU291cmNlKGAke3N0cmlwcGVkRW5kcG9pbnR9YCwgZW5kcG9pbnQpO1xuXG4gICAgICBodHRwUmVzb2x2ZXJzLmZvckVhY2goKHJlc29sdmVyOiBDZGtUcmFuc2Zvcm1lckh0dHBSZXNvbHZlcikgPT4ge1xuICAgICAgICBuZXcgUmVzb2x2ZXIodGhpcy5uZXN0ZWRBcHBzeW5jU3RhY2ssIGAke3Jlc29sdmVyLnR5cGVOYW1lfS0ke3Jlc29sdmVyLmZpZWxkTmFtZX0tcmVzb2x2ZXJgLCB7XG4gICAgICAgICAgYXBpOiB0aGlzLmFwcHN5bmNBUEksXG4gICAgICAgICAgdHlwZU5hbWU6IHJlc29sdmVyLnR5cGVOYW1lLFxuICAgICAgICAgIGZpZWxkTmFtZTogcmVzb2x2ZXIuZmllbGROYW1lLFxuICAgICAgICAgIGRhdGFTb3VyY2U6IGh0dHBEYXRhU291cmNlLFxuICAgICAgICAgIHJlcXVlc3RNYXBwaW5nVGVtcGxhdGU6IE1hcHBpbmdUZW1wbGF0ZS5mcm9tU3RyaW5nKHJlc29sdmVyLmRlZmF1bHRSZXF1ZXN0TWFwcGluZ1RlbXBsYXRlKSxcbiAgICAgICAgICByZXNwb25zZU1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcocmVzb2x2ZXIuZGVmYXVsdFJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlKSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRMYW1iZGFEYXRhU291cmNlQW5kUmVzb2x2ZXJzKGZ1bmN0aW9uTmFtZTogc3RyaW5nLCBpZDogc3RyaW5nLCBsYW1iZGFGdW5jdGlvbjogSUZ1bmN0aW9uLCBvcHRpb25zPzogRGF0YVNvdXJjZU9wdGlvbnMpOiBMYW1iZGFEYXRhU291cmNlIHtcbiAgICBjb25zdCBmdW5jdGlvbkRhdGFTb3VyY2UgPSB0aGlzLmFwcHN5bmNBUEkuYWRkTGFtYmRhRGF0YVNvdXJjZShpZCwgbGFtYmRhRnVuY3Rpb24sIG9wdGlvbnMpO1xuXG4gICAgZm9yIChjb25zdCByZXNvbHZlciBvZiB0aGlzLmZ1bmN0aW9uUmVzb2x2ZXJzW2Z1bmN0aW9uTmFtZV0pIHtcbiAgICAgIG5ldyBSZXNvbHZlcih0aGlzLm5lc3RlZEFwcHN5bmNTdGFjaywgYCR7cmVzb2x2ZXIudHlwZU5hbWV9LSR7cmVzb2x2ZXIuZmllbGROYW1lfS1yZXNvbHZlcmAsIHtcbiAgICAgICAgYXBpOiB0aGlzLmFwcHN5bmNBUEksXG4gICAgICAgIHR5cGVOYW1lOiByZXNvbHZlci50eXBlTmFtZSxcbiAgICAgICAgZmllbGROYW1lOiByZXNvbHZlci5maWVsZE5hbWUsXG4gICAgICAgIGRhdGFTb3VyY2U6IGZ1bmN0aW9uRGF0YVNvdXJjZSxcbiAgICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcocmVzb2x2ZXIuZGVmYXVsdFJlcXVlc3RNYXBwaW5nVGVtcGxhdGUpLFxuICAgICAgICByZXNwb25zZU1hcHBpbmdUZW1wbGF0ZTogTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcocmVzb2x2ZXIuZGVmYXVsdFJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlKSwgLy8gVGhpcyBkZWZhdWx0cyB0byBhbGxvdyBlcnJvcnMgdG8gcmV0dXJuIHRvIHRoZSBjbGllbnQgaW5zdGVhZCBvZiB0aHJvd2luZ1xuICAgICAgfSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGZ1bmN0aW9uRGF0YVNvdXJjZTtcbiAgfVxufSJdfQ==