"""Serializer implementation that packages and unpackages paths (files or directories) in the local filesystem using compressed tarfiles."""

import os
import tarfile
from typing import Any, BinaryIO, Optional

from dagger import DeserializationError


class AsTar:
    """Serializer implementation that packages and unpackages paths (files or directories) in the local filesystem using compressed tarfiles."""

    EXTENSIONS_BY_COMPRESSION = {
        "gzip": "tar.gz",
        "bz2": "tar.bz2",
        "xz": "tar.xz",
    }

    MODE_BY_COMPRESSION = {
        "gzip": "gz",
        "bz2": "bz2",
        "xz": "xz",
        "": "",
    }

    def __init__(self, output_dir: str, compression: Optional[str] = "gzip"):
        """
        Initialize an instance of the serializer.

        Parameters
        ----------
        output_dir: str
            The path to a local directory we can extract the directory into.

        compression: str, optional, default="gzip"
            The compression algorithm to use. When None, the file will be uncompressed.
            Accepted values are {"gzip", "bz2", "xz", None}.

        bufsize: int

        """
        assert compression is None or compression in ["gzip", "bz2", "xz"]

        self._output_dir = output_dir
        self._compression = compression

    def serialize(self, value: Any, writer: BinaryIO):
        """Serialize 'value', which is expected to be a path to the local filesystem, as a compressed tar file written to 'writer'."""
        with tarfile.open(
            fileobj=writer,
            mode=f"w|{self.MODE_BY_COMPRESSION[self._compression or '']}",
        ) as tar:
            tar.add(value, arcname=os.path.basename(value))

    def deserialize(self, reader: BinaryIO) -> Any:
        """Extract a tarfile into the output directory the serializer was initialized with."""
        try:
            with tarfile.open(
                fileobj=reader,
                mode=f"r|{self.MODE_BY_COMPRESSION[self._compression or '']}",
            ) as tar:
                tar.extractall(path=self._output_dir)

                tar_members = tar.getnames()
                return os.path.join(self._output_dir, tar_members[0])

        except tarfile.TarError as e:
            raise DeserializationError(e)

    @property
    def extension(self) -> str:
        """Extension to use for files generated by this serializer."""
        return self.EXTENSIONS_BY_COMPRESSION.get(self._compression or "", "tar")
