# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['pulse_throw']
install_requires = \
['Authlib>=1.0.1,<2.0.0', 'requests>=2.28.1,<3.0.0']

setup_kwargs = {
    'name': 'pulse-throw',
    'version': '0.1.0',
    'description': 'Tools for acquiring and analyzing Pulse API data.',
    'long_description': '# Pulse Throw for Python <!-- omit in toc -->\n\nTools for acquiring and analyzing Pulse API data.\n\n[Pulse](https://www.drivelinebaseball.com/pulse/) is a wearable sensor for baseball players to monitor throwing workload.\n\n- [Installation](#installation)\n- [Getting Started](#getting-started)\n- [API Requests](#api-requests)\n  - [Get Profile](#get-profile)\n  - [Get Team](#get-team)\n  - [Get Snapshots](#get-snapshots)\n  - [Get Events](#get-events)\n- [Data Filtering Functions](#data-filtering-functions)\n  - [Filter By Tag](#filter-by-tag)\n  - [Filter Simulated](#filter-simulated)\n  - [Filter High Effort](#filter-high-effort)\n- [Workload Functions](#workload-functions)\n  - [Sum Workload](#sum-workload)\n  - [Compute Acute Workload](#compute-acute-workload)\n  - [Compute Chronic Workload](#compute-chronic-workload)\n  - [Compute Acute/Chronic Workload Ratio](#compute-acutechronic-workload-ratio)\n- [Additional Resources](#additional-resources)\n\n## Installation\n\nThe `pulse` module can be installed via pip:\n\n`pip install pulse-throw`\n\n## Getting Started\n\nIn order to use the Pulse client, you must have a `client_id`, `client_secret`, and `refresh_token` provided by the Pulse team at Driveline.\n\nIt is best practice to use these values stored in a `.env` file:\n\n```bash\n# Pulse credentials\nCLIENT_ID="<CLIENT_ID>"\nCLIENT_SECRET="<CLIENT_SECRET>"\nREFRESH_TOKEN="<REFRESH_TOKEN>"\n```\n\nYou can use [`python-dotenv`](https://github.com/theskumar/python-dotenv) to load the enviroment variables for use in code:\n\n```python\nimport os\nfrom dotenv import load_dotenv\n\nload_dotenv()\n\nclient_id = os.getenv("CLIENT_ID") or ""\nclient_secret = os.getenv("CLIENT_SECRET") or ""\nrefresh_token = os.getenv("REFRESH_TOKEN") or ""\n```\n\nOnce the environment variables are loaded, a `PulseClient` object can created:\n\n```python\nimport pulse\n\n# Using a traditional constructor and destructor\nclient = pulse.PulseClient(client_id, client_secret, refresh_token)\n...\ndel client\n\n# Using a context manager that destructs automatically\nwith pulse.PulseClient(client_id, client_secret, refresh_token) as client:\n    ...\n```\n\nThe Pulse client will authenticate the client upon construction by default. This involves fetching an access token from the API. If you don\'t want this request to happen automatically, pass `authenticate=False` into the object constructor. In order to make other requests, you will need to manually call the `authenticate()` method so that the other requests have the proper authorization headers:\n\n```python\nimport pulse\n\nclient = pulse.PulseClient(\n    client_id, client_secret, refresh_token, authenticate=False\n)\n\nclient.authenticate()\n\n...\n\ndel client\n```\n\n## API Requests\n\nThere are four different API requests that `PulseClient` can make.\n\n### Get Profile\n\nReturns info about the owner of the session.\n\n**Method**: `get_profile()`\n\n**Payload**: None\n\n**Example Response**:\n\n```json\n{\n  "id": "<id>",\n  "firstName": "<first-name>",\n  "lastName": "<last-name>",\n  "email": "<email>"\n}\n```\n\n### Get Team\n\nReturns info about the owner of the session\'s team.\n\n**Method**: `get_team()`\n\n**Payload**: None\n\n**Example Response**:\n\n```json\n{\n    "team": {\n        "name": "TEAMNAME",\n        "id": "JQtyNOYdDH"\n    },\n    "members": [\n        {\n            "userId": "r5FiwuBlYZ",\n            "teamMemberId": "yvUsIxsjrg",\n            "athleteProfileId": "dgTHp0nlN0",\n            "firstName": "<player-first-name>",\n            "lastName": "<player-last-name>",\n            "email": "<player-email>"\n        },\n        ...\n```\n\n### Get Snapshots\n\nGets daily snapshots generated for one or more users over a range of dates. The owner of the session must have permission to access their data (e.g. the requested users must be either the current user or a member of a team for which the current user is a coach).\n\n**Method**: `get_snapshots(start_date: str = <end_date - 8 days>, end_date: str = <today\'s date>, user_ids: str | list[str] = <user_id>)`\n\n**Payload**:\n\n- `startDate`: The earliest date for which to get data, pulled from the `start_date` parameter. Expected in ISO 8601 format (YYYY-MM-DD). Defaults to eight days before the `end_date` parameter.\n- `endDate`: The latest date for which to get data, pulled from the `end_date` parameter. Expected in ISO 8601 format (YYYY-MM-DD). Defaults to today\'s date.\n- `pulseUserIds`: User IDs for whom to get data, pulled from the `user_ids` parameter. IDs must belong to the owner of the session or a member of their team. Defaults to the ID of the owner of the session.\n\n**Example Response**:\n\n```json\n{\n    "<id>": [\n        {\n            "date": "2021-08-01",\n            "throwCount": 0,\n            "highEffortThrowCount": 0,\n            "acuteWorkload": 2041.948878326791,\n            "chronicWorkload": 3229.3557437324716,\n            "normAcuteWorkload": 3.1660202416772414,\n            "normChronicWorkload": 5.007082087486716,\n            "workloadRatio": 0.6323084356035417,\n            "dailyWorkload": 0,\n            "normDailyWorkload": 0,\n            "baseballProjectedOneDayWorkloads": [\n                0,\n                30,\n                26.82887058971028,\n                ...\n            ]\n        },\n        ...\n```\n\n### Get Events\n\nGets all of the individual throw events for one or more users over a range of dates. The owner of the session must have permission to access their data (e.g. the requested users must be either the current user or a member of a team for which the current user is a coach).\n\n**Method**: `get_events(start_date: str = <end_date - 8 days>, end_date: str = <today\'s date>, user_ids: str | list[str] = <user_id>)`\n\n**Payload**:\n\n- `startDate`: The earliest date for which to get data, pulled from the `start_date` parameter. Expected in ISO 8601 format (YYYY-MM-DD). Defaults to eight days before the `end_date` parameter.\n- `endDate`: The latest date for which to get data, pulled from the `end_date` parameter. Expected in ISO 8601 format (YYYY-MM-DD). Defaults to today\'s date.\n- `pulseUserIds`: User IDs for whom to get data, pulled from the `user_ids` parameter. IDs must belong to the owner of the session or a member of their team. Defaults to the ID of the owner of the session.\n\n**Example Response**:\n\n```json\n{\n    "<id>": [\n        {\n            "eventId": "POC6TE5b8V",\n            "scaler": null,\n            "datetime": "2021-03-01T15:49:55.000Z",\n            "tag": null,\n            "armSlot": 59.782794823856534,\n            "armSpeed": 452.4706718068326,\n            "shoulderRotation": 155.58127276383868,\n            "torque": 34.744537353515625,\n            "ballVelocity": null,\n            "highEffort": false,\n            "ballWeight (oz)": 5.11472,\n            "preferredBallWeightUnit": "OZ",\n            "simulated": null,\n            "workload": 100.728515625,\n            "normalizedWorkload": 0.10925094783306122\n        },\n        ...\n```\n\n## Data Filtering Functions\n\nThe `pulse` module contains functions that can filter event data based on certain characteristics.\n\n### Filter By Tag\n\nFilter throw events by tag or tags.\n\n**Function**: `filter_by_tag(events: list[dict[str, Any]], tags: str | list[str], blacklist: bool = False)`\n\n```python\n# Get all throw events from a single date\nevents = client.get_events(\n    start_date="2022-05-01", end_date="2022-05-01"\n)\n\n# Get all throw events marked with the desired tag\npre_game = pulse.filter_by_tag(\n    events[client.user_id], tags="Pre-Game"\n)\n\n# Get all throw events marked with one of multiple tags\nnon_game = pulse.filter_by_tag(\n    events[client.user_id], tags=["Pre-Game", "Plyo", "Warmup"]\n)\n\n# Get all throw events without the desired tag\nnon_plyos = pulse.filter_by_tag(\n    events[client.user_id], tags="Plyo", blacklist=True\n)\n```\n\n### Filter Simulated\n\nFilter throw events based on simulated status.\n\n**Function**: `filter_simulated(events: list[dict[str, Any]], get_simulated: bool = False)`\n\n```python\n# Get all throw events from a single date\nevents = client.get_events(\n    start_date="2022-05-01", end_date="2022-05-01"\n)\n\n# Get all throw events that are not simulated\nnon_simulated = pulse.filter_simulated(events[client.user_id])\n\n# Get all throw events that are simulated\nsimulated = pulse.filter_simulated(\n    events[client.user_id], get_simulated=True\n)\n```\n\n### Filter High Effort\n\nFilter throw events based on high effort status.\n\n**Function**: `filter_high_effort(events: list[dict[str, Any]], get_high_effort: bool = True)`\n\n```python\n# Get all throw events from a single date\nevents = client.get_events(\n    start_date="2022-05-01", end_date="2022-05-01"\n)\n\n# Get all throw events that are high effort\nhigh_effort = pulse.filter_high_effort(events[client.user_id])\n\n# Get all throw events that are not high effort\nlow_effort = pulse.filter_high_effort(\n    events[client.user_id], get_high_effort=False\n)\n```\n\n## Workload Functions\n\nThe `pulse` module contains functions that are useful when manually performing workload calculations.\n\nSince throw events and daily snapshots are stored in a dict indexed by user ID, the desired user ID must be accessed by key when passing data into the following functions. For indiviudal users (i.e. not coaches), the desired user ID will be the same as the owner of the session and can be accessed using `client.user_id`. For coaches, the `get_team()` endpoint can match players with their respective user IDs.\n\n### Sum Workload\n\nCompute sum of `workload` or `normalizedWorkload` values from individual throw events returned by the `get_events()` endpoint.\n\n**Function**: `sum_workload(events: list[dict[str, Any]], normalized: bool = True)`\n\n```python\n# Get all throw events from a single date\nevents = client.get_events(\n    start_date="2022-05-01", end_date="2022-05-01"\n)\n\n# Make sure to access the desired user from events\nnorm_workload = pulse.sum_workload(events[client.user_id])\n\n# Compute sum using unnormalized values\nworkload = pulse.sum_workload(\n    events[client.user_id], normalized=False\n)\n```\n\nThis function can be useful for summing the workload of throws with a certain tag:\n\n```python\npre_game = pulse.filter_by_tag(events, tag="Pre-Game")\n\npre_game_workload = pulse.sum_workload(pre_game)\n```\n\n### Compute Acute Workload\n\nCompute acute workload using `dailyWorkload` or `normDailyWorkload` values from daily snapshots returned by the `get_snapshots()` endpoint.\n\n**Function**: `compute_acute_workload(snapshots: list[dict[str, Any]], end_date: str = <most recent date in snapshots>, normalized: bool = True)`\n\n```python\n# Get daily snapshots for a two week time period.\nsnapshots = client.get_snapshots(\n    start_date="2022-05-01", end_date="2022-05-14"\n)\n\n# Make sure to access the desired user from snapshots\nnorm_acute_workload = pulse.compute_acute_workload(\n    snapshots[client.user_id]\n)\n\n# Compute acute workload using unnormalized values\nacute_workload = pulse.compute_acute_workload(\n    snapshots[client.user_id], normalized=False\n)\n```\n\nAcute workload is the weighted average of one-day workloads over a 9-day period. The weights for the nine days are defined by `pulse.ACUTE_WEIGHTS`:\n\n```python\nACUTE_WEIGHTS = [1.3, 1.225, 1.15, 1.075, 1.0, 0.925, 0.85, 0.775, 0.7]\n```\n\nwhere the current day is multiplied by 1.3 and the last day is multipled by 0.7.\n\nThe divisor for acute workload is usually 9, but it can be less if there have been less than 7 days of throwing (3 after 1 day of throwing, 4 after 2 days, ..., 9 after 7+ days). This function assumes that the dates in `snapshots` are the only days of throwing and will adjust the acute divisor accordingly.\n\n### Compute Chronic Workload\n\nCompute chronic workload using `dailyWorkload` or `normDailyWorkload` values from daily snapshots returned by the `get_snapshots()` endpoint.\n\n**Function**: `compute_chronic_workload(snapshots: list[dict[str, Any]], end_date: str = <most recent date in snapshots>, normalized: bool = True)`\n\n```python\n# Get daily snapshots for a one month time period.\nsnapshots = client.get_snapshots(\n    start_date="2022-05-01", end_date="2022-06-01"\n)\n\n# Make sure to access the desired user from snapshots\nnorm_chronic_workload = pulse.compute_chronic_workload(\n    snapshots[client.user_id]\n)\n\n# Compute acute workload using unnormalized values\nchronic_workload = pulse.compute_chronic_workload(\n    snapshots[client.user_id], normalized=False\n)\n```\n\nChronic workload is the average of one-day workloads over a 28-day period.\n\nThe divisor for chronic workload is usually 28, but it can be less if there have been less than 24 days of throwing (5 after 1 day of throwing, 6 after 2 days, ..., 28 after 24+ days). This function assumes that the dates in `snapshots` are the only days of throwing and will adjust the chronic divisor accordingly.\n\n### Compute Acute/Chronic Workload Ratio\n\nCompute acute/chronic workload ratio using `dailyWorkload` or `normDailyWorkload` values from daily snapshots returned by the `get_snapshots()` endpoint.\n\n**Function**: `compute_acr(snapshots: list[dict[str, Any]], end_date: str = <most recent date in snapshots>, normalized: bool = True)`\n\n```python\n# Get daily snapshots for a one month time period.\nsnapshots = client.get_snapshots(\n    start_date="2022-05-01", end_date="2022-06-01"\n)\n\n# Make sure to access the desired user from snapshots\nnorm_acr = pulse.compute_acr(snapshots[client.user_id])\n\n# Compute ACR using unnormalized values\nacr = pulse.compute_acr(snapshots[client.user_id], normalized=False)\n```\n\nAcute/chronic workload ratio is the quotient of acute workload and chronic workload over a 28-day period.\n\n## Additional Resources\n\nYou can learn more about Pulse and throwing workload at the following links:\n\n- [Using Pulse to Define Throwing Workload](https://www.drivelinebaseball.com/2020/04/what-is-throwing-workload/)\n- [Optimized Acute Workload Computation for Baseball Pitchers](https://www.researchgate.net/publication/336345883_OPTIMIZED_ACUTE_WORKLOAD_COMPUTATION_FOR_BASEBALL_PITCHERS_COUPLED_9-DAY_EXPONENTIALLY_WEIGHTED_AVERAGES_WITH_DYNAMIC_DIVISORS)\n',
    'author': 'Josh Hejka',
    'author_email': 'josh@joshhejka.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/hedgertronic/pulse-throw',
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
