from abc import ABCMeta
import contextlib
import io
import logging
import os
import shlex
import subprocess
import tarfile
import time
import tempfile

import docker
import paramiko

LOG = logging.getLogger("pubtools.quay")


# Docker client is called differently based on the version used. Unify the calls.
# Only use old call if version 1.X.X is used
class APIClient(docker.APIClient if int(docker.__version__[0]) > 1 else docker.client.Client):
    """Unify the call of Docker client for old and new version."""

    pass


# Python 2.6 version of paramiko doesn't support the usage
# of SSHClient as a context manager. This wrapper adds the functionality
@contextlib.contextmanager
def open_ssh_client():
    """Use SSHClient as a context manager."""
    client = paramiko.client.SSHClient()
    try:
        yield client
    finally:
        client.close()


class Commands(object):
    """Set of skopeo commands ran on executor."""

    def __init__(self, executor):
        """Commands initializer.

        Args:
            executor(ContainerExecutorBearer):
                Executor implementation instance.
        """
        self.executor = executor


class Executor(object):
    """
    Base executor class.

    Aggregator of commands class and executor bearer class.
    To use create specific executor simply create new class like this:

    class MyExecutor(Executor):
        executor_bearer_cls = LocalExecutorBearer
        commands_cls = FirewallCommands

    Then after initialization, use executor instance like this:

    my_executor.commands.add_rule(.....)

    """

    __metaclass__ = ABCMeta
    executor_bearer_cls = None
    commands_cls = None

    def __init__(self, *args, **kwargs):
        """Executor initilizer.

        All arguments are passed to executor implementation class
        """
        self.executor = self.executor_bearer_cls(*args, **kwargs)
        self.commands = self.commands_cls(self.executor)

    def __enter__(self):
        """Run _enter method on executor implementaion."""
        self.executor._enter()
        return self

    def __exit__(self, exc_type, exc_value, exc_tb):
        """Run __exit__ on executor implementation."""
        return self.executor._exit(exc_type, exc_value, exc_tb)


class ExecutorBearer(object):
    """
    Base executor class.

    Abstract class used as base for implementation of
    specific exeutor bearers.
    """

    __metaclass__ = ABCMeta

    def __enter__(self):
        """Use the class as context manager. Returns instance upon invocation."""
        return self._enter()

    def _enter(self):
        """Execute at context manager enter."""
        return self

    def __exit__(self, exc_type, exc_value, exc_tb):
        """Cleanup when used as context manager. No-op by default."""
        self._exit(exc_type, exc_value, exc_tb)

    def _exit(self, exc_type, exc_value, exc_tb):
        """Execute at context manager exit."""
        pass

    def _run_cmd(self, cmd, err_msg=None, tolerate_err=False, stdin=None):
        """Run a bash command."""
        raise NotImplementedError  # pragma: no cover"

    def _add_file(self, data, file_name):
        """
        Add a text file to tmp location.

        The primary use-case is to store a secret.
        Args:
            data (str):
                Data that should be stored in the container.
            file_name (str):
                Name of the file.
        """
        raise NotImplementedError  # pragma: no cover"


class LocalExecutorBearer(ExecutorBearer):
    """Run commands locally."""

    def __init__(self, params={}):
        """
        Initialize.

        Args:
            params (dict):
                Custom parameters to be applied when running the shell commands.
        """
        self.params = params
        self.params.setdefault("universal_newlines", True)
        self.params.setdefault("stderr", subprocess.PIPE)
        self.params.setdefault("stdout", subprocess.PIPE)
        self.params.setdefault("stdin", subprocess.PIPE)

    def _run_cmd(self, cmd, err_msg=None, tolerate_err=False, stdin=None):
        """
        Run a command locally.

        Args:
            cmd (str):
                Shell command to be executed.
            error_msg (str):
                Error message written when the command fails.
            tolerate_err (bool):
                Whether to tolerate a failed command.
            stdin (str):
                String to send to standard input for a command.

        Returns (str, str):
            Tuple of stdout and stderr generated by the command.
        """
        err_msg = err_msg or "An error has occured when executing a command."

        p = subprocess.Popen(shlex.split(cmd), **self.params)
        out, err = p.communicate(input=stdin)

        if p.returncode != 0 and not tolerate_err:
            LOG.error("Command {0} failed with {1}".format(cmd, err))
            raise RuntimeError(err_msg)

        return out, err

    def _add_file(self, data, file_name):
        """
        Add a text file to tmp location.

        The primary use-case is to store a secret.
        Args:
            data (str):
                Data that should be stored in the container.
            file_name (str):
                Name of the file.
        """
        encoded_data = data.encode("utf-8")
        with open("/tmp/%s" % file_name, "w") as tmpf:
            tmpf.write(encoded_data)


class RemoteExecutorBearer(ExecutorBearer):
    """Run commands remotely via SSH."""

    def __init__(
        self,
        hostname,
        username=None,
        key_filename=None,
        password=None,
        port=None,
        accept_unknown_host=True,
    ):
        """
        Initialize.

        Args:
            hostname (str):
                Host to connect to.
            username (str):
                Username to authenticate as. Defaults to local username.
            key_filename (str):
                Path to a private key for authentication. Default location will be used if omitted.
            password (str):
                Password for ssh authentication. Has lower precedence than private key.
            port (int):
                Optional port of the host.
            accept_unknown_host (bool):
                Whether to accept an unknown host key. Defaults to True.
        """
        self.hostname = hostname
        self.username = username
        self.key_filename = key_filename
        self.password = password
        if accept_unknown_host:
            self.missing_host_policy = paramiko.client.WarningPolicy()
        else:
            self.missing_host_policy = paramiko.client.RejectPolicy()
        self.port = port if port else 22

    def _run_cmd(self, cmd, err_msg=None, tolerate_err=False, stdin=None):
        """
        Run a command remotely via SSH.

        Args:
            cmd (str):
                Shell command to be executed.
            error_msg (str):
                Error message written when the command fails.
            tolerate_err (bool):
                Whether to tolerate a failed command.
            stdin (str):
                String to send to standard input for a command.

        Returns (str, str):
            Tuple of stdout and stderr generated by the command.
        """
        err_msg = err_msg or "An error has occured when executing a command."
        with open_ssh_client() as client:
            client.load_system_host_keys()
            client.set_missing_host_key_policy(self.missing_host_policy)
            client.connect(
                self.hostname,
                username=self.username,
                port=self.port,
                password=self.password,
                key_filename=self.key_filename,
            )

            ssh_in, out, err = client.exec_command(cmd)
            if stdin:
                ssh_in.channel.send(stdin)
                ssh_in.channel.shutdown_write()

            out_text = out.read().decode("utf-8")
            err_text = err.read().decode("utf-8")
            if out.channel.recv_exit_status() != 0 and not tolerate_err:
                LOG.error("Command {0} failed with {1}".format(cmd, err_text))
                raise RuntimeError(err_msg)

        return out_text, err_text

    def _add_file(self, data, file_name):
        """
        Add a text file to the running container.

        The primary use-case is to store a secret which will be accessed from inside the container.
        File will be stored in the path /tmp/<file_name>.

        Args:
            data (str):
                Data that should be stored in the container.
            file_name (str):
                Name of the file.
        """
        with open_ssh_client() as client:
            client.load_system_host_keys()
            client.set_missing_host_key_policy(self.missing_host_policy)
            client.connect(
                self.hostname,
                username=self.username,
                port=self.port,
                password=self.password,
                key_filename=self.key_filename,
            )
            sftp = client.open_sftp()
            with tempfile.TemporaryFile() as tmpfp:
                encoded_data = data.encode("utf-8")
                tmpfp.write(encoded_data)
                tmpfp.close()
                sftp.put(tmpfp.name, remote_path="/tmp/%s" % file_name)
                sftp.chmod("/tmp/%s" % file_name, 600)


class ContainerExecutorBearer(ExecutorBearer):
    """Run commands in a Docker container."""

    def __init__(
        self,
        image,
        base_url="unix://var/run/docker.sock",
        timeout=None,
        verify_tls=False,
        cert_path=None,
    ):
        """
        Initialize.

        Args:
            image (str):
                Path to an image which will be used for performing skopeo operations. Must be
                downloadable by Docker.
            base_url (str):
                Base URL of the Docker client.
            timeout (int):
                Default timeout for API calls, in seconds.
            verify_tls (bool):
                Whether to use TLS verification.
            cert_path (str|None):
                Custom path to TLS certificates. If not specified, '~/.docker' is used.
        """
        self.image = image
        api_kwargs = self.get_api_client_args(
            base_url=base_url,
            timeout=timeout,
            tls=verify_tls,
            cert_path=cert_path,
        )
        self.client = APIClient(**api_kwargs)
        repo, tag = self.image.split(":", 1)
        self.client.pull(repo, tag=tag)
        self.container = self.client.create_container(self.image, detach=True, tty=True)
        self.client.start(self.container["Id"])

    def get_api_client_args(self, **api_args):
        """Return arguments for docker client class.

        Args:
            api_args(dict): dict of arguments for docker client class.
        """
        kwargs = {}
        kwargs["base_url"] = api_args["base_url"]
        kwargs["version"] = "auto"
        if api_args["timeout"]:
            kwargs["timeout"] = api_args["timeout"]
        if api_args.get("tls"):
            kwargs["base_url"] = kwargs["base_url"].replace("tcp://", "https://")
            cert_path = api_args["cert_path"] or os.path.join(os.path.expanduser("~"), ".docker")
            if os.path.isdir(cert_path):
                kwargs["tls"] = docker.tls.TLSConfig(
                    client_cert=(
                        os.path.join(cert_path, "cert.pem"),
                        os.path.join(cert_path, "key.pem"),
                    ),
                    verify=os.path.join(cert_path, "ca.pem"),
                )
        return kwargs

    def _exit(self, exc_type, exc_value, exc_tb):
        """Cleanup the container when used as a context manager."""
        self.client.remove_container(self.container["Id"], force=True)

    def _run_cmd(self, cmd, err_msg=None, tolerate_err=False, stdin=None):
        """
        Run a command locally.

        NOTE: Older versions of Docker API don't support demuxing of stdout and stderr.
        This means that data from both streams will be mixed together. To maintain compatibility
        with the other classes, same output will be returned twice as a tuple. Each string
        will contain the same mix of stdout and stderr messages.

        Args:
            cmd (str):
                Shell command to be executed.
            error_msg (str):
                Error message written when the command fails.
            tolerate_err (bool):
                Whether to tolerate a failed command.
            stdin (None|str):
                This parameter exists only for compatibility with parent class. Sending input
                to containers is not supported.

        Returns (str, str):
            Tuple of stdout and stderr generated by the command.
        """
        err_msg = err_msg or "An error has occured when executing a command."
        cmd_exec = self.client.exec_create(self.container["Id"], cmd)
        # Unfortunately, older versions of Docker API don't support demuxing stdout and stderr
        stdout = self.client.exec_start(cmd_exec["Id"])
        if self.client.exec_inspect(cmd_exec["Id"]).get("ExitCode") != 0 and not tolerate_err:
            LOG.error("Command {0} failed with {1}".format(cmd, stdout))
            raise RuntimeError(err_msg)

        if stdout is None:
            stdout = b""
        out_str = stdout.decode("utf-8")

        return (out_str, out_str)

    def _add_file(self, data, file_name):
        """
        Add a text file to the running container.

        The primary use-case is to store a secret which will be accessed from inside the container.
        File will be stored in the path /tmp/<file_name>.

        Args:
            data (str):
                Data that should be stored in the container.
            file_name (str):
                Name of the file.
        """
        data_stream = io.BytesIO()
        data_tar = tarfile.TarFile(fileobj=data_stream, mode="w")
        encoded_data = data.encode("utf-8")
        tarinfo = tarfile.TarInfo(name=file_name)
        tarinfo.size = len(encoded_data)
        tarinfo.mtime = time.time()
        data_tar.addfile(tarinfo, io.BytesIO(encoded_data))
        data_tar.close()

        data_stream.seek(0)
        success = self.client.put_archive(
            container=self.container["Id"], path="/tmp", data=data_stream
        )

        if not success:
            raise RuntimeError("File was not successfully added to the container")
