"""
Test step definitions for behave tests generated by the Omnata Plugin Devkit
"""
import datetime
import importlib
import json
import logging
import os
import sys
import unittest
from types import ModuleType
from typing import List
from omnata_plugin_devkit.snowcli.cli.app.snow_connector import connect_to_snowflake
from snowflake.snowpark import Session
import pandas
import vcr

# monkey-patch time.sleep to avoid waiting in tests
import time
def dont_sleep(seconds):
    pass
time.sleep = dont_sleep

# see (https://github.com/behave/behave/issues/641)
from behave import given, step, then, when  # pylint: disable=no-name-in-module
from behave_pandas import table_to_dataframe
from omnata_plugin_runtime.configuration import (
    InboundSyncConfigurationParameters,
    OutboundSyncConfigurationParameters,
    StoredStreamConfiguration,
)
from omnata_plugin_runtime.logging import OmnataPluginLogHandler
from omnata_plugin_runtime.omnata_plugin import InboundSyncRequest, OutboundSyncRequest
from pandas.testing import assert_frame_equal
from pydantic import TypeAdapter

case = unittest.TestCase()
from logging import getLogger

logger = getLogger(__name__)

@given("the following records")
def step_impl_records(context):
    logger.info(f"context.table: {context.table}")
    logger.info(f"context.table.headings: {context.table.headings}")
    context.source_records = table_to_dataframe(context.table)
    logger.info(f"source dataframe: {context.source_records}")


@given("the following streams state")
def step_impl_streams_state(context):
    context.streams_state = {}
    for row in context.table:
        if "Stream" not in row.headings:
            raise ValueError("Streams state must have a Stream column")
        if "Value" not in row.headings:
            raise ValueError("Streams state must have a Value column")
        context.streams_state[row["Stream"]] = json.loads(row["Value"])
    logger.info(f"streams state: {context.streams_state}")


@when("we perform an outbound sync with configuration parameters")
def step_impl_outbound_sync(context):
    if context.plugin_class is None:
        raise ValueError(
            'You must define which plugin class and module is to be used ("we use the x class from the y module")'
        )
    strategy = None
    connection_parameters = {}
    connection_secrets = {}
    sync_parameters = {}
    field_mappings = []
    api_limits = []
    connection_method = None

    for row in context.table:
        if row["Property"] == "strategy":
            strategy = json.loads(row["Value"])
        elif row["Property"] == "connection_method":
            connection_method = row["Value"]
        elif row["Property"] == "connection_parameters":
            connection_parameters = json.loads(row["Value"])
        elif row["Property"] == "connection_secrets":
            connection_secrets = json.loads(row["Value"])
        elif row["Property"] == "sync_parameters":
            sync_parameters = json.loads(row["Value"])
        elif row["Property"] == "field_mappings":
            field_mappings = json.loads(row["Value"])
        elif row["Property"] == "api_limits":
            api_limits = json.loads(row["Value"])
        else:
            raise ValueError(f"Unknown apply parameter {row['Property']}")

    parameters = OutboundSyncConfigurationParameters.parse_obj(
        {
            "sync_strategy": strategy,
            "connection_method": connection_method,
            "connection_parameters": connection_parameters,
            "connection_secrets": connection_secrets,
            "sync_parameters": sync_parameters,
            "field_mappings": field_mappings,
        }
    )
    # With API Limits, we remove rate limits to remove needless waiting
    # TODO: override concurrency setting due to pyvcr's thread non-safety
    context.plugin_instance = context.plugin_class()
    outbound_sync_request = OutboundSyncRequest(
        run_id=None,
        session=None,
        source_app_name=None,
        records_schema_name="",
        records_table_name="",
        results_schema_name="",
        results_table_name="",
        plugin_instance=context.plugin_instance,
        api_limits=api_limits,
        rate_limit_state_all={},
        rate_limit_state_this_sync_and_branch={},
        # make it 5 mins since we should be using vcr
        run_deadline=datetime.datetime.now(datetime.timezone.utc) + datetime.timedelta(minutes=5),
        development_mode=True,
        test_replay_mode=True,
    )

    outbound_sync_request._prebaked_record_state = (
        context.source_records
    )  # pylint: disable=protected-access
    context.error = None
    try:
        context.sync_direction = "outbound"
        context.sync_outbound_result = context.plugin_instance.sync_outbound(
            parameters, outbound_sync_request
        )
        # when using a managed_outbound_processing decorator, the results aren't returned from sync_outbound
        if context.sync_outbound_result is None:
            context.sync_outbound_result = outbound_sync_request.get_queued_results()
    except Exception as exception:  # pylint: disable=broad-except
        context.error = exception
        logging.exception(exception)


@when("we perform an inbound sync with configuration parameters")
def step_impl_inbound_sync(context):
    if context.plugin_class is None:
        raise ValueError(
            'You must define which plugin class and module is to be used ("we use the x class from the y module")'
        )
    connection_parameters = {}
    connection_secrets = {}
    sync_parameters = {}
    streams = []
    api_limits = []
    connection_method = None
    session = None
    if hasattr(context, "session"):
        session = context.session

    for row in context.table:
        if row["Property"] == "connection_method":
            connection_method = row["Value"]
        elif row["Property"] == "connection_parameters":
            connection_parameters = json.loads(row["Value"])
        elif row["Property"] == "connection_secrets":
            connection_secrets = json.loads(row["Value"])
        elif row["Property"] == "sync_parameters":
            sync_parameters = json.loads(row["Value"])
        elif row["Property"] == "streams":
            streams = json.loads(row["Value"])
        elif row["Property"] == "api_limits":
            api_limits = json.loads(row["Value"])
        else:
            raise ValueError(f"Unknown apply parameter {row['Property']}")

    parameters = InboundSyncConfigurationParameters.model_validate(
        {
            "connection_method": connection_method,
            "connection_parameters": connection_parameters,
            "connection_secrets": connection_secrets,
            "sync_parameters": sync_parameters,
        }
    )
    # With API Limits, we remove rate limits to remove needless waiting
    # TODO: override concurrency setting due to pyvcr's thread non-safety
    context.plugin_instance = context.plugin_class()
    streams_list = TypeAdapter(List[StoredStreamConfiguration]).validate_python(streams)
    for stream in streams_list:
        if stream.stream_name in context.streams_state:
            stream.latest_state = context.streams_state[stream.stream_name]
    omnata_log_handler = OmnataPluginLogHandler(
        session=None,
        sync_id=None,
        sync_branch_id=None,
        connection_id=None,
        sync_run_id=None,
    )
    inbound_sync_request = InboundSyncRequest(
        run_id=None,
        session=session,
        source_app_name=None,
        results_schema_name="",
        results_table_name="",
        plugin_instance=context.plugin_instance,
        api_limits=api_limits,
        rate_limit_state_all={},
        rate_limit_state_this_sync_and_branch={},
        run_deadline=datetime.datetime.now(datetime.timezone.utc) + datetime.timedelta(minutes=5),
        development_mode=True,
        streams=streams_list,
        test_replay_mode=True,
        omnata_log_handler=omnata_log_handler
    )

    # outbound_sync_request._prebaked_record_state = context.source_records
    context.error = None
    try:
        context.sync_direction = "inbound"
        context.plugin_instance.sync_inbound(parameters, inbound_sync_request)
        if (
            len(inbound_sync_request._apply_results) == 0
        ):  # pylint: disable=protected-access
            raise ValueError("No results returned from sync_inbound")
        context.sync_inbound_result = (
            inbound_sync_request._apply_results
        )  # pylint: disable=protected-access
    except Exception as exception:  # pylint: disable=broad-except
        context.error = exception
        logging.exception(exception)


def load_script_from_file(app_name, action_name):
    f = open(os.path.join("scripts", app_name, f"{action_name}.py"), encoding="utf-8")
    script_contents = f.read()
    mod = ModuleType("whatever.py", "")
    exec(script_contents, mod.__dict__)  # pylint: disable=exec-used
    return mod

@step("we attach a Snowpark session to the sync request")
def step_impl_attach_snowpark_session(context: any):
    snowflake_connection = connect_to_snowflake(connection_name="test")
    builder = Session.builder
    builder._options["connection"] = snowflake_connection
    context.session = builder.create()

@step("we use the {plugin_class} class from the {plugin_module} module")
def step_impl_use_plugin_class(context, plugin_class, plugin_module):
    # Assuming the module is defined in the parent of the tests directory
    sys.path.insert(
        0,
        os.path.abspath(
            os.path.join(context._root["config"].base_dir, "..")
        ),  # pylint: disable=protected-access
    )
    module = importlib.import_module(plugin_module)
    context.plugin_class = getattr(module, plugin_class)


@step("when matching requests, we ignore query parameters {params}")
def step_impl_ignore_params(context, params):
    context.ignore_query_params = params.split(",")

@step("we use the HTTP recordings from {filename}")
def step_impl_use_http_recordings(context, filename):
    if hasattr(context,"cassette"):
        context.cassette.__exit__(None, None, None)
    file_name = os.path.join(
        context._root["config"].base_dir,
        "vcr_cassettes",
        filename,  # pylint: disable=protected-access
    )
    logger.info(f"using cassette at {file_name}")
    my_vcr = vcr.VCR(record_mode="none")
    if "VCRPY_LOG_LEVEL" in os.environ:
        logging.basicConfig(
            format="%(asctime)s %(levelname)-8s %(message)s",
            level=logging.INFO,
            datefmt="%Y-%m-%d %H:%M:%S",
        )
        vcr_log = logging.getLogger("vcr")
        vcr_log.setLevel(
            logging._nameToLevel[os.environ["VCRPY_LOG_LEVEL"]]
        )  # pylint: disable=protected-access
    filter_query_params = None
    if hasattr(context, "ignore_query_params"):
        filter_query_params = context.ignore_query_params
    context.cassette = my_vcr.use_cassette(file_name, filter_query_parameters=filter_query_params)
    context.cassette.__enter__()


@step("the response will be")
def step_impl_response_outbound(context):
    expected_result = table_to_dataframe(context.table)
    # cast IDENTIFIER to string (it appears as object for some reason)
    expected_result["IDENTIFIER"] = expected_result["IDENTIFIER"].astype(str)
    expected_result = (
        expected_result.sort_values(by=["IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    context.sync_outbound_result = (
        context.sync_outbound_result.sort_values(by=["IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    # because behave tables work best with a string representation of json, we'll convert the results objects to string
    # expected_result['RESULT'] = expected_result['RESULT'].apply(json.dumps)
    context.sync_outbound_result["RESULT"] = context.sync_outbound_result[
        "RESULT"
    ].apply(json.dumps)
    assert list(expected_result.columns) == list(
        context.sync_outbound_result.columns
    ), f"Column headings didn't match. Expected: {expected_result.columns}, actual: {context.sync_outbound_result.columns}"
    pandas.set_option("display.max_columns", 10)
    pandas.set_option("display.width", 150)
    logger.info(f"expected_result: {expected_result}")
    logger.info(f"expected_result single: {expected_result['RESULT'][0]}")
    logger.info(f"expected_result dtypes: {expected_result.dtypes}")
    logger.info(f"apply_result: {context.sync_outbound_result}")
    logger.info(f"apply_result single: {context.sync_outbound_result['RESULT'][0]}")
    logger.info(f"apply_result dtypes: {context.sync_outbound_result.dtypes}")
    logger.info(f"differences: {expected_result.compare(context.sync_outbound_result)}")
    case.assertCountEqual(
        expected_result.columns.to_list(),
        context.sync_outbound_result.columns.to_list(),
    )
    assert_frame_equal(expected_result, context.sync_outbound_result)


@step("the response for the {stream_name} stream will be")
def step_impl_response_inbound(context, stream_name):
    expected_result = table_to_dataframe(context.table)
    expected_result = (
        expected_result.sort_values(by=["APP_IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    if stream_name not in context.sync_inbound_result:
        #logger.info(context.sync_inbound_result)
        raise ValueError(
            f"Stream {stream_name} not found in sync_inbound_result. Keys: {list(context.sync_inbound_result.keys())}"
        )
    
    sync_inbound_result_for_stream:pandas.DataFrame = (
        pandas.concat(context.sync_inbound_result[stream_name])
        .sort_values(by=["APP_IDENTIFIER"])
        .sort_index(axis=1)
        .reset_index(drop=True)
    )
    # convert APP_IDENTIFIER to string (it appears as object for some reason)
    sync_inbound_result_for_stream["APP_IDENTIFIER"] = sync_inbound_result_for_stream[
        "APP_IDENTIFIER"
    ].astype(str)
    # re-sort by APP_IDENTIFIER
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_values(by=["APP_IDENTIFIER"]).sort_index(axis=1).reset_index(drop=True)
    # because behave tables work best with a string representation of json, we'll convert the results objects to string
    # expected_result['RESULT'] = expected_result['RESULT'].apply(json.dumps)
    #sync_inbound_result_for_stream["RECORD_DATA"] = sync_inbound_result_for_stream[
    #    "RECORD_DATA"
    #].apply(json.dumps)
    
    # exclude RETRIEVE_DATE column from both dataframes, naturally they will differ
    expected_result = expected_result.drop(columns=["RETRIEVE_DATE"])
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.drop(
        columns=["RETRIEVE_DATE"]
    )
    # also convert the expected_result APP_IDENTIFIER to string
    expected_result["APP_IDENTIFIER"] = expected_result["APP_IDENTIFIER"].astype(str)
    # in order to not break existing tests, if the actual result contains the IS_DELETED column but the expected result doesn't, we'll remove it from the actual result
    if "IS_DELETED" in sync_inbound_result_for_stream.columns and "IS_DELETED" not in expected_result.columns:
        sync_inbound_result_for_stream = sync_inbound_result_for_stream.drop(columns=["IS_DELETED"])

    #expected_result = expected_result.sort_index(axis=1)
    #sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_index(axis=1)
    logger.info("Setting index of expected_result to APP_IDENTIFIER")
    expected_result.set_index("APP_IDENTIFIER", inplace=True, drop=False)
    logger.info("Sorting index of expected result")
    expected_result = expected_result.sort_index()
    logger.info("Setting index of sync_inbound_result to APP_IDENTIFIER")
    sync_inbound_result_for_stream.set_index("APP_IDENTIFIER", inplace=True, drop=False)
    logger.info("Sorting index of sync_inbound_result")
    sync_inbound_result_for_stream = sync_inbound_result_for_stream.sort_index()
    logger.info(f"expected_result dtypes:\n{expected_result.dtypes}")
    logger.info(f"sync_inbound_result dtypes:\n{sync_inbound_result_for_stream.dtypes}")    
    #logger.info("Checking for duplicates in expected_result")
    #duplicates = expected_result[expected_result.duplicated(['APP_IDENTIFIER'], keep="first")]['APP_IDENTIFIER'].unique().tolist()
    #case.assertEqual(
    #    len(duplicates),0, f"Duplicate APP_IDENTIFIER in expected_result:\n{duplicates.tolist()}"
    #)
    #logger.info("Checking for duplicates in sync_inbound_result")
    #duplicates = sync_inbound_result_for_stream[sync_inbound_result_for_stream.duplicated(['APP_IDENTIFIER'], keep="first")]['APP_IDENTIFIER'].unique().tolist()
    #case.assertFalse(
    #    len(duplicates),0, f"Duplicate APP_IDENTIFIER in sync_inbound_result:\n{duplicates.tolist()}"
    #)
    # remove duplicates from expected_result and sync_inbound_result
    expected_result.drop_duplicates(subset=['APP_IDENTIFIER'], keep="first", inplace=True)
    sync_inbound_result_for_stream.drop_duplicates(subset=['APP_IDENTIFIER'], keep="first", inplace=True)
    logger.info("Resetting index of expected_result to match sync_inbound_result")
    expected_result = expected_result.reindex_like(sync_inbound_result_for_stream)
    assert list(expected_result.columns) == list(
        sync_inbound_result_for_stream.columns
    ), f"Column headings didn't match. Expected: {expected_result.columns}, actual: {sync_inbound_result_for_stream.columns}"
    pandas.set_option("display.max_columns", 10)
    pandas.set_option("display.width", 150)
    logger.info(f"expected_result:\n{expected_result}")
    logger.info(f"sync_inbound_result:\n{sync_inbound_result_for_stream}")
    case.assertCountEqual(
        expected_result.columns.to_list(), sync_inbound_result_for_stream.columns.to_list()
    )
    logger.info(f"differences:\n{expected_result.compare(sync_inbound_result_for_stream)}")
    assert_frame_equal(expected_result, sync_inbound_result_for_stream)


@then("no error will be raised")
def step_impl_no_error(context):
    assert (
        context.error is None
    ), f"Expected no error from action, instead got {context.error}"
