# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fpbot']

package_data = \
{'': ['*'], 'fpbot': ['data/*']}

install_requires = \
['SQLAlchemy>=1.4.39,<3.0.0',
 'astropy>=5.1,<6.0',
 'backoff>=2.1.2,<3.0.0',
 'fastapi>=0.79,<0.93',
 'keyring>=23.7.0,<24.0.0',
 'matplotlib>=3.5.2,<4.0.0',
 'mypy>=0.991,<1.1',
 'numpy>=1.23.1,<2.0.0',
 'pandas>=1.4.3,<2.0.0',
 'pymongo>=4.2.0,<5.0.0',
 'requests>=2.28.1,<3.0.0',
 'scipy>=1.8.1,<2.0.0',
 'sfdmap>=0.1.1,<0.2.0',
 'tdqm>=0.0.1,<0.0.2',
 'ztflc>=0.3.1,<0.4.0',
 'ztfquery>=1.18.4,<2.0.0']

extras_require = \
{':extra == "slack"': ['slackclient>=2.9.4,<3.0.0']}

entry_points = \
{'console_scripts': ['fp = forcedphotometry:run', 'fpbulk = bulk:main']}

setup_kwargs = {
    'name': 'fpbot',
    'version': '1.1.1',
    'description': 'Forced photometry pipeline for the Zwicky Transient Facility',
    'long_description': '[![DOI](https://zenodo.org/badge/DOI/10.5281/zenodo.7404998.svg)](https://doi.org/10.5281/zenodo.7404998)\n[![CI](https://github.com/simeonreusch/fpbot/actions/workflows/continous_integration.yml/badge.svg)](https://github.com/simeonreusch/fpbot/actions/workflows/continous_integration.yml)\n[![Coverage Status](https://coveralls.io/repos/github/simeonreusch/fpbot/badge.svg?branch=main)](https://coveralls.io/github/simeonreusch/fpbot?branch=main)\n\n# fpbot\n\nProvides a Forced Photometry Pipeline based on [ztfquery](https://github.com/mickaelrigault/ztfquery) and [ztflc](https://github.com/mickaelrigault/ztfquery), needs [IPAC](https://irsa.ipac.caltech.edu/account/signon/login.do?josso_back_to=https://irsa.ipac.caltech.edu/frontpage/&ts=517) as well as [AMPEL](https://github.com/ampelproject) access.\n\nNote: Requires Python >= 3.10. Also requires a MongoDB instance for storing the metadata, reachable under port 27017. This can be modified in database.py.\n\n## Installation\n\n1. Note that libpq-dev needs to be present. On Debian/Ubuntu, issue ```sudo apt install libpq-dev```. On Mac OS, run ```brew install postgresql```.\n\n2. Then install via: ```pip install fpbot```.\n\n3. If MongoDB is not present, it can easily be installed.\nOn Debian/Ubuntu, just follow this [instruction set](https://docs.mongodb.com/manual/tutorial/install-mongodb-on-debian/#install-mongodb-community-edition). After this, make sure the demon runs. Issue  ```sudo systemctl start mongod``` and ```sudo systemctl enable mongod```. On MacOS, make sure brew is present follow [this tutorial](https://docs.mongodb.com/manual/tutorial/install-mongodb-on-os-x/).\n\n4. fpbot requires an environment variable so it knows where to store the data. Include a line in your .bashrc or .zshrc like ```export ZTFDATA=\'/absolute/path/to/ZTF-data-folder/\'```. If you don\'t need AMPEL access, you are done!\n\n---\n\n5. If you want to use the [AMPEL API](https://ampel.zeuthen.desy.de/api/ztf/archive/v3/docs) for alert data (you don\'t have to!), you need credentials for the API. You can get these [here](https://ampel.zeuthen.desy.de/live/dashboard/tokens).\n\n6. NOTE: If you are planning to run fpbot on a headless system which does not provide the luxury of a systemwide keychain, please add ```export ZTFHUB_MODE=\'HEADLESS\'``` to your .bashrc or .zshrc. The pipeline will then uses `ztfquery`\'s base64-obfuscated password storage.\n\n## ALTERNATIVE: Use Docker container\nfpbot comes shipped with a Dockerfile and a docker-compose.yml. Use them to build the docker container (this includes all dependencies as well as a MongoDB instance). Note: You have to provide a .ztfquery file in the fpbot directory containing access data for ztfquery (see [ztfquery](https://github.com/mickaelrigault/ztfquery) or [ztflc](https://github.com/mickaelrigault/ztfquery) for details).\n\nFirst, clone this project: ```git clone https://github.com/simeonreusch/fpbot```\n\nThe container can be built by navigating to the just cloned directory and issuing\n\n```docker-compose build```\n\nin the directory containing 1) the Dockerfile, 2) the docker-compose.yml and 3) the .ztfquery credentials file and run with\n\n```docker-compose run -p 8000:8000 fpbot```. This exposes the web API to port 8000 of your local machine.\n\n### Troubleshooting\nMake sure that `ztfquery` and `ztflc` are installed with the latest version.\n\nIn case way too few images are downloaded, check your IRSA credentials. These are stored in `~.ztfquery`. If there is a problem with these, `ztfquery` will not complain but simply only download publicly accessible images.\n\n## Usage\n\n### By importing class\nAll functionality of the command-line tool is present in the class. Just call it according to the commands available in `pipeline.py`.\n\nFor example:\n\n```python\nfrom fpbot.pipeline import ForcedPhotometryPipeline\n\npl = ForcedPhotometryPipeline(\n    file_or_name="ZTF19aatubsj",\n    daysago=90,\n    nprocess=24\n)\n\npl.download()\npl.psffit()\npl.plot()\n```\n\n### By systemwide command (`fp name -operations --options`)\n\nAlways:\n\n`name` A ZTF name has to be provided, or an ASCII file containing one ZTF name in each line or an arbitrary name if followed by the ra/dec-option as to be provided.\n\noptionally:\n\n`-radec [RA DEC]`\tIf this is given, the name can be chosen arbitrarily (but a name MUST be provided). Radec must be given in a format that can be parsed by astropy; e.g. `-radec 218.487548 +40.243758`.\n\n#### Additional commands\n\n`-dl`        Downloads the images used for forced photometry from [IPAC](https://irsa.ipac.caltech.edu/account/signon/login.do?josso_back_to=https://irsa.ipac.caltech.edu/frontpage/&ts=517). Needs a valid IPAC account.\n\n`-fit`       Performs the PSF-photometry fit and generates plots of the lightcurve(s).\n\n`-plot`     Plots the lightcurve(s).\n\n`-plotflux`     Plots the lightcurve(s), but with flux instead of magnitude.\n\n`-saltfit`   Fits the lightcurve using SALT2 as provided by [sncosmo](https://github.com/sncosmo/).\n\n`-sciimg`  Experimental: Also downloads the science images from IPAC (note: to create thumbnails if specified)\n\n`-thumbnails` Experimental: Generates thumbnails for all science-images. Science images have to be downloaded (see `-sciimg`)\n\n#### Options\n\n`--nprocess [int]`  Specifies the number of processes spawned for parallel computing. Default is 4. Note: download is always performed with 32 processes in parallel, as IPAC upload-speed is the bottleneck there.\n\n`--daysago [int]`  Determines how old the photometric data should be. Default: all.\n\n`--daysuntil [int]`  Determines how new the photometric data should be. Default: all.\n\n`--snt [float]` Specifies the signal-to-noise ratio for plotting and SALT-fitting.\n\n`--magrange [float float]` Defines upper and lower magnitude bound for plotting the lightcurves; order is irrelevant.\n\n`--fluxrange [float float]` Defines lower and upper flux bound for plotting the flux lightcurves; order is irrelevant.\n\n#### Examples\n`fp ZTF19aatubsj` downloads this ZTF object, does forced photometry, plots it and saves it to the default directory in "forcephotometry" (ZTFDATA, located at $ZTFDATA in your .bashrc/.zshrc/..., see ztfquery doc).\n\n`fp example_download.dat -dl -fit` downloads the two lightcurves in the file `example_download.dat`, fits, but does not plots them.\n\n`fp ZTF19abimkwn -dl -fit -saltfit --nprocess 16` downloads all images for ZTF18abtmbaz found on IPAC, performs PSF-fitting, plots a lightcurve and fits the lightcurve with a SALT2 template with 16 processes in parallel.\n\n`fp supernovae.txt -dl` Downloads all difference images for ZTF transients found in supernovae.txt, each line a ZTFname. To get a cool example of ZTF lightcurves, issue: example_download.txt Note: Downloading the images usually takes a considerable amount of time.\n\n`fp this_looks_interesting -radec 143.3123 66.42342 -dl -fit -plot --daysago 10 -magrange 18 20` Downloads all images of the last ten days of the location given in ra and dec, performs PSF-fits and plots the lightcurve in the 18--20 magnitude range.\n\n### By systemwide bulk command (`fpbulk file.txt -operations --options`)\n`file.txt` must be an ASCII file containing one ZTF-ID per line. The usual options apply (e.g. `-dl`, `-fit`).\n\n## Requirements\n- [ztfquery](https://github.com/mickaelrigault/ztfquery) is used to download the image files from IPAC.\n- [ztflc](https://github.com/mickaelrigault/ztflc) is used for PSF-fitting.\n- [AMPEL](https://github.com/ampelproject) credentials are neccessary for the pipeline to work.\n\n## Notes\n### Slackbot\nThere is a bot for Slack included, based on the SlackRTM-API.\nYou have to create a classic Slack app for this, because the newer version depends on the Events API, which itself seems to need a web server to run.\nClassic slack Apps can be created [here](https://api.slack.com/apps?new_classic_app=1). Make sure not to convert to the new permission/privilege system in the process (Slack tries to push you towards it, be careful).\nAfter successfully setting up the App/bot and giving it permissions, change the bot-username to the one of your bot in start_slackbot.py and it should basically work (first start requires you to enter the bot- and bot-user credentials, also provided by Slack).\n\n### Saltfit module\nStill experimental! Performs saltfits on the generated lightcurves.\n\n### Resulting dataframe\nThe dataframes resulting after plotting (located at `ZTDATA/forcephotometry/plot/dataframes`) consists of the following columns:\n- **sigma(.err)**: The intrinsic error\n- **ampl(.err)**: The flux amplitude (error)\n- **fval**: Total minimized value\n- **chi2(dof)**: PSF-fit chi square (per degrees of freedom)\n- **Columns 9-39**: The science image header\n- **target_x/y**: pixel position of target\n- **data_hasnan**: Data contains NaN-values (should always be False)\n- **F0**: Zero point magnitude from header converted to flux\n- **Fratio(.err)**: Flux to flux zero point ratio (error)\n- **upper_limit**: For forced photometry result < signal to noise threshold, this is the limiting magnitude from the Marshal (see **maglim** column)\n- **mag(_err)**: Flux amplitude (error) converted to magnitude. For detections below signal to noise threshold, this is 99\n',
    'author': 'simeonreusch',
    'author_email': 'simeon.reusch@desy.de',
    'maintainer': 'Simeon Reusch',
    'maintainer_email': 'simeon.reusch@desy.de',
    'url': 'https://github.com/simeonreusch/fpbot',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.10,<4',
}


setup(**setup_kwargs)
