from bottle import request
from elasticapm.utils import compat, get_url_dict

HTTP_WITH_BODY = {"POST", "PUT", "PATCH", "DELETE"}


def get_data_from_request(server_request, capture_body=False, capture_headers=True):
    data = {
        "method": server_request.method,
        "socket": {
            "remote_address": server_request.remote_addr,
            "encrypted": "https" == server_request.urlparts[0],
        },
        "cookies": dict(**server_request.cookies),
        "url": get_url_dict(server_request.url),
    }
    if capture_headers:
        data["headers"] = dict(**server_request.headers)
        data["headers"].pop("Cookie", None)
    if request.method in HTTP_WITH_BODY:
        if request.content_type == "application/x-www-form-urlencoded":
            body = compat.multidict_to_dict(request.forms)
        elif request.content_type and request.content_type.startswith(
                "multipart/form-data"
        ):
            body = convert_list_in_dict(request.forms)
            if request.files:
                body["_files"] = convert_list_file_in_dict(request.files)
        elif request.json:
            body = request.json
        else:
            body = request.body.read().decode("utf-8")

        if body is not None:
            data["body"] = body if capture_body else "[REDACTED]"
    return data


def get_data_from_response(server_response, capture_headers=True):
    data = {"status_code": server_response.status_code}
    if capture_headers and server_response.headers:
        data["headers"] = {
            key: ";".join(server_response.headers.getall(key))
            for key in compat.iterkeys(server_response.headers)
        }
    return data


def convert_list_in_dict(values):
    return {key: value for (key, value) in values.items()}


def convert_list_file_in_dict(values):
    return {
        key: value.filename for (key, value) in values.items() if not value.filename
    }
