import re
import secrets
import string
from random import SystemRandom
from typing import ClassVar, Union

from faker import Faker
from rstr import Rstr

rs = Rstr(random=SystemRandom())  # type: ignore

RegexGeneratedStr = str
FakeEmail = str


class InfoType:
    expr: ClassVar[str] = NotImplemented
    name: ClassVar[str] = NotImplemented

    @classmethod
    def generate(cls) -> Union[RegexGeneratedStr, str]:
        return rs.xeger(cls.expr).strip()

    @classmethod
    def compiled(cls) -> re.Pattern:
        return re.compile(cls.expr)


class LongDigit(InfoType):
    min_length: int = 6
    expr = rf'\d{{{min_length},}}'
    name = 'LongDigit'

    @classmethod
    def generate(cls) -> str:
        return ''.join([secrets.choice(string.digits) for _ in range(cls.min_length)])


class Email(InfoType):
    """Email
    This InfoType catches email address and replaces them with a fake one generated by the Faker library.
    The regex for this has been lifted from http://emailregex.com/ which claims a 99.99% accuracy
    """

    name = 'Email'
    expr = (
        r"([a-z0-9!#$%&'*+/=?^_`"
        r"{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|\"(?:"
        r'[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])'
        r"*\")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\["
        r'(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|'
        r'[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|'
        r'\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])'
    )
    faker: Faker = Faker()

    @classmethod
    def generate(cls) -> FakeEmail:
        return cls.faker.safe_email()


class AusPostCode(InfoType):
    name = 'AusPostCode'
    expr = (
        r'(0[289][0-9]{2}\)'
        r'|([1345689][0-9]{3})'
        r'|(2[0-8][0-9]{2})'
        r'|(290[0-9])'
        r'|(291[0-4])'
        r'|(7[0-4][0-9]{2})'
        r'|(7[8-9][0-9]{2}))'
    )


class AusDriversLicence(InfoType):
    name = 'AusDriversLicence'
    expr = r'[A-Z0-9][0-9]{5,7}'


class AusPassport(InfoType):
    name = 'AusPassport'
    expr = r'[A-Z][0-9]{7}'


class AusTaxFileNumber(InfoType):
    name = 'AusTaxFileNumber'
    expr = '[0-9]{3}( ?)[0-9]{3}\1[0-9]{2,3}'


class AusLicensePlate(InfoType):
    name = 'AusLicencePlate'
    expr = r'(([a-zA-Z0-9]{3})([\s,-.]?)([a-zA-Z0-9]{3}))'

    @classmethod
    def generate(cls) -> RegexGeneratedStr:
        return rs.xeger(cls.expr).strip().upper()


AusInfoTypes = [
    Email,
    AusPassport,
    AusDriversLicence,
    AusTaxFileNumber,
    AusPostCode,
    AusLicensePlate,
    LongDigit,
]
