import os
import sys
import pathlib
import json
import shutil

sys.path.insert(
    0, 
    os.path.abspath(
        os.path.join(os.path.dirname(__file__), 
        '..')
    )
)

from click.testing import CliRunner
from fairscape_cli.__main__ import cli as fairscape_cli_app

###########################
# Test Metadata for ROCrate 
###########################


test_dataset = {
    "id": "ARK:APMS_embedding.MuSIC.1/5b3793b6-2bd0-4c51-9f35-c5cd7ddd366c.csv",
    "name": "AP-MS embeddings",
    "type": "schema:Dataset",
    "author": "Gygi lab (https://gygi.hms.harvard.edu/team.html)",
    "datePublished": "2021-04-23",
    "version": "1.0",
    "description": "Affinity purification mass spectrometer (APMS) embeddings for each protein in the study,  generated by node2vec predict.",
    "keywords": ["dataset", "gygi", "AP-MS"],
    "associatedPublication": 
        "Qin, Y. et al. A multi-scale map of cell structure fusing protein images and interactions",
    "additionalDocumentation": [
        "https://idekerlab.ucsd.edu/music/"
    ],
    "format": "CSV",
    "dataSchema": 
      'APMS_ID str, "APMS_1, APMS_2, ...", protein, embedding array of float X 1024',
    "derivedFrom": ["node2vec predict"],
    "generatedBy": [],
    "usedBy": ["create labeled training & test sets  random_forest_samples.py"],
    "contentUrl": "https://github.com/idekerlab/MuSIC/blob/master/Examples/APMS_embedding.MuSIC.csv"
}

test_software = {
    "id": "ARK:calibrate_pariwise_distance.1/467f5ebd-cb29-43a1-beab-aa2d50606eff.py",
    "name": "calibrate pairwise distance",
    "type": "evi:Software",
    "author": "Qin, Y.",
    "dateModified": "2021-06-20",
    "version": "1.0",
    "description": "script written in python to calibrate pairwise distance.",
    "keywords": ["python", "clustering"],
    "associatedPublication": "Qin, Y. et al. A multi-scale map of cell structure fusing protein images and interactions. Nature 600, 536–542 2021",
    "additionalDocumentation": ["https://idekerlab.ucsd.edu/music/"],
    "format": "py",
    "usedByComputation": ["ARK:compute_standard_proximities.1/f9aa5f3f-665a-4ab9-8879-8d0d52f05265"],
    "contentUrl": "https://github.com/idekerlab/MuSIC/blob/master/calibrate_pairwise_distance.py"
}
test_computation = {
    "id": "ARK:average_predicted_protein_proximities.1/c295abcd-8ad8-44ff-95e3-e5e65f1667da",
    "name": "average predicted protein proximities",
    "type": "evi:Computation",
    "runBy": "Qin, Y.",
    "dateCreated": "2021-05-23",
    "description": "Average the predicted proximities",
    "keywords": ["clustering", "ppi"],
    "usedSoftware":[
      "random_forest_output (https://github.com/idekerlab/MuSIC/blob/master/random_forest_output.py)"
    ],
    "usedDataset": [ 
    "IF_emd_1_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_1.pkl",
    "IF_emd_2_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_1.pkl",
    "IF_emd_1_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_2.pkl",
    "IF_emd_2_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_2.pkl",
"""Fold 1 proximities:
      IF_emd_1_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_3.pkl""",
    "IF_emd_2_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_3.pkl",
"""Fold 1 proximities:
      IF_emd_1_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_4.pkl""",
    "IF_emd_2_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_4.pkl",
"""Fold 1 proximities:
      IF_emd_1_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_5.pkl""",
"IF_emd_2_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_5.pkl"
    ],
"associatedPublication": "Qin, Y. et al. A multi-scale map of cell structure fusing protein images and interactions. Nature 600, 536–542 2021",
    "additionalDocumentation": ["https://idekerlab.ucsd.edu/music/"],
    "generated": [
    "averages of predicted protein proximities (https://github.com/idekerlab/MuSIC/blob/master/Examples/MuSIC_predicted_proximity.txt)"
    ]
}

class TestROCrateUtils():
    def test_read_rocrate_metadata(self):
        test_crate = ReadROCrateMetadata('./tests/data/test_crates/build_test_rocrate/ro-crate-metadata.json')

        assert test_crate.guid is not None


class TestROCrateSuccess():
    runner = CliRunner()
    rocrate_path = "./tests/data/automated_test_rocrate"
 
    def test_rocrate_create(self): 
        """ Remove existing ROCrate and then create a new rocrate
        """

        try:
            # remove existing crate
            shutil.rmtree(self.rocrate_path)
        except:
            pass
         

        crate_id = "ark:59853/UVA/B2AI/rocrate_test"
        crate_name = 'test rocrate'

        create_rocrate = [
            "rocrate", 
            "create", 
            f"--guid '{crate_id}'",
            f"--name '{crate_name}'",
            "--description 'A test crate for provenance of protien protien interactions of the MUSIC pipeline using a U2OS cell line'",
            "--keywords 'CM4AI'",
            "--keywords 'B2AI'",
            "--organization-name 'UVA'",
            "--project-name 'B2AI'",
            f"{self.rocrate_path}"
        ]
        
        result = self.runner.invoke(
            fairscape_cli_app, 
            ' '.join(create_rocrate) 
        )
        print(result.stdout)

        assert result.exit_code == 0
        assert "ark:59853/UVA/B2AI/rocrate_test" in result.stdout

        # check that the ro-crate-metadata.json is correct
        rocrate_metadata_path = self.rocrate_path + "/ro-crate-metadata.json"

        with open(rocrate_metadata_path, 'r') as metadata_file:
            rocrate_metadata = json.load(metadata_file)

        assert rocrate_metadata['@id'] == crate_id
        assert rocrate_metadata['name'] == crate_name


    def test_register_dataset(self):
        add_dataset = [
            "rocrate",
            "register",
            "dataset",
            f"--name '{test_dataset['name']}'",
            "--guid 'ark:59853/UVA/B2AI/rocrate_test/music_data'",
            "--name 'AP-MS embeddings'",
            f"--description '{test_dataset['description']}'" ,
            f"--keywords '{test_dataset['keywords'][0]}'",
            f"--keywords '{test_dataset['keywords'][1]}'",
            f"--keywords '{test_dataset['keywords'][2]}'",
            f"--date-published '{test_dataset['datePublished']}'",
            f"--author '{test_dataset['author']}'",
            "--version '1.0.0'",
            f"--associated-publication '{test_dataset['associatedPublication']}'",
            f"--additional-documentation '{test_dataset['additionalDocumentation'][0]}'",
            f"--data-format '{test_dataset['format']}'",
            f"--filepath '{self.rocrate_path + '/APMS_embedding_MUSIC.csv'}'",
            f"'{self.rocrate_path}'",
        ]

        print(' '.join(add_dataset))

        result = self.runner.invoke(
            fairscape_cli_app, 
            ' '.join(add_dataset) 
        )
        print(result.stdout)

        assert result.exit_code == 0


    def test_add_dataset(self):

        add_dataset = [
            "rocrate",
            "add",
            "dataset",
            f"--name '{test_dataset['name']}'",
            "--guid 'ark:59853/UVA/B2AI/rocrate_test/music_data'",
            "--name 'AP-MS embeddings'",
            f"--description '{test_dataset['description']}'" ,
            f"--keywords '{test_dataset['keywords'][0]}'",
            f"--keywords '{test_dataset['keywords'][1]}'",
            f"--date-published '{test_dataset['datePublished']}'",
            f"--author '{test_dataset['author']}'",
            "--version '1.0.0'",
            f"--associated-publication '{test_dataset['associatedPublication']}'",
            f"--additional-documentation '{test_dataset['additionalDocumentation'][0]}'",
            f"--data-format '{test_dataset['format']}'",
            "--source-filepath './tests/data/APMS_embedding_MUSIC.csv'",
            f"--destination-filepath '{self.rocrate_path +'/APMS_embedding_MUSIC.csv'}'",
            f"'{self.rocrate_path}'",
        ]

        print(' '.join(add_dataset))

        result = self.runner.invoke(
            fairscape_cli_app, 
            ' '.join(add_dataset) 
        )
        print(result.stdout)

        assert result.exit_code == 0


    def test_register_software_wo_contentURL(self):
        """ Register Software without contentURL
        """

        add_software = [
            "rocrate",
            "register",
            "software",
            "--guid 'ark:59853/UVA/B2AI/rocrate_test/music_software_no_content_url'",
            "--name 'MuSIC'",
            f"--author '{test_software['author']}'",
            "--version '1.0'",
            f"--description '{test_software['description']}'",
            f"--keywords '{test_software['keywords'][0]}'",
            f"--associated-publication '{test_software['associatedPublication']}'",
            "--file-format '.py'",
            f"--date-modified '{test_software['dateModified']}'",
            f"'{self.rocrate_path}'",
        ]

        result = self.runner.invoke(
            fairscape_cli_app, 
            ' '.join(add_software) 
        )
        print(result.stdout)

        assert result.exit_code == 0
        

    def test_register_software(self):

        add_software = [
            "rocrate",
            "register",
            "software",
            "--guid ark:59853/UVA/B2AI/rocrate_test/music_software",
            "--name MuSIC",
            f"--author '{test_software['author']}'",
            "--version '1.0'",
            f"--description '{test_software['description']}'",
            f"--keywords '{test_software['keywords'][0]}'",
            f"--associated-publication '{test_software['associatedPublication']}'",
            "--file-format '.py'",
            f"--date-modified '{test_software['dateModified']}'",
            f"'{self.rocrate_path}'",
        ]

        result = self.runner.invoke(
            fairscape_cli_app, 
            ' '.join(add_software) 
        )
        print(result.stdout)

        assert result.exit_code == 0
        

    def test_add_software(self):

        add_software = [
            "rocrate",
            "add",
            "software",
            "--guid ark:59853/UVA/B2AI/rocrate_test/music_software",
            "--name MuSIC",
            f"--author '{test_software['author']}'",
            "--version '1.0'",
            f"--description '{test_software['description']}'",
            f"--keywords '{test_software['keywords'][0]}'",
            f"--associated-publication '{test_software['associatedPublication']}'",
            "--file-format '.py'",
            f"--date-modified '{test_software['dateModified']}'",
            "--source-filepath './tests/data/calibrate_pairwise_distance.py'",
            f"--destination-filepath '{self.rocrate_path + '/calibrate_pairwise_distance.py'}'",
            f"'{self.rocrate_path}'",
        ]

        result = self.runner.invoke(
            fairscape_cli_app, 
            ' '.join(add_software) 
        )
        print(result.stdout)

        assert result.exit_code == 0


    def test_add_computation(self):
        """
        Add a computation wherein a single software was used on a single dataset
        producing a single result
        """


        #software = [f'\"{element}\"' for element in test_computation['usedSoftware']]
        #datasets = [f'\"{element}\"' for element in test_computation['usedDataset']]
        #generated = [f'\"{element}\"' for element in test_computation['generated']]

        add_computation = [
            "rocrate",
            "register",
            "computation",
            "--guid 'ark:59853/UVA/B2AI/rocrate_test/music_test_run'",
            f"--name '{test_computation['name']}'",
            "--run-by 'Max Levinson'",
            "--date-created '03-17-2023'",
            "--description 'test run of music pipeline using example data'",
            f"--keywords '{test_computation['keywords'][0]}'",
            f"--keywords '{test_computation['keywords'][1]}'",
            #f"--used-software '[{','.join(software)}]'",
            #f"--used-dataset '[{','.join(datasets)}]'",
            "--command 'wingardium leviosa'",
            f"--used-software ['{test_computation['usedSoftware'][0]}']",
            "--used-dataset 'IF_emd_1_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_1.pkl'",
            "--used-dataset 'IF_emd_2_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_1.pkl'",
            "--used-dataset 'IF_emd_1_APMS_emd_1.RF_maxDep_30_nEst_1000.fold_2.pkl'",
            f"--generated ['{test_computation['generated'][0]}']",
            f"'{self.rocrate_path}'",
        ]

        print(' '.join(add_computation))

        result = self.runner.invoke(
            fairscape_cli_app, 
            ' '.join(add_computation) 
        )
        print(result.stdout)

        assert result.exit_code == 0


class TestGenerateModels():
    cratePath = pathlib.Path('tests/data/automated_test_rocrate/generate_crate/')

    def test_0_generate_rocrate(self):
        passed_crate = ROCrate(
                name='null',
                organizationName = 'org',
                projectName = 'proj',
                description = 'a null ro crate',
                keywords = ['test', 'keywords'],
                path = self.cratePath, 
                metadataGraph = []
        )

        passed_crate.initCrate()

        # assert that the crate was created
        # by asserting that ro-crate-metadata.json exists at the specified path
        metadata_path = self.cratePath / 'ro-crate-metadata.json'
        assert metadata_path.exists()

        # check that the identifiers are proper
        passed_crate.

    def test_1_generate_dataset(self):
        local_path = pathlib.Path('tests/data/crates/1.cm4ai_chromatin_mda-mb-468_untreated_apmsloader_initialrun0.1alpha/ppi_edgelist')
        remote_path = 'https://g-9b3b6e.9ad93.a567.data.globus.org/Data/cm4ai_0.1alpha/cm4ai_chromatin_mda-mb-468_paclitaxel_crispr_1channel_0.1_alpha.tar.gz'
        remote_dataset = GenerateDataset(
            guid=None, 
            url='https://cm4ai.org',
            author='null',
            description='test',
            name='example',
            keywords=['cm4ai', 'b2ai', 'download'],
            datePublished='02-29-2024',
            version='1',
            associatedPublication=None,
            additionalDocumentation=None,
            dataFormat='tsv',
            schema=None,
            derivedFrom=None,
            usedBy=None,
            filepath=remote_path,
            cratePath=self.cratePath
        )

    def test_2_generate_software(self):
        remote_software = GenerateSoftware(
            guid=None, 
            url='https://github.com/idekerlab/cellmaps_imagedownloader',
            author='Chris Churas',
            description='Software Tool to download IF images from HPA for the Cellmaps for AI project',
            name='Cellmaps Imagedownloader',
            keywords=['cm4ai', 'b2ai', 'download'],
        )
        pass

    def test_3_generate_computation(self):
        pass

    def test_4_generate_schema(self):
        pass


