from typing import Optional

from ghostos_container import Container, Provider, INSTANCE
from ghostos.libraries.terminal.abcd import Terminal, TerminalContext
from ghostos.prompter import POM
from ghostos.contracts.logger import LoggerItf
import subprocess
import shlex

__all__ = ['TerminalExecutor', 'TerminalProvider', 'TerminalImpl']


class TerminalExecutor(Terminal):
    """
    generated by deepseek-reasoner
    """

    def __init__(self, logger: LoggerItf, safe_mode: bool = True):
        self.safe_mode = safe_mode  # 保持安全模式配置项
        self.logger = logger

    def _sanitize_command(self, command: str) -> str:
        """防御性命令处理策略"""
        if self.safe_mode:
            # 移除危险字符并限制命令长度
            cleaned = command.strip().replace(';', '').replace('&', '')[:500]
            return shlex.quote(cleaned)
        return command

    def exec(self, *commands: str, timeout: float = 10.0) -> Terminal.CommandResult:
        try:
            # 安全预处理流程
            executing = []
            for command in commands:
                # sanitized_cmd = self._sanitize_command(command)
                # args = shlex.split(sanitized_cmd)
                # executing.append(args)
                executing.append(command)
            full_command = ";".join(executing)
            self.logger.info(f'Executing command: {full_command}')
            # args = shlex.split(command)

            # 执行带超时控制的命令
            process = subprocess.Popen(
                # args,
                full_command,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                shell=True,
                text=True,
                encoding='utf-8',
                errors='replace',  # 错误字符替换策略
            )
            output, error = process.communicate(timeout=timeout)

            return self.CommandResult(
                exit_code=process.returncode,
                stdout=output,
                stderr=error,
            )

        except subprocess.TimeoutExpired:
            # 专用超时处理
            return self.CommandResult(
                exit_code=-1,
                stdout="",
                stderr=f"Command timed out after {timeout} seconds",
            )
        except Exception as e:
            # 通用异常处理
            return self.CommandResult(
                exit_code=-2,
                stdout="",
                stderr=f"Execution failed: {str(e)}",
            )


class TerminalImpl(POM, Terminal):

    def __init__(self, logger: LoggerItf, safe_mode: bool = True):
        self.logger = logger
        self._executor = TerminalExecutor(logger, safe_mode)
        self._context = TerminalContext()

    def exec(self, *commands: str, timeout: float = 10.0) -> Terminal.CommandResult:
        result = self._executor.exec(*commands, timeout=timeout)
        if result.exit_code != 0:
            raise RuntimeError(f"Command failed with exit code {result.exit_code}: {result.stderr}")
        return result

    def self_prompt(self, container: Container) -> str:
        context_prompt = self._context.generate_prompt()
        return f"""
basic information about the current terminal: 
```bash
{context_prompt}
```
"""

    def get_title(self) -> str:
        return "Terminal Context"


class TerminalProvider(Provider[Terminal]):

    def __init__(self, safe_mode: bool = True):
        self._safe_mode = safe_mode

    def singleton(self) -> bool:
        return False

    def factory(self, con: Container) -> Optional[INSTANCE]:
        logger = con.force_fetch(LoggerItf)
        return TerminalImpl(logger, safe_mode=self._safe_mode)


if __name__ == '__main__':
    from logging import getLogger

    t = TerminalImpl(logger=getLogger('ghostos'))
    r = t.exec('poetry add fs')
    print(r.stdout)
