import logging
from typing import List

from slither.core.declarations.function import Function
from slither.slithir.operations.lvalue import OperationWithLValue
from slither.slithir.utils.utils import is_valid_lvalue, is_valid_rvalue
from slither.slithir.variables import TupleVariable, ReferenceVariable
from slither.core.source_mapping.source_mapping import SourceMapping
from slither.core.variables.variable import Variable


logger = logging.getLogger("AssignmentOperationIR")


class Assignment(OperationWithLValue):
    def __init__(
        self, left_variable: Variable, right_variable: SourceMapping, variable_return_type
    ) -> None:
        assert is_valid_lvalue(left_variable)
        assert is_valid_rvalue(right_variable) or isinstance(
            right_variable, (Function, TupleVariable)
        )
        super().__init__()
        self._variables = [left_variable, right_variable]
        self._lvalue = left_variable
        self._rvalue = right_variable
        self._variable_return_type = variable_return_type

    @property
    def variables(self):
        return list(self._variables)

    @property
    def read(self) -> List[SourceMapping]:
        return [self.rvalue]

    @property
    def variable_return_type(self):
        return self._variable_return_type

    @property
    def rvalue(self) -> SourceMapping:
        return self._rvalue

    def __str__(self):
        if isinstance(self.lvalue, ReferenceVariable):
            points = self.lvalue.points_to
            while isinstance(points, ReferenceVariable):
                points = points.points_to
            return f"{self.lvalue} (->{points}) := {self.rvalue}({self.rvalue.type})"
        return f"{self.lvalue}({self.lvalue.type}) := {self.rvalue}({self.rvalue.type})"
