# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['billbeeapi',
 'billbeeapi.controllers',
 'billbeeapi.exceptions',
 'billbeeapi.http',
 'billbeeapi.http.auth',
 'billbeeapi.models',
 'billbeeapi.utilities']

package_data = \
{'': ['*']}

install_requires = \
['cachecontrol>=0.11.7,<1.0',
 'enum34>=1.1.6',
 'jsonpickle>=0.7.1,<1.0',
 'python-dateutil>=2.5.3,<3.0',
 'requests>=2.9.1,<3.0']

setup_kwargs = {
    'name': 'billbeeapi',
    'version': '1.1.0',
    'description': "API client for BillBee, fork of 'billbee-1'",
    'long_description': '# Getting Started with Billbee API\n\n## Getting Started\n\n### Introduction\n\nDocumentation of the Billbee REST API to connect a Billbee account to external aplications.\n\n#### Endpoint\n\nThe Billbee API endpoint base url is https://app.billbee.io/api/v1\n\n#### Activation\n\nYou have to enable the API in the settings of your Billbee account. In addition you need a Billbee API Key identifying the application you develop. To get an API key, send a mail to support@billbee.io and send us a short note about what you are building.\n\n#### Authorization & security\n\nBecause you can access private data with the Billbee API, every request has to be sent over https and must\n\n* Contain a valid API Key identifying the application/developer. It has to be sent as the HTTP header X-Billbee-Api-Key\n* Contain a valid user login with billbee username and api password in form of a basic auth HTTP header\n\n#### Throttling\n\nEach endpoint has a throttle of max 2 requests per second per combination of API Key and Billbee user.\n\nWhen you exceed these 2 calls, the API will return a HTTP 429 status code\n\n### Install the Package\n\nThe package is compatible with Python versions `2 >=2.7.9` and `3 >=3.4`.\n\nDownload the wheel file an using the following pip command\n```python\npip install billbee_api-1.1.0-py3-none-any.whl\n```\n\n#### package note\n\nThis is a fork from \nhttps://pypi.python.org/pypi/billbee-1\nwhich you can install from PyPi using the following pip command.\n```python\npip install billbee-api\n```\nBut this package does not support API_KEYs \n\n### Initialize the API Client\n\nThe following parameters are configurable for the API Client:\n\n| Parameter | Type | Description |\n|  --- | --- | --- |\n| `basic_auth_user_name` | `string` | The username to use with basic authentication |\n| `basic_auth_password` | `string` | The password to use with basic authentication |\n| `timeout` | `float` | The value to use for connection timeout. <br> **Default: 60** |\n| `max_retries` | `int` | The number of times to retry an endpoint call if it fails. <br> **Default: 3** |\n| `backoff_factor` | `float` | A backoff factor to apply between attempts after the second try. <br> **Default: 0** |\n\nThe API client can be initialized as follows:\n\n```python\nfrom billbeeapi.billbeeapi_client import BillbeeapiClient\nfrom billbeeapi.configuration import Environment\n\nclient = BillbeeapiClient(\n    basic_auth_user_name=\'BasicAuthUserName\',\n    basic_auth_password=\'BasicAuthPassword\',\n    api_key=\'ApiKey\'\n    environment = ,)\n```\n\n### Authorization\n\nThis API uses `Basic Authentication` and the API-KEY in the HTTP header `X-Billbee-Api-Key`\n\n## Client Class Documentation\n\n### Billbee API Client\n\nThe gateway for the SDK. This class acts as a factory for the Controllers and also holds the configuration of the SDK.\n\n### Controllers\n\n| Name | Description |\n|  --- | --- |\n| products | Gets ProductsController |\n| provisioning | Gets ProvisioningController |\n| cloud_storage | Gets CloudStorageController |\n| customers | Gets CustomersController |\n| customer_addresses | Gets CustomerAddressesController |\n| enum_api | Gets EnumApiController |\n| events | Gets EventsController |\n| orders | Gets OrdersController |\n| shipments | Gets ShipmentsController |\n| webhooks | Gets WebhooksController |\n\n## API Reference\n\n### List of APIs\n\n* [Products](#products)\n* [Provisioning](#provisioning)\n* [Cloud Storage](#cloud-storage)\n* [Customers](#customers)\n* [Customer Addresses](#customer-addresses)\n* [Enum Api](#enum-api)\n* [Events](#events)\n* [Orders](#orders)\n* [Shipments](#shipments)\n* [Webhooks](#webhooks)\n\n### Products\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `ProductsController` class can be accessed from the API Client.\n\n```\nproducts_controller = client.products\n```\n\n#### Article Update Stock\n\nThe article is specified by sku. You have to send the absolute value for the current stock\n\n```python\ndef article_update_stock(self,\n                        model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelUpdateStockApiModel`](#billbee-interfaces-billbee-api-model-update-stock-api-model) | Body, Required | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelUpdateStockResponseData`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-update-stock-response-data)\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelUpdateStockApiModel()\n\nresult = products_controller.article_update_stock(model)\n```\n\n#### Article Get Reserved Amount\n\nQueries the reserved amount for a single article by id or by sku\n\n```python\ndef article_get_reserved_amount(self,\n                               id,\n                               lookup_by=None,\n                               stock_id=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `string` | Query, Required | The id or the sku of the article to query |\n| `lookup_by` | `string` | Query, Optional | Either the value id or the value sku to specify the meaning of the id parameter |\n| `stock_id` | `long\\|int` | Query, Optional | Optional the stock id if the multi stock feature is enabled |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelGetReservedAmountResponseData`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-get-reserved-amount-response-data)\n\n##### Example Usage\n\n```python\nid = \'id0\'\n\nresult = products_controller.article_get_reserved_amount(id)\n```\n\n#### Article Update Stock Multiple\n\nUpdate the stock qty for multiple articles at once\n\n```python\ndef article_update_stock_multiple(self,\n                                 models)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `models` | [`List of BillbeeInterfacesBillbeeAPIModelUpdateStockApiModel`](#billbee-interfaces-billbee-api-model-update-stock-api-model) | Body, Required | - |\n\n##### Response Type\n\n[`List of RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelUpdateStockResponseData`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-update-stock-response-data)\n\n##### Example Usage\n\n```python\nmodels = []\n\nmodels.append(BillbeeInterfacesBillbeeAPIModelUpdateStockApiModel())\n\nmodels.append(BillbeeInterfacesBillbeeAPIModelUpdateStockApiModel())\n\n\nresult = products_controller.article_update_stock_multiple(models)\n```\n\n#### Article Update Stock Code\n\nUpdate the stock code of an article\n\n```python\ndef article_update_stock_code(self,\n                             model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelUpdateStockCodeApiModel`](#billbee-interfaces-billbee-api-model-update-stock-code-api-model) | Body, Required | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelUpdateStockCodeResponseData`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-update-stock-code-response-data)\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelUpdateStockCodeApiModel()\n\nresult = products_controller.article_update_stock_code(model)\n```\n\n#### Article Get Article\n\nQueries a single article by id or by sku\n\n```python\ndef article_get_article(self,\n                       id,\n                       lookup_by=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `string` | Template, Required | The id or the sku of the article to query |\n| `lookup_by` | `string` | Query, Optional | Either the value id, ean or the value sku to specify the meaning of the id parameter |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-api-model)\n\n##### Example Usage\n\n```python\nid = \'id0\'\n\nresult = products_controller.article_get_article(id)\n```\n\n#### Article Delete Article\n\nDeletes a product\n\n```python\ndef article_delete_article(self,\n                          id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the Product |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = products_controller.article_delete_article(id)\n```\n\n#### Article Patch Article\n\nUpdates one or more fields of a product\n\n```python\ndef article_patch_article(self,\n                         id,\n                         model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the Product |\n| `model` | `object` | Body, Required | - |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\nmodel = jsonpickle.decode(\'{"key1":"val1","key2":"val2"}\')\n\nresult = products_controller.article_patch_article(id, model)\n```\n\n#### Article Get List\n\nGet a list of all products\n\n```python\ndef article_get_list(self,\n                    page=None,\n                    page_size=None,\n                    min_created_at=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Query, Optional | The current page to request starting with 1 |\n| `page_size` | `int` | Query, Optional | The pagesize for the result list. Values between 1 and 250 are allowed |\n| `min_created_at` | `datetime` | Query, Optional | Optional the oldest create date of the articles to be returned |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleApiModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-api-model)\n\n##### Example Usage\n\n```python\nresult = products_controller.article_get_list()\n```\n\n#### Article Create Article\n\nCreates a new product\n\n```python\ndef article_create_article(self,\n                          model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelArticleApiModel`](#billbee-interfaces-billbee-api-model-article-api-model) | Body, Required | - |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelArticleApiModel()\nmodel.vat_index = 104\nmodel.price = 72.12\nmodel.vat_1_rate = 42.06\nmodel.vat_2_rate = 167.28\nmodel.mtype = 2\nmodel.is_digital = False\nmodel.is_customizable = False\n\nresult = products_controller.article_create_article(model)\n```\n\n#### Article Get Custom Fields\n\nQueries a list of all custom fields\n\n```python\ndef article_get_custom_fields(self,\n                             page=None,\n                             page_size=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Query, Optional | - |\n| `page_size` | `int` | Query, Optional | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model)\n\n##### Example Usage\n\n```python\nresult = products_controller.article_get_custom_fields()\n```\n\n#### Article Get Custom Field\n\nQueries a single custom field\n\n```python\ndef article_get_custom_field(self,\n                            id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the custom field to query |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model)\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = products_controller.article_get_custom_field(id)\n```\n\n#### Article Get Patchable Fields\n\nReturns a list of fields which can be updated with the patch call\n\n```python\ndef article_get_patchable_fields(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = products_controller.article_get_patchable_fields()\n```\n\n#### Article Get Category\n\nGEts a list of all defined categories\n\n```python\ndef article_get_category(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = products_controller.article_get_category()\n```\n\n#### Article Get Images\n\nReturns a list of all images of the product\n\n```python\ndef article_get_images(self,\n                      product_id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `product_id` | `long\\|int` | Template, Required | The Id of the product |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-image-relation-api-model)\n\n##### Example Usage\n\n```python\nproduct_id = 98\n\nresult = products_controller.article_get_images(product_id)\n```\n\n#### Article Put Images\n\nAdd multiple images to a product or replace the product images by the given images\n\n```python\ndef article_put_images(self,\n                      product_id,\n                      models,\n                      replace=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `product_id` | `long\\|int` | Template, Required | The id of the product |\n| `models` | [`List of BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#billbee-interfaces-billbee-api-model-article-image-relation-api-model) | Body, Required | An array of ArticleApiImageModel |\n| `replace` | `bool` | Query, Optional | If you pass true, the images will be replaced by the passed images. Otherwise the passed images will be appended to the product. |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-image-relation-api-model)\n\n##### Example Usage\n\n```python\nproduct_id = 98\nmodels = []\n\nmodels.append(BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel())\n\nmodels.append(BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel())\n\n\nresult = products_controller.article_put_images(product_id, models)\n```\n\n#### Article Get Image From Product\n\nReturns a single image by id\n\n```python\ndef article_get_image_from_product(self,\n                                  product_id,\n                                  image_id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `product_id` | `long\\|int` | Template, Required | The Id of the product |\n| `image_id` | `long\\|int` | Template, Required | The Id of the image |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-image-relation-api-model)\n\n##### Example Usage\n\n```python\nproduct_id = 98\nimage_id = 126\n\nresult = products_controller.article_get_image_from_product(product_id, image_id)\n```\n\n#### Article Put Image\n\nAdd or update an existing image of a product\n\n```python\ndef article_put_image(self,\n                     product_id,\n                     image_id,\n                     model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `product_id` | `long\\|int` | Template, Required | The product id |\n| `image_id` | `long\\|int` | Template, Required | The image id. If you pass 0, the image will be added |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#billbee-interfaces-billbee-api-model-article-image-relation-api-model) | Body, Required | The ArticleApiImageModel |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-image-relation-api-model)\n\n##### Example Usage\n\n```python\nproduct_id = 98\nimage_id = 126\nmodel = BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel()\n\nresult = products_controller.article_put_image(product_id, image_id, model)\n```\n\n#### Article Delete Image From Product\n\nDeletes a single image from a product\n\n```python\ndef article_delete_image_from_product(self,\n                                     product_id,\n                                     image_id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `product_id` | `long\\|int` | Template, Required | The product id |\n| `image_id` | `long\\|int` | Template, Required | The image id |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nproduct_id = 98\nimage_id = 126\n\nresult = products_controller.article_delete_image_from_product(product_id, image_id)\n```\n\n#### Article Get Image\n\nReturns a single image by id\n\n```python\ndef article_get_image(self,\n                     image_id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `image_id` | `long\\|int` | Template, Required | The Id of the image |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-image-relation-api-model)\n\n##### Example Usage\n\n```python\nimage_id = 126\n\nresult = products_controller.article_get_image(image_id)\n```\n\n#### Article Delete Image\n\nDeletes a single image by id\n\n```python\ndef article_delete_image(self,\n                        image_id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `image_id` | `long\\|int` | Template, Required | The image id |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nimage_id = 126\n\nresult = products_controller.article_delete_image(image_id)\n```\n\n#### Article Delete Images\n\nDelete multiple images by id\n\n```python\ndef article_delete_images(self,\n                         image_ids)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `image_ids` | `List of long\\|int` | Body, Required | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelDeletedImagesModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-deleted-images-model)\n\n##### Example Usage\n\n```python\nimage_ids = [87, 88, 89]\n\nresult = products_controller.article_delete_images(image_ids)\n```\n\n#### Search Search\n\nSearch for products, customers and orders.\nType can be "order", "product" and / or "customer"\nTerm can contains lucene query syntax\n\n```python\ndef search_search(self,\n                 model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`RechnungsdruckWebAppControllersApiSearchControllerSearchModel`](#rechnungsdruck-web-app-controllers-api-search-controller-search-model) | Body, Required | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultRechnungsdruckWebAppControllersApiSearchControllerSearchResultsModel`](#rechnungsdruck-web-app-controllers-api-api-result-rechnungsdruck-web-app-controllers-api-search-controller-search-results-model)\n\n##### Example Usage\n\n```python\nmodel = RechnungsdruckWebAppControllersApiSearchControllerSearchModel()\n\nresult = products_controller.search_search(model)\n```\n\n### Provisioning\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `ProvisioningController` class can be accessed from the API Client.\n\n```\nprovisioning_controller = client.provisioning\n```\n\n#### Automatic Provisioning Create Account\n\nCreates a new Billbee user account with the data passed\n\n```python\ndef automatic_provisioning_create_account(self,\n                                         model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`RechnungsdruckWebAppControllersApiAutomaticProvisioningControllerCreateAccountContainer`](#rechnungsdruck-web-app-controllers-api-automatic-provisioning-controller-create-account-container) | Body, Required | - |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nmodel = RechnungsdruckWebAppControllersApiAutomaticProvisioningControllerCreateAccountContainer()\nmodel.e_mail = \'EMail2\'\n\nresult = provisioning_controller.automatic_provisioning_create_account(model)\n```\n\n#### Automatic Provisioning Terms Info\n\nReturns infos about Billbee terms and conditions\n\n```python\ndef automatic_provisioning_terms_info(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = provisioning_controller.automatic_provisioning_terms_info()\n```\n\n### Cloud Storage\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `CloudStorageController` class can be accessed from the API Client.\n\n```\ncloud_storage_controller = client.cloud_storage\n```\n\n#### Cloud Storage Api Get List\n\nGets a list of all connected cloud storage devices\n\n```python\ndef cloud_storage_api_get_list(self)\n```\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCloudStorageApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-cloud-storage-api-model)\n\n##### Example Usage\n\n```python\nresult = cloud_storage_controller.cloud_storage_api_get_list()\n```\n\n### Customers\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `CustomersController` class can be accessed from the API Client.\n\n```\ncustomers_controller = client.customers\n```\n\n#### Customer Get All\n\nGet a list of all customers\n\n```python\ndef customer_get_all(self,\n                    page=None,\n                    page_size=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Query, Optional | The current page to request starting with 1 |\n| `page_size` | `int` | Query, Optional | The pagesize for the result list. Values between 1 and 250 are allowed |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-api-model)\n\n##### Example Usage\n\n```python\nresult = customers_controller.customer_get_all()\n```\n\n#### Customer Create\n\nCreates a new customer\n\n```python\ndef customer_create(self,\n                   model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelCreateCustomerApiModel`](#billbee-interfaces-billbee-api-model-create-customer-api-model) | Body, Required | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-api-model)\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelCreateCustomerApiModel()\n\nresult = customers_controller.customer_create(model)\n```\n\n#### Customer Get One\n\nQueries a single customer by id\n\n```python\ndef customer_get_one(self,\n                    id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the customer to query |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-api-model)\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = customers_controller.customer_get_one(id)\n```\n\n#### Customer Update\n\nUpdates a customer by id\n\n```python\ndef customer_update(self,\n                   model,\n                   id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#billbee-interfaces-billbee-api-model-customer-api-model) | Body, Required | - |\n| `id` | `long\\|int` | Template, Required | The id of the customer |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-api-model)\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelCustomerApiModel()\nid = 112\n\nresult = customers_controller.customer_update(model, id)\n```\n\n#### Customer Get Customer Orders\n\nQueries a list of orders from a customer\n\n```python\ndef customer_get_customer_orders(self,\n                                id,\n                                page=None,\n                                page_size=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the customer |\n| `page` | `int` | Query, Optional | The current page to request starting with 1 |\n| `page_size` | `int` | Query, Optional | The pagesize for the result list. Values between 1 and 250 are allowed |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListRechnungsdruckWebAppControllersApiOrder`](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-rechnungsdruck-web-app-controllers-api-order)\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = customers_controller.customer_get_customer_orders(id)\n```\n\n#### Customer Get Customer Addresses\n\nQueries a list of addresses from a customer\n\n```python\ndef customer_get_customer_addresses(self,\n                                   id,\n                                   page=None,\n                                   page_size=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the customer |\n| `page` | `int` | Query, Optional | The current page to request starting with 1 |\n| `page_size` | `int` | Query, Optional | The pagesize for the result list. Values between 1 and 250 are allowed |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = customers_controller.customer_get_customer_addresses(id)\n```\n\n#### Customer Add Customer Address\n\nId and  CustomerId will be ignored in model. If Id is set, the addition will be stopped.\n\n```python\ndef customer_add_customer_address(self,\n                                 id,\n                                 model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | CustomerId to attach the new address to. |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#billbee-interfaces-billbee-api-model-customer-address-api-model) | Body, Required | Model containing the address, that should be attached. |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nid = 112\nmodel = BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel()\n\nresult = customers_controller.customer_add_customer_address(id, model)\n```\n\n#### Customer Get Customer Address\n\nQueries a single address from a customer\n\n```python\ndef customer_get_customer_address(self,\n                                 id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the address |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = customers_controller.customer_get_customer_address(id)\n```\n\n#### Customer Update Address\n\nId and CustomerId cannot be changed. Fields you do not send will become empty.\n\n```python\ndef customer_update_address(self,\n                           model,\n                           id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#billbee-interfaces-billbee-api-model-customer-address-api-model) | Body, Required | The updated address. Please query an address via (todo) to see all fields. Note that Id and CustomerId cannot be changed. |\n| `id` | `long\\|int` | Template, Required | The id of the address |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel()\nid = 112\n\nresult = customers_controller.customer_update_address(model, id)\n```\n\n#### Customer Patch Address\n\nId and CustomerId cannot be changed\n\n```python\ndef customer_patch_address(self,\n                          id,\n                          model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the address |\n| `model` | `object` | Body, Required | The address fields to be changed. Please query an address via (todo) to see all fields. Note that Id and CustomerId cannot be changed. |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nid = 112\nmodel = jsonpickle.decode(\'{"key1":"val1","key2":"val2"}\')\n\nresult = customers_controller.customer_patch_address(id, model)\n```\n\n### Customer Addresses\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `CustomerAddressesController` class can be accessed from the API Client.\n\n```\ncustomer_addresses_controller = client.customer_addresses\n```\n\n#### Customer Addresses Get All\n\nGet a list of all customer addresses\n\n```python\ndef customer_addresses_get_all(self,\n                              page=None,\n                              page_size=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Query, Optional | The current page to request starting with 1 (default is 1) |\n| `page_size` | `int` | Query, Optional | The page size for the result list. Values between 1 and 250 are allowed. (default is 50) |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nresult = customer_addresses_controller.customer_addresses_get_all()\n```\n\n#### Customer Addresses Create\n\nCreates a new customer address\n\n```python\ndef customer_addresses_create(self,\n                             model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#billbee-interfaces-billbee-api-model-customer-address-api-model) | Body, Required | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel()\n\nresult = customer_addresses_controller.customer_addresses_create(model)\n```\n\n#### Customer Addresses Get One\n\nQueries a single customer address by id\n\n```python\ndef customer_addresses_get_one(self,\n                              id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the address to query |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = customer_addresses_controller.customer_addresses_get_one(id)\n```\n\n#### Customer Addresses Update\n\nUpdates a customer address by id\n\n```python\ndef customer_addresses_update(self,\n                             model,\n                             id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#billbee-interfaces-billbee-api-model-customer-address-api-model) | Body, Required | - |\n| `id` | `long\\|int` | Template, Required | The id of the address |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-address-api-model)\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel()\nid = 112\n\nresult = customer_addresses_controller.customer_addresses_update(model, id)\n```\n\n### Enum Api\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `EnumApiController` class can be accessed from the API Client.\n\n```\nenum_api_controller = client.enum_api\n```\n\n#### Enum Api Get Payment Types\n\nReturns a list with all defined paymenttypes\n\n```python\ndef enum_api_get_payment_types(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = enum_api_controller.enum_api_get_payment_types()\n```\n\n#### Enum Api Get Shipping Carriers\n\nReturns a list with all defined shippingcarriers\n\n```python\ndef enum_api_get_shipping_carriers(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = enum_api_controller.enum_api_get_shipping_carriers()\n```\n\n#### Enum Api Get Order States\n\nReturns a list with all defined orderstates\n\n```python\ndef enum_api_get_order_states(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = enum_api_controller.enum_api_get_order_states()\n```\n\n### Events\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `EventsController` class can be accessed from the API Client.\n\n```\nevents_controller = client.events\n```\n\n#### Event Api Get List\n\nGet a list of all events optionally filtered by date. This request is extra throttled to 2 calls per page per hour.\n\n```python\ndef event_api_get_list(self,\n                      min_date=None,\n                      max_date=None,\n                      page=None,\n                      page_size=None,\n                      type_id=None,\n                      order_id=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `min_date` | `datetime` | Query, Optional | Specifies the oldest date to include in the response |\n| `max_date` | `datetime` | Query, Optional | Specifies the newest date to include in the response |\n| `page` | `int` | Query, Optional | Specifies the page to request |\n| `page_size` | `int` | Query, Optional | Specifies the pagesize. Defaults to 50, max value is 250 |\n| `type_id` | `List of int` | Query, Optional | Filter for specific event types |\n| `order_id` | `int` | Query, Optional | Filter for specific order id |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = events_controller.event_api_get_list()\n```\n\n### Orders\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `OrdersController` class can be accessed from the API Client.\n\n```\norders_controller = client.orders\n```\n\n#### Layout Api Get List\n\n```python\ndef layout_api_get_list(self)\n```\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelsLayoutTemplate`](#rechnungsdruck-web-app-controllers-api-api-result-system-collections-generic-list-billbee-interfaces-billbee-api-models-layout-template)\n\n##### Example Usage\n\n```python\nresult = orders_controller.layout_api_get_list()\n```\n\n#### Order Api Get List\n\nGet a list of all orders optionally filtered by date\n\n```python\ndef order_api_get_list(self,\n                      min_order_date=None,\n                      max_order_date=None,\n                      page=None,\n                      page_size=None,\n                      shop_id=None,\n                      order_state_id=None,\n                      tag=None,\n                      minimum_bill_bee_order_id=None,\n                      modified_at_min=None,\n                      modified_at_max=None,\n                      article_title_source=None,\n                      exclude_tags=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `min_order_date` | `datetime` | Query, Optional | Specifies the oldest order date to include in the response |\n| `max_order_date` | `datetime` | Query, Optional | Specifies the newest order date to include in the response |\n| `page` | `int` | Query, Optional | Specifies the page to request |\n| `page_size` | `int` | Query, Optional | Specifies the pagesize. Defaults to 50, max value is 250 |\n| `shop_id` | `List of long\\|int` | Query, Optional | Specifies a list of shop ids for which invoices should be included |\n| `order_state_id` | `List of int` | Query, Optional | Specifies a list of state ids to include in the response |\n| `tag` | `List of string` | Query, Optional | Specifies a list of tags the order must have attached to be included in the response |\n| `minimum_bill_bee_order_id` | `long\\|int` | Query, Optional | If given, all delivered orders have an Id greater than or equal to the given minimumOrderId |\n| `modified_at_min` | `datetime` | Query, Optional | If given, the last modification has to be newer than the given date |\n| `modified_at_max` | `datetime` | Query, Optional | If given, the last modification has to be older or equal than the given date. |\n| `article_title_source` | [`ArticleTitleSourceEnum`](#article-title-source) | Query, Optional | The source field for the article title. 0 = Order Position (default), 1 = Article Title, 2 = Article Invoice Text |\n| `exclude_tags` | `bool` | Query, Optional | If true the list of tags passed to the call are used to filter orders to not include these tags |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelOrder`](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-order)\n\n##### Example Usage\n\n```python\nresult = orders_controller.order_api_get_list()\n```\n\n#### Order Api Post New Order\n\nTo create an order POST an JSON object to the orders endpoint with the shown properties.\nNot all properties are required.\n\n```python\ndef order_api_post_new_order(self,\n                            order_data,\n                            shop_id=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `order_data` | [`BillbeeInterfacesBillbeeAPIModelOrder`](#billbee-interfaces-billbee-api-model-order) | Body, Required | - |\n| `shop_id` | `long\\|int` | Query, Optional | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelOrder`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-order)\n\n##### Example Usage\n\n```python\norder_data = BillbeeInterfacesBillbeeAPIModelOrder()\n\nresult = orders_controller.order_api_post_new_order(order_data)\n```\n\n##### Errors\n\n| HTTP Status Code | Error Description | Exception Class |\n|  --- | --- | --- |\n| 400 | Invalid data was received in the request | `APIException` |\n| 500 | An internal exception occured. Order was not created | `APIException` |\n\n#### Order Api Tags Update\n\nUpdates/Sets the tags attached to an order\n\n```python\ndef order_api_tags_update(self,\n                         tag_data,\n                         id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `tag_data` | [`RechnungsdruckWebAppControllersApiOrderTagCreate`](#rechnungsdruck-web-app-controllers-api-order-tag-create) | Body, Required | Tags to attach |\n| `id` | `long\\|int` | Template, Required | The internal id of the order |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\ntag_data = RechnungsdruckWebAppControllersApiOrderTagCreate()\nid = 112\n\nresult = orders_controller.order_api_tags_update(tag_data, id)\n```\n\n#### Order Api Tags Create\n\nWhen a tag is already attached, it is ignored. Tags are not case sensitive.\n\n```python\ndef order_api_tags_create(self,\n                         tag_data,\n                         id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `tag_data` | [`RechnungsdruckWebAppControllersApiOrderTagCreate`](#rechnungsdruck-web-app-controllers-api-order-tag-create) | Body, Required | Tags to attach |\n| `id` | `long\\|int` | Template, Required | The internal id of the order |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\ntag_data = RechnungsdruckWebAppControllersApiOrderTagCreate()\nid = 112\n\nresult = orders_controller.order_api_tags_create(tag_data, id)\n```\n\n#### Order Api Get\n\nGet a single order by its internal billbee id. This request is throttled to 6 calls per order in one minute\n\n```python\ndef order_api_get(self,\n                 id,\n                 article_title_source=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The internal billbee id of the order |\n| `article_title_source` | [`ArticleTitleSourceEnum`](#article-title-source) | Query, Optional | The source field for the article title. 0 = Order Position (default), 1 = Article Title, 2 = Article Invoice Text |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelOrder`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-order)\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = orders_controller.order_api_get(id)\n```\n\n#### Order Api Patch Order\n\nUpdates one or more fields of an order\n\n```python\ndef order_api_patch_order(self,\n                         id,\n                         model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | - |\n| `model` | `object` | Body, Required | - |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelOrder`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-order)\n\n##### Example Usage\n\n```python\nid = 112\nmodel = jsonpickle.decode(\'{"key1":"val1","key2":"val2"}\')\n\nresult = orders_controller.order_api_patch_order(id, model)\n```\n\n#### Order Api Get by Ext Ref\n\nGet a single order by its external order number\n\n```python\ndef order_api_get_by_ext_ref(self,\n                            ext_ref)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `ext_ref` | `string` | Template, Required | The extern order number of the order |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelOrder`](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-order)\n\n##### Example Usage\n\n```python\next_ref = \'extRef8\'\n\nresult = orders_controller.order_api_get_by_ext_ref(ext_ref)\n```\n\n#### Order Api Update State\n\n### REMARKS\n\nUse this call to change the state of an order to i.e. paid or sent.\n\nThe state is transfered to the external shop/marketplace if configured.\nThis is the list of known states:\n\n- 1: ordered\n- 2: confirmed\n- 3: paid\n- 4: shipped\n- 5: reclamation\n- 6: deleted\n- 7: closed\n- 8: canceled\n- 9: archived\n- 10: not used\n- 11: demand note 1\n- 12: demand note 2\n- 13: packed\n- 14: offered\n- 15: payment reminder\n- 16: fulfilling\n\n```python\ndef order_api_update_state(self,\n                          id,\n                          model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The internal id of the order |\n| `model` | [`RechnungsdruckWebAppControllersApiOrderStateUpdate`](#rechnungsdruck-web-app-controllers-api-order-state-update) | Body, Required | The data used to change the state |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\nmodel = RechnungsdruckWebAppControllersApiOrderStateUpdate()\n\nresult = orders_controller.order_api_update_state(id, model)\n```\n\n#### Order Api Add Shipment\n\nAdd a shipment to a given order\n\n```python\ndef order_api_add_shipment(self,\n                          id,\n                          model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The internal billbee id of the order |\n| `model` | [`RechnungsdruckWebAppControllersApiApiAddShipmentToOrderModel`](#rechnungsdruck-web-app-controllers-api-api-add-shipment-to-order-model) | Body, Required | The shipment data to create the shipment |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\nmodel = RechnungsdruckWebAppControllersApiApiAddShipmentToOrderModel()\n\nresult = orders_controller.order_api_add_shipment(id, model)\n```\n\n#### Order Api Get Invoice List\n\nGet a list of all invoices optionally filtered by date. This request ist throttled to 1 per 1 minute for same page and minInvoiceDate\n\n```python\ndef order_api_get_invoice_list(self,\n                              min_invoice_date=None,\n                              max_invoice_date=None,\n                              page=None,\n                              page_size=None,\n                              shop_id=None,\n                              order_state_id=None,\n                              tag=None,\n                              min_pay_date=None,\n                              max_pay_date=None,\n                              include_positions=None,\n                              exclude_tags=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `min_invoice_date` | `datetime` | Query, Optional | Specifies the oldest invoice date to include |\n| `max_invoice_date` | `datetime` | Query, Optional | Specifies the newest invoice date to include |\n| `page` | `int` | Query, Optional | Specifies the page to request |\n| `page_size` | `int` | Query, Optional | Specifies the pagesize. Defaults to 50, max value is 250 |\n| `shop_id` | `List of long\\|int` | Query, Optional | Specifies a list of shop ids for which invoices should be included |\n| `order_state_id` | `List of int` | Query, Optional | Specifies a list of state ids to include in the response |\n| `tag` | `List of string` | Query, Optional | - |\n| `min_pay_date` | `datetime` | Query, Optional | - |\n| `max_pay_date` | `datetime` | Query, Optional | - |\n| `include_positions` | `bool` | Query, Optional | - |\n| `exclude_tags` | `bool` | Query, Optional | If true the list of tags passed to the call are used to filter orders to not include these tags |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = orders_controller.order_api_get_invoice_list()\n```\n\n#### Order Api Find\n\nFind a single order by its external id (order number)\n\n```python\ndef order_api_find(self,\n                  id,\n                  partner)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `string` | Template, Required | The order id from the external system |\n| `partner` | `string` | Template, Required | Optional the name of the shop/marketplace the order was imported from |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = \'id0\'\npartner = \'partner0\'\n\nresult = orders_controller.order_api_find(id, partner)\n```\n\n#### Order Api Create Delivery Note\n\nCreate an delivery note for an existing order. This request is extra throttled by order and api key to a maximum of 1 per 5 minutes.\n\n```python\ndef order_api_create_delivery_note(self,\n                                  id,\n                                  include_pdf=None,\n                                  send_to_cloud_id=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The internal billbee id of the order |\n| `include_pdf` | `bool` | Query, Optional | If true, the PDF is included in the response as base64 encoded string |\n| `send_to_cloud_id` | `long\\|int` | Query, Optional | Optionally specify the id of a billbee connected cloud device to send the pdf to |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = orders_controller.order_api_create_delivery_note(id)\n```\n\n#### Order Api Create Invoice\n\nCreate an invoice for an existing order. This request is extra throttled by order and api key to a maximum of 1 per 5 minutes.\n\n```python\ndef order_api_create_invoice(self,\n                            id,\n                            include_invoice_pdf=None,\n                            template_id=None,\n                            send_to_cloud_id=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The internal billbee id of the order |\n| `include_invoice_pdf` | `bool` | Query, Optional | If true, the PDF is included in the response as base64 encoded string |\n| `template_id` | `long\\|int` | Query, Optional | You can pass the id of an invoice template to overwrite the assigned template for invoice creation |\n| `send_to_cloud_id` | `long\\|int` | Query, Optional | You can pass the id of a connected cloud printer/storage to send the invoice to it |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\n\nresult = orders_controller.order_api_create_invoice(id)\n```\n\n#### Order Api Get Patchable Fields\n\nReturns a list of fields which can be updated with the orders/{id} patch call\n\n```python\ndef order_api_get_patchable_fields(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = orders_controller.order_api_get_patchable_fields()\n```\n\n#### Order Api Send Message\n\nSends a message to the buyer\n\n```python\ndef order_api_send_message(self,\n                          id,\n                          model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the order |\n| `model` | [`RechnungsdruckWebAppControllersApiOrderApiControllerSendMessageModel`](#rechnungsdruck-web-app-controllers-api-order-api-controller-send-message-model) | Body, Required | The message model |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\nmodel = RechnungsdruckWebAppControllersApiOrderApiControllerSendMessageModel()\n\nresult = orders_controller.order_api_send_message(id, model)\n```\n\n#### Order Api Trigger Event\n\nTriggers a rule event\n\n```python\ndef order_api_trigger_event(self,\n                           id,\n                           model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the order |\n| `model` | [`RechnungsdruckWebAppControllersApiOrderApiControllerTriggerEventContainer`](#rechnungsdruck-web-app-controllers-api-order-api-controller-trigger-event-container) | Body, Required | - |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\nmodel = RechnungsdruckWebAppControllersApiOrderApiControllerTriggerEventContainer()\n\nresult = orders_controller.order_api_trigger_event(id, model)\n```\n\n#### Order Api Parse Placeholders\n\nParses a text and replaces all placeholders\n\n```python\ndef order_api_parse_placeholders(self,\n                                id,\n                                container)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Template, Required | The id of the order |\n| `container` | [`RechnungsdruckWebAppControllersApiOrderApiControllerParseTextContainer`](#rechnungsdruck-web-app-controllers-api-order-api-controller-parse-text-container) | Body, Required | - |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = 112\ncontainer = RechnungsdruckWebAppControllersApiOrderApiControllerParseTextContainer()\n\nresult = orders_controller.order_api_parse_placeholders(id, container)\n```\n\n### Shipments\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `ShipmentsController` class can be accessed from the API Client.\n\n```\nshipments_controller = client.shipments\n```\n\n#### Shipment Post Shipment\n\nCreates a new shipment with the selected Shippingprovider\n\n```python\ndef shipment_post_shipment(self,\n                          model)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `model` | [`BillbeeInterfacesBillbeeAPIModelCreateShipmentApiModel`](#billbee-interfaces-billbee-api-model-create-shipment-api-model) | Body, Required | Data to specify shipment parameters |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nmodel = BillbeeInterfacesBillbeeAPIModelCreateShipmentApiModel()\n\nresult = shipments_controller.shipment_post_shipment(model)\n```\n\n#### Shipment Get Shippingproviders\n\nQuery all defined shipping providers\n\n```python\ndef shipment_get_shippingproviders(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = shipments_controller.shipment_get_shippingproviders()\n```\n\n#### Shipment Ship With Label\n\nCreates a shipment for an order in billbee\n\n```python\ndef shipment_ship_with_label(self,\n                            shipment_information)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `shipment_information` | [`RechnungsdruckWebAppControllersApiShipmentWithLabel`](#rechnungsdruck-web-app-controllers-api-shipment-with-label) | Body, Required | Details on the order and the shipping methods, that should be used. |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiResultRechnungsdruckWebAppControllersApiShipmentWithLabelResult`](#rechnungsdruck-web-app-controllers-api-api-result-rechnungsdruck-web-app-controllers-api-shipment-with-label-result)\n\n##### Example Usage\n\n```python\nshipment_information = RechnungsdruckWebAppControllersApiShipmentWithLabel()\n\nresult = shipments_controller.shipment_ship_with_label(shipment_information)\n```\n\n#### Shipment Get Shipping Carrier\n\nQueries the currently available shipping carriers.\n\n```python\ndef shipment_get_shipping_carrier(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = shipments_controller.shipment_get_shipping_carrier()\n```\n\n#### Shipment Get Ping\n\n```python\ndef shipment_get_ping(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = shipments_controller.shipment_get_ping()\n```\n\n#### Shipment Get List\n\nGet a list of all shipments optionally filtered by date. All parameters are optional.\n\n```python\ndef shipment_get_list(self,\n                     page=None,\n                     page_size=None,\n                     created_at_min=None,\n                     created_at_max=None,\n                     order_id=None,\n                     minimum_shipment_id=None,\n                     shipping_provider_id=None)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Query, Optional | Specifies the page to request. |\n| `page_size` | `int` | Query, Optional | Specifies the pagesize. Defaults to 50, max value is 250 |\n| `created_at_min` | `datetime` | Query, Optional | Specifies the oldest shipment date to include in the response |\n| `created_at_max` | `datetime` | Query, Optional | Specifies the newest shipment date to include in the response |\n| `order_id` | `long\\|int` | Query, Optional | Get shipments for this order only. |\n| `minimum_shipment_id` | `long\\|int` | Query, Optional | Get Shipments with a shipment greater or equal than this id. New shipments have a greater id than older shipments. |\n| `shipping_provider_id` | `long\\|int` | Query, Optional | Get Shippings for the specified shipping provider only. <seealso cref="M:Rechnungsdruck.WebApp.Controllers.Api.ShipmentController.GetShippingproviders" /> |\n\n##### Response Type\n\n[`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelShipment`](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-shipment)\n\n##### Example Usage\n\n```python\nresult = shipments_controller.shipment_get_list()\n```\n\n### Webhooks\n\n#### Overview\n\n##### Get instance\n\nAn instance of the `WebhooksController` class can be accessed from the API Client.\n\n```\nwebhooks_controller = client.webhooks\n```\n\n#### Web Hook Management Get\n\nGets all registered WebHooks for a given user.\n\n```python\ndef web_hook_management_get(self)\n```\n\n##### Response Type\n\n[`List of MicrosoftAspNetWebHooksWebHook`](#microsoft-asp-net-web-hooks-web-hook)\n\n##### Example Usage\n\n```python\nresult = webhooks_controller.web_hook_management_get()\n```\n\n#### Web Hook Management Post\n\nRegisters a new WebHook for a given user.\n\n```python\ndef web_hook_management_post(self,\n                            web_hook)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `web_hook` | [`MicrosoftAspNetWebHooksWebHook`](#microsoft-asp-net-web-hooks-web-hook) | Body, Required | The webhook to create. Attach ?noecho to the uri to prevent echo test. |\n\n##### Response Type\n\n[`MicrosoftAspNetWebHooksWebHook`](#microsoft-asp-net-web-hooks-web-hook)\n\n##### Example Usage\n\n```python\nweb_hook = MicrosoftAspNetWebHooksWebHook()\nweb_hook.web_hook_uri = \'WebHookUri2\'\n\nresult = webhooks_controller.web_hook_management_post(web_hook)\n```\n\n#### Web Hook Management Delete All\n\nDeletes all existing WebHook registrations.\n\n```python\ndef web_hook_management_delete_all(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = webhooks_controller.web_hook_management_delete_all()\n```\n\n#### Web Hook Management Lookup\n\nLooks up a registered WebHook with the given {id} for a given user.\n\n```python\ndef web_hook_management_lookup(self,\n                              id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `string` | Template, Required | - |\n\n##### Response Type\n\n[`MicrosoftAspNetWebHooksWebHook`](#microsoft-asp-net-web-hooks-web-hook)\n\n##### Example Usage\n\n```python\nid = \'id0\'\n\nresult = webhooks_controller.web_hook_management_lookup(id)\n```\n\n#### Web Hook Management Put\n\nUpdates an existing WebHook registration.\n\n```python\ndef web_hook_management_put(self,\n                           id,\n                           web_hook)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `string` | Template, Required | The WebHook ID. |\n| `web_hook` | [`MicrosoftAspNetWebHooksWebHook`](#microsoft-asp-net-web-hooks-web-hook) | Body, Required | The new webhook to use. |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = \'id0\'\nweb_hook = MicrosoftAspNetWebHooksWebHook()\nweb_hook.web_hook_uri = \'WebHookUri2\'\n\nresult = webhooks_controller.web_hook_management_put(id, web_hook)\n```\n\n#### Web Hook Management Delete\n\nDeletes an existing WebHook registration.\n\n```python\ndef web_hook_management_delete(self,\n                              id)\n```\n\n##### Parameters\n\n| Parameter | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `string` | Template, Required | The WebHook ID. |\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nid = \'id0\'\n\nresult = webhooks_controller.web_hook_management_delete(id)\n```\n\n#### Web Hook Management Get Filters\n\nReturns a list of all known filters you can use to register webhooks\n\n```python\ndef web_hook_management_get_filters(self)\n```\n\n##### Response Type\n\n`object`\n\n##### Example Usage\n\n```python\nresult = webhooks_controller.web_hook_management_get_filters()\n```\n\n## Model Reference\n\n### Structures\n\n* [Billbee Interfaces Billbee API Model Update Stock Api Model](#billbee-interfaces-billbee-api-model-update-stock-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Update Stock Response Data](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-update-stock-response-data)\n* [Billbee Interfaces Billbee API Model Update Stock Response Data](#billbee-interfaces-billbee-api-model-update-stock-response-data)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Get Reserved Amount Response Data](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-get-reserved-amount-response-data)\n* [Billbee Interfaces Billbee API Model Get Reserved Amount Response Data](#billbee-interfaces-billbee-api-model-get-reserved-amount-response-data)\n* [Billbee Interfaces Billbee API Model Update Stock Code Api Model](#billbee-interfaces-billbee-api-model-update-stock-code-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Update Stock Code Response Data](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-update-stock-code-response-data)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Article Api Model](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-api-model)\n* [Billbee Interfaces Billbee API Model Article Api Model](#billbee-interfaces-billbee-api-model-article-api-model)\n* [Billbee Interfaces Order Multi Language String](#billbee-interfaces-order-multi-language-string)\n* [Billbee Interfaces Billbee API Model Article Image Relation Api Model](#billbee-interfaces-billbee-api-model-article-image-relation-api-model)\n* [Billbee Interfaces Billbee API Model Article Source Api Model](#billbee-interfaces-billbee-api-model-article-source-api-model)\n* [Billbee Interfaces Billbee API Model Stock Article Api Model](#billbee-interfaces-billbee-api-model-stock-article-api-model)\n* [Billbee Interfaces Billbee API Model Article Category Api Model](#billbee-interfaces-billbee-api-model-article-category-api-model)\n* [Billbee Interfaces Billbee API Model Bom Sub Article Api Model](#billbee-interfaces-billbee-api-model-bom-sub-article-api-model)\n* [Billbee Interfaces Billbee API Model Article Api Custom Field Value Model](#billbee-interfaces-billbee-api-model-article-api-custom-field-value-model)\n* [Billbee Interfaces Billbee API Model Article Api Custom Field Definition Model](#billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Article Api Model](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Article Api Model](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Article Api Custom Field Definition Model](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Article Api Custom Field Definition Model](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Article Api Custom Field Definition Model](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model)\n* [Rechnungsdruck Web App Controllers Api Api Result System Collections Generic List Billbee Interfaces Billbee API Model Article Image Relation Api Model](#rechnungsdruck-web-app-controllers-api-api-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-image-relation-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Article Image Relation Api Model](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-article-image-relation-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Deleted Images Model](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-deleted-images-model)\n* [Billbee Interfaces Billbee API Model Deleted Images Model](#billbee-interfaces-billbee-api-model-deleted-images-model)\n* [Rechnungsdruck Web App Controllers Api Automatic Provisioning Controller Create Account Container](#rechnungsdruck-web-app-controllers-api-automatic-provisioning-controller-create-account-container)\n* [Rechnungsdruck Web App Controllers Api Automatic Provisioning Controller Create Account Container User Address](#rechnungsdruck-web-app-controllers-api-automatic-provisioning-controller-create-account-container-user-address)\n* [Rechnungsdruck Web App Controllers Api Api Result System Collections Generic List Billbee Interfaces Billbee API Model Cloud Storage Api Model](#rechnungsdruck-web-app-controllers-api-api-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-cloud-storage-api-model)\n* [Billbee Interfaces Billbee API Model Cloud Storage Api Model](#billbee-interfaces-billbee-api-model-cloud-storage-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Customer Api Model](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Customer Api Model](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-api-model)\n* [Billbee Interfaces Billbee API Model Customer Api Model](#billbee-interfaces-billbee-api-model-customer-api-model)\n* [Billbee Interfaces Billbee API Model Create Customer Api Model](#billbee-interfaces-billbee-api-model-create-customer-api-model)\n* [Billbee Interfaces Billbee API Model Customer Address Api Model](#billbee-interfaces-billbee-api-model-customer-address-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Customer Api Model](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Rechnungsdruck Web App Controllers Api Order](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-rechnungsdruck-web-app-controllers-api-order)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Rechnungsdruck Web App Controllers Api Order](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-rechnungsdruck-web-app-controllers-api-order)\n* [Rechnungsdruck Web App Controllers Api Order](#rechnungsdruck-web-app-controllers-api-order)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Customer Address Api Model](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-address-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Customer Address Api Model](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-address-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Customer Address Api Model](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-customer-address-api-model)\n* [Rechnungsdruck Web App Controllers Api Api Result System Collections Generic List Billbee Interfaces Billbee API Models Layout Template](#rechnungsdruck-web-app-controllers-api-api-result-system-collections-generic-list-billbee-interfaces-billbee-api-models-layout-template)\n* [Billbee Interfaces Billbee API Models Layout Template](#billbee-interfaces-billbee-api-models-layout-template)\n* [Billbee Interfaces Billbee API Model Order](#billbee-interfaces-billbee-api-model-order)\n* [Billbee Interfaces Billbee API Model Shipment](#billbee-interfaces-billbee-api-model-shipment)\n* [Billbee Interfaces Billbee API Model Comment Api Model](#billbee-interfaces-billbee-api-model-comment-api-model)\n* [Billbee Interfaces Billbee API Model Order Address Api Model](#billbee-interfaces-billbee-api-model-order-address-api-model)\n* [Billbee Interfaces Billbee API Model Order Item](#billbee-interfaces-billbee-api-model-order-item)\n* [Billbee Interfaces Billbee API Model Order User](#billbee-interfaces-billbee-api-model-order-user)\n* [Billbee Interfaces Shipping Product Service](#billbee-interfaces-shipping-product-service)\n* [Billbee Interfaces Order History Entry](#billbee-interfaces-order-history-entry)\n* [Billbee Interfaces Billbee API Models Order Payment](#billbee-interfaces-billbee-api-models-order-payment)\n* [Billbee Interfaces Billbee API Model Sold Product](#billbee-interfaces-billbee-api-model-sold-product)\n* [Billbee Interfaces Billbee API Model Order Item Attribute](#billbee-interfaces-billbee-api-model-order-item-attribute)\n* [System Collections Generic Key Value Pair System String System Collections Generic List System Collections Generic Key Value Pair System Int 32 System String](#system-collections-generic-key-value-pair-system-string-system-collections-generic-list-system-collections-generic-key-value-pair-system-int-32-system-string)\n* [Billbee Interfaces Billbee API Model Product Image](#billbee-interfaces-billbee-api-model-product-image)\n* [System Collections Generic Key Value Pair System Int 32 System String](#system-collections-generic-key-value-pair-system-int-32-system-string)\n* [Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Order](#rechnungsdruck-web-app-controllers-api-api-result-billbee-interfaces-billbee-api-model-order)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Order](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-order)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Order](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-order)\n* [Rechnungsdruck Web App Controllers Api Order Tag Create](#rechnungsdruck-web-app-controllers-api-order-tag-create)\n* [Rechnungsdruck Web App Controllers Api Order State Update](#rechnungsdruck-web-app-controllers-api-order-state-update)\n* [Rechnungsdruck Web App Controllers Api Api Add Shipment to Order Model](#rechnungsdruck-web-app-controllers-api-api-add-shipment-to-order-model)\n* [Rechnungsdruck Web App Controllers Api Order Api Controller Send Message Model](#rechnungsdruck-web-app-controllers-api-order-api-controller-send-message-model)\n* [Rechnungsdruck Web App Controllers Api Order Api Controller Trigger Event Container](#rechnungsdruck-web-app-controllers-api-order-api-controller-trigger-event-container)\n* [Rechnungsdruck Web App Controllers Api Order Api Controller Parse Text Container](#rechnungsdruck-web-app-controllers-api-order-api-controller-parse-text-container)\n* [Rechnungsdruck Web App Controllers Api Search Controller Search Model](#rechnungsdruck-web-app-controllers-api-search-controller-search-model)\n* [Rechnungsdruck Web App Controllers Api Api Result Rechnungsdruck Web App Controllers Api Search Controller Search Results Model](#rechnungsdruck-web-app-controllers-api-api-result-rechnungsdruck-web-app-controllers-api-search-controller-search-results-model)\n* [Rechnungsdruck Web App Controllers Api Search Controller Search Results Model](#rechnungsdruck-web-app-controllers-api-search-controller-search-results-model)\n* [Rechnungsdruck Web App Controllers Api Search Controller Product Result](#rechnungsdruck-web-app-controllers-api-search-controller-product-result)\n* [Rechnungsdruck Web App Controllers Api Search Controller Order Result](#rechnungsdruck-web-app-controllers-api-search-controller-order-result)\n* [Rechnungsdruck Web App Controllers Api Search Controller Customer Result](#rechnungsdruck-web-app-controllers-api-search-controller-customer-result)\n* [Billbee Interfaces Billbee API Model Create Shipment Api Model](#billbee-interfaces-billbee-api-model-create-shipment-api-model)\n* [Billbee Interfaces Billbee API Model Shipment Address Api Model](#billbee-interfaces-billbee-api-model-shipment-address-api-model)\n* [Billbee Interfaces Shipping Shipment Data Dimensions](#billbee-interfaces-shipping-shipment-data-dimensions)\n* [Rechnungsdruck Web App Controllers Api Shipment With Label](#rechnungsdruck-web-app-controllers-api-shipment-with-label)\n* [Rechnungsdruck Web App Controllers Api Api Result Rechnungsdruck Web App Controllers Api Shipment With Label Result](#rechnungsdruck-web-app-controllers-api-api-result-rechnungsdruck-web-app-controllers-api-shipment-with-label-result)\n* [Rechnungsdruck Web App Controllers Api Shipment With Label Result](#rechnungsdruck-web-app-controllers-api-shipment-with-label-result)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Shipment](#rechnungsdruck-web-app-controllers-api-api-paged-result-system-collections-generic-list-billbee-interfaces-billbee-api-model-shipment)\n* [Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Shipment](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-shipment)\n* [Microsoft Asp Net Web Hooks Web Hook](#microsoft-asp-net-web-hooks-web-hook)\n\n#### Billbee Interfaces Billbee API Model Update Stock Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelUpdateStockApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `billbee_id` | `long\\|int` | Optional | Optional the ID of the Billbee product to update |\n| `sku` | `string` | Optional | The SKU of the product to update |\n| `stock_id` | `long\\|int` | Optional | Optional the stock id if the feature multi stock is activated |\n| `reason` | `string` | Optional | Optional a reason text for the stock update |\n| `old_quantity` | `float` | Optional | This parameter is currently ignored |\n| `new_quantity` | `float` | Optional | The new absolute stock quantity for the product you want to set |\n| `delta_quantity` | `float` | Optional | This parameter is currently ignored |\n| `force_send_stock_to_shops` | `bool` | Optional | If true, every sent stockchange is stored and transmitted to the connected shop, even if the value has not changed |\n| `autosubtract_reserved_amount` | `bool` | Optional | Automatically reduce the NewQuantity by the currently not fulfilled amount |\n\n##### Example (as JSON)\n\n```json\n{\n  "BillbeeId": null,\n  "Sku": null,\n  "StockId": null,\n  "Reason": null,\n  "OldQuantity": null,\n  "NewQuantity": null,\n  "DeltaQuantity": null,\n  "ForceSendStockToShops": null,\n  "AutosubtractReservedAmount": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Update Stock Response Data\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelUpdateStockResponseData`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelUpdateStockResponseData`](#billbee-interfaces-billbee-api-model-update-stock-response-data) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Update Stock Response Data\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelUpdateStockResponseData`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `sku` | `string` | Optional | The SKU of the article to update the current stock |\n| `current_stock` | `float` | Optional | The new value for current stock after the update |\n| `unfulfilled_amount` | `float` | Optional | The value of the unfulfilled amount (reserved) of the article |\n| `message` | `string` | Optional | A human readable message that explains the result of the operation |\n\n##### Example (as JSON)\n\n```json\n{\n  "SKU": null,\n  "CurrentStock": null,\n  "UnfulfilledAmount": null,\n  "Message": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Get Reserved Amount Response Data\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelGetReservedAmountResponseData`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelGetReservedAmountResponseData`](#billbee-interfaces-billbee-api-model-get-reserved-amount-response-data) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Get Reserved Amount Response Data\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelGetReservedAmountResponseData`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `reserved_amount` | `float` | Optional | The reserve (not fulfilled) qty of the article |\n\n##### Example (as JSON)\n\n```json\n{\n  "ReservedAmount": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Update Stock Code Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelUpdateStockCodeApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `billbee_id` | `long\\|int` | Optional | - |\n| `sku` | `string` | Optional | - |\n| `stock_id` | `long\\|int` | Optional | - |\n| `stock_code` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "BillbeeId": null,\n  "Sku": null,\n  "StockId": null,\n  "StockCode": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Update Stock Code Response Data\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelUpdateStockCodeResponseData`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | `object` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Article Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelArticleApiModel`](#billbee-interfaces-billbee-api-model-article-api-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Article Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelArticleApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `invoice_text` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `manufacturer` | `string` | Optional | - |\n| `id` | `long\\|int` | Optional | - |\n| `title` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `description` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `short_description` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `basic_attributes` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `images` | [`List of BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#billbee-interfaces-billbee-api-model-article-image-relation-api-model) | Optional | - |\n| `vat_index` | `int` | Required | - |\n| `price` | `float` | Required | - |\n| `cost_price` | `float` | Optional | - |\n| `vat_1_rate` | `float` | Required | - |\n| `vat_2_rate` | `float` | Required | - |\n| `stock_desired` | `float` | Optional | - |\n| `stock_current` | `float` | Optional | - |\n| `stock_warning` | `float` | Optional | - |\n| `sku` | `string` | Optional | - |\n| `ean` | `string` | Optional | - |\n| `materials` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `tags` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `sources` | [`List of BillbeeInterfacesBillbeeAPIModelArticleSourceApiModel`](#billbee-interfaces-billbee-api-model-article-source-api-model) | Optional | - |\n| `weight` | `int` | Optional | - |\n| `weight_net` | `int` | Optional | - |\n| `low_stock` | `bool` | Optional | - |\n| `stock_code` | `string` | Optional | - |\n| `stock_reduce_items_per_sale` | `float` | Optional | - |\n| `stocks` | [`List of BillbeeInterfacesBillbeeAPIModelStockArticleApiModel`](#billbee-interfaces-billbee-api-model-stock-article-api-model) | Optional | - |\n| `category_1` | [`BillbeeInterfacesBillbeeAPIModelArticleCategoryApiModel`](#billbee-interfaces-billbee-api-model-article-category-api-model) | Optional | - |\n| `category_2` | [`BillbeeInterfacesBillbeeAPIModelArticleCategoryApiModel`](#billbee-interfaces-billbee-api-model-article-category-api-model) | Optional | - |\n| `category_3` | [`BillbeeInterfacesBillbeeAPIModelArticleCategoryApiModel`](#billbee-interfaces-billbee-api-model-article-category-api-model) | Optional | - |\n| `mtype` | `int` | Required | - |\n| `unit` | `int` | Optional | - |\n| `units_per_item` | `float` | Optional | - |\n| `sold_amount` | `float` | Optional | - |\n| `sold_sum_gross` | `float` | Optional | - |\n| `sold_sum_net` | `float` | Optional | - |\n| `sold_sum_net_last_30_days` | `float` | Optional | - |\n| `sold_sum_gross_last_30_days` | `float` | Optional | - |\n| `sold_amount_last_30_days` | `float` | Optional | - |\n| `shipping_product_id` | `int` | Optional | - |\n| `is_digital` | `bool` | Required | - |\n| `is_customizable` | `bool` | Required | - |\n| `delivery_time` | `int` | Optional | - |\n| `recipient` | `int` | Optional | - |\n| `occasion` | `int` | Optional | - |\n| `country_of_origin` | `string` | Optional | - |\n| `export_description` | `string` | Optional | - |\n| `taric_number` | `string` | Optional | - |\n| `condition` | `int` | Optional | - |\n| `width_cm` | `float` | Optional | - |\n| `length_cm` | `float` | Optional | - |\n| `height_cm` | `float` | Optional | - |\n| `bill_of_material` | [`List of BillbeeInterfacesBillbeeAPIModelBomSubArticleApiModel`](#billbee-interfaces-billbee-api-model-bom-sub-article-api-model) | Optional | - |\n| `custom_fields` | [`List of BillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldValueModel`](#billbee-interfaces-billbee-api-model-article-api-custom-field-value-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "InvoiceText": null,\n  "Manufacturer": null,\n  "Id": null,\n  "Title": null,\n  "Description": null,\n  "ShortDescription": null,\n  "BasicAttributes": null,\n  "Images": null,\n  "VatIndex": 40,\n  "Price": 6.2,\n  "CostPrice": null,\n  "Vat1Rate": 232.14,\n  "Vat2Rate": 233.2,\n  "StockDesired": null,\n  "StockCurrent": null,\n  "StockWarning": null,\n  "SKU": null,\n  "EAN": null,\n  "Materials": null,\n  "Tags": null,\n  "Sources": null,\n  "Weight": null,\n  "WeightNet": null,\n  "LowStock": null,\n  "StockCode": null,\n  "StockReduceItemsPerSale": null,\n  "Stocks": null,\n  "Category1": null,\n  "Category2": null,\n  "Category3": null,\n  "Type": 190,\n  "Unit": null,\n  "UnitsPerItem": null,\n  "SoldAmount": null,\n  "SoldSumGross": null,\n  "SoldSumNet": null,\n  "SoldSumNetLast30Days": null,\n  "SoldSumGrossLast30Days": null,\n  "SoldAmountLast30Days": null,\n  "ShippingProductId": null,\n  "IsDigital": false,\n  "IsCustomizable": false,\n  "DeliveryTime": null,\n  "Recipient": null,\n  "Occasion": null,\n  "CountryOfOrigin": null,\n  "ExportDescription": null,\n  "TaricNumber": null,\n  "Condition": null,\n  "WidthCm": null,\n  "LengthCm": null,\n  "HeightCm": null,\n  "BillOfMaterial": null,\n  "CustomFields": null\n}\n```\n\n#### Billbee Interfaces Order Multi Language String\n\n##### Class Name\n\n`BillbeeInterfacesOrderMultiLanguageString`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `text` | `string` | Optional | - |\n| `language_code` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Text": null,\n  "LanguageCode": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Article Image Relation Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `url` | `string` | Optional | - |\n| `id` | `long\\|int` | Optional | - |\n| `thumb_path_ext` | `string` | Optional | - |\n| `thumb_url` | `string` | Optional | - |\n| `position` | `int` | Optional | - |\n| `is_default` | `bool` | Optional | - |\n| `article_id` | `long\\|int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Url": null,\n  "Id": null,\n  "ThumbPathExt": null,\n  "ThumbUrl": null,\n  "Position": null,\n  "IsDefault": null,\n  "ArticleId": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Article Source Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelArticleSourceApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `int` | Optional | - |\n| `source` | `string` | Required | - |\n| `source_id` | `string` | Required | - |\n| `api_account_name` | `string` | Optional | - |\n| `api_account_id` | `long\\|int` | Optional | - |\n| `export_factor` | `float` | Optional | - |\n| `stock_sync_inactive` | `bool` | Optional | - |\n| `stock_sync_min` | `float` | Optional | - |\n| `stock_sync_max` | `float` | Optional | - |\n| `units_per_item` | `float` | Optional | - |\n| `custom` | `object` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "Source": "Source0",\n  "SourceId": "SourceId2",\n  "ApiAccountName": null,\n  "ApiAccountId": null,\n  "ExportFactor": null,\n  "StockSyncInactive": null,\n  "StockSyncMin": null,\n  "StockSyncMax": null,\n  "UnitsPerItem": null,\n  "Custom": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Stock Article Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelStockArticleApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `name` | `string` | Optional | - |\n| `stock_id` | `int` | Optional | - |\n| `stock_current` | `float` | Optional | - |\n| `stock_warning` | `float` | Optional | - |\n| `stock_code` | `string` | Optional | - |\n| `unfulfilled_amount` | `float` | Optional | - |\n| `stock_desired` | `float` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Name": null,\n  "StockId": null,\n  "StockCurrent": null,\n  "StockWarning": null,\n  "StockCode": null,\n  "UnfulfilledAmount": null,\n  "StockDesired": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Article Category Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelArticleCategoryApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `name` | `string` | Optional | - |\n| `id` | `long\\|int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Name": null,\n  "Id": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Bom Sub Article Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelBomSubArticleApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `amount` | `float` | Optional | - |\n| `article_id` | `long\\|int` | Optional | - |\n| `sku` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Amount": null,\n  "ArticleId": null,\n  "SKU": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Article Api Custom Field Value Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldValueModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | - |\n| `definition_id` | `long\\|int` | Optional | - |\n| `definition` | [`BillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`](#billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model) | Optional | - |\n| `article_id` | `long\\|int` | Optional | - |\n| `value` | `object` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "DefinitionId": null,\n  "Definition": null,\n  "ArticleId": null,\n  "Value": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Article Api Custom Field Definition Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | - |\n| `name` | `string` | Optional | - |\n| `configuration` | `object` | Optional | - |\n| `mtype` | [`TypeEnum`](#type) | Optional | - |\n| `is_nullable` | `bool` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "Name": null,\n  "Configuration": null,\n  "Type": null,\n  "IsNullable": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Article Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `paging` | [`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleApiModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-api-model) | Optional | - |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelArticleApiModel`](#billbee-interfaces-billbee-api-model-article-api-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Paging": null,\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Article Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Optional | - |\n| `total_pages` | `int` | Optional | - |\n| `total_rows` | `int` | Optional | - |\n| `page_size` | `int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Page": null,\n  "TotalPages": null,\n  "TotalRows": null,\n  "PageSize": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Article Api Custom Field Definition Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `paging` | [`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model) | Optional | - |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`](#billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Paging": null,\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Article Api Custom Field Definition Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Optional | - |\n| `total_pages` | `int` | Optional | - |\n| `total_rows` | `int` | Optional | - |\n| `page_size` | `int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Page": null,\n  "TotalPages": null,\n  "TotalRows": null,\n  "PageSize": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Article Api Custom Field Definition Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelArticleApiCustomFieldDefinitionModel`](#billbee-interfaces-billbee-api-model-article-api-custom-field-definition-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result System Collections Generic List Billbee Interfaces Billbee API Model Article Image Relation Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#billbee-interfaces-billbee-api-model-article-image-relation-api-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Article Image Relation Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelArticleImageRelationApiModel`](#billbee-interfaces-billbee-api-model-article-image-relation-api-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Deleted Images Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelDeletedImagesModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelDeletedImagesModel`](#billbee-interfaces-billbee-api-model-deleted-images-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Deleted Images Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelDeletedImagesModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `deleted` | `List of long\\|int` | Optional | - |\n| `not_found` | `List of long\\|int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Deleted": null,\n  "NotFound": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Automatic Provisioning Controller Create Account Container\n\nData used to create a new Billbee user account\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiAutomaticProvisioningControllerCreateAccountContainer`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `e_mail` | `string` | Required | The Email address of the user to create |\n| `password` | `string` | Optional | - |\n| `accept_terms` | `bool` | Optional | Set to true, if the user has accepted the Billbee terms &amp; conditions |\n| `address` | [`RechnungsdruckWebAppControllersApiAutomaticProvisioningControllerCreateAccountContainerUserAddress`](#rechnungsdruck-web-app-controllers-api-automatic-provisioning-controller-create-account-container-user-address) | Optional | Represents the invoice address of a Billbee user |\n| `newsletter` | `bool` | Optional | Gets or sets if the user is interested in the Billbee newsletter |\n| `affiliate_coupon_code` | `string` | Optional | Specifies an billbee affiliate code to attach to the user |\n| `vat_1_rate` | `float` | Optional | Optionally specify the vat1 (normal) rate of the user |\n| `vat_2_rate` | `float` | Optional | Optionally specify the vat2 (reduced) rate of the user |\n| `default_vat_mode` | [`DefaultVatModeEnum`](#default-vat-mode) | Optional | Optionally specify the default vat mode of the user |\n| `default_currrency` | `string` | Optional | Optionally specify the default currency of the user |\n| `default_vat_index` | `int` | Optional | Optionally specify the default vat index of the user |\n\n##### Example (as JSON)\n\n```json\n{\n  "EMail": "EMail0",\n  "Password": null,\n  "AcceptTerms": null,\n  "Address": null,\n  "Newsletter": null,\n  "AffiliateCouponCode": null,\n  "Vat1Rate": null,\n  "Vat2Rate": null,\n  "DefaultVatMode": null,\n  "DefaultCurrrency": null,\n  "DefaultVatIndex": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Automatic Provisioning Controller Create Account Container User Address\n\nRepresents the invoice address of a Billbee user\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiAutomaticProvisioningControllerCreateAccountContainerUserAddress`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `company` | `string` | Optional | - |\n| `name` | `string` | Optional | - |\n| `address_1` | `string` | Optional | - |\n| `address_2` | `string` | Optional | - |\n| `zip` | `string` | Optional | - |\n| `city` | `string` | Optional | - |\n| `country` | `string` | Optional | The ISO2 country code of the users country |\n| `vat_id` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Company": null,\n  "Name": null,\n  "Address1": null,\n  "Address2": null,\n  "Zip": null,\n  "City": null,\n  "Country": null,\n  "VatId": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result System Collections Generic List Billbee Interfaces Billbee API Model Cloud Storage Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCloudStorageApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelCloudStorageApiModel`](#billbee-interfaces-billbee-api-model-cloud-storage-api-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Cloud Storage Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelCloudStorageApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | - |\n| `name` | `string` | Optional | - |\n| `mtype` | `string` | Optional | - |\n| `used_as_printer` | `bool` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "Name": null,\n  "Type": null,\n  "UsedAsPrinter": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Customer Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `paging` | [`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-api-model) | Optional | - |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#billbee-interfaces-billbee-api-model-customer-api-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Paging": null,\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Customer Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Optional | - |\n| `total_pages` | `int` | Optional | - |\n| `total_rows` | `int` | Optional | - |\n| `page_size` | `int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Page": null,\n  "TotalPages": null,\n  "TotalRows": null,\n  "PageSize": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Customer Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelCustomerApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | The Billbee Id of the customer |\n| `name` | `string` | Optional | - |\n| `email` | `string` | Optional | - |\n| `tel_1` | `string` | Optional | - |\n| `tel_2` | `string` | Optional | - |\n| `number` | `int` | Optional | - |\n| `price_group_id` | `long\\|int` | Optional | - |\n| `language_id` | `int` | Optional | - |\n| `vat_id` | `string` | Optional | - |\n| `mtype` | `int` | Optional | Customer Type |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "Name": null,\n  "Email": null,\n  "Tel1": null,\n  "Tel2": null,\n  "Number": null,\n  "PriceGroupId": null,\n  "LanguageId": null,\n  "VatId": null,\n  "Type": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Create Customer Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelCreateCustomerApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `address` | [`BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#billbee-interfaces-billbee-api-model-customer-address-api-model) | Optional | Container for passing address data |\n| `id` | `long\\|int` | Optional | The Billbee Id of the customer |\n| `name` | `string` | Optional | - |\n| `email` | `string` | Optional | - |\n| `tel_1` | `string` | Optional | - |\n| `tel_2` | `string` | Optional | - |\n| `number` | `int` | Optional | - |\n| `price_group_id` | `long\\|int` | Optional | - |\n| `language_id` | `int` | Optional | - |\n| `vat_id` | `string` | Optional | - |\n| `mtype` | `int` | Optional | Customer Type |\n\n##### Example (as JSON)\n\n```json\n{\n  "Address": null,\n  "Id": null,\n  "Name": null,\n  "Email": null,\n  "Tel1": null,\n  "Tel2": null,\n  "Number": null,\n  "PriceGroupId": null,\n  "LanguageId": null,\n  "VatId": null,\n  "Type": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Customer Address Api Model\n\nContainer for passing address data\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | The internal Billbee ID of the address record. Can be null if a new address is created |\n| `address_type` | [`AddressTypeEnum`](#address-type) | Optional | The type of the address |\n| `customer_id` | `long\\|int` | Optional | The internal Billbee id of the customer the address belongs to |\n| `company` | `string` | Optional | The name of the company |\n| `first_name` | `string` | Optional | - |\n| `last_name` | `string` | Optional | - |\n| `name_2` | `string` | Optional | Optionally an additional name field |\n| `street` | `string` | Optional | - |\n| `housenumber` | `string` | Optional | - |\n| `zip` | `string` | Optional | - |\n| `city` | `string` | Optional | - |\n| `state` | `string` | Optional | - |\n| `country_code` | `string` | Optional | The ISO2 code of the country |\n| `email` | `string` | Optional | - |\n| `tel_1` | `string` | Optional | - |\n| `tel_2` | `string` | Optional | - |\n| `fax` | `string` | Optional | - |\n| `address_addition` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "AddressType": null,\n  "CustomerId": null,\n  "Company": null,\n  "FirstName": null,\n  "LastName": null,\n  "Name2": null,\n  "Street": null,\n  "Housenumber": null,\n  "Zip": null,\n  "City": null,\n  "State": null,\n  "CountryCode": null,\n  "Email": null,\n  "Tel1": null,\n  "Tel2": null,\n  "Fax": null,\n  "AddressAddition": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Customer Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#billbee-interfaces-billbee-api-model-customer-api-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Rechnungsdruck Web App Controllers Api Order\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListRechnungsdruckWebAppControllersApiOrder`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `paging` | [`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListRechnungsdruckWebAppControllersApiOrder`](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-rechnungsdruck-web-app-controllers-api-order) | Optional | - |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of RechnungsdruckWebAppControllersApiOrder`](#rechnungsdruck-web-app-controllers-api-order) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Paging": null,\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Rechnungsdruck Web App Controllers Api Order\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListRechnungsdruckWebAppControllersApiOrder`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Optional | - |\n| `total_pages` | `int` | Optional | - |\n| `total_rows` | `int` | Optional | - |\n| `page_size` | `int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Page": null,\n  "TotalPages": null,\n  "TotalRows": null,\n  "PageSize": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Order\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiOrder`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `int` | Optional | - |\n| `external_id` | `string` | Optional | - |\n| `invoice_number` | `string` | Optional | - |\n| `invoice_created_at` | `datetime` | Optional | - |\n| `invoice_date` | `datetime` | Optional | - |\n| `created_at` | `datetime` | Optional | - |\n| `paid_at` | `datetime` | Optional | - |\n| `shipped_at` | `datetime` | Optional | - |\n| `has_invoice` | `bool` | Optional | - |\n| `order_state_id` | `int` | Optional | - |\n| `order_state_text` | `string` | Optional | - |\n| `total_gross` | `float` | Optional | - |\n| `shop_name` | `string` | Optional | - |\n| `can_create_auto_invoice` | `bool` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "ExternalId": null,\n  "InvoiceNumber": null,\n  "InvoiceCreatedAt": null,\n  "InvoiceDate": null,\n  "CreatedAt": null,\n  "PaidAt": null,\n  "ShippedAt": null,\n  "HasInvoice": null,\n  "OrderStateId": null,\n  "OrderStateText": null,\n  "TotalGross": null,\n  "ShopName": null,\n  "CanCreateAutoInvoice": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Customer Address Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `paging` | [`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-customer-address-api-model) | Optional | - |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#billbee-interfaces-billbee-api-model-customer-address-api-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Paging": null,\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Customer Address Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Optional | - |\n| `total_pages` | `int` | Optional | - |\n| `total_rows` | `int` | Optional | - |\n| `page_size` | `int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Page": null,\n  "TotalPages": null,\n  "TotalRows": null,\n  "PageSize": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Customer Address Api Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelCustomerAddressApiModel`](#billbee-interfaces-billbee-api-model-customer-address-api-model) | Optional | Container for passing address data |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result System Collections Generic List Billbee Interfaces Billbee API Models Layout Template\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelsLayoutTemplate`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelsLayoutTemplate`](#billbee-interfaces-billbee-api-models-layout-template) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Billbee Interfaces Billbee API Models Layout Template\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelsLayoutTemplate`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | - |\n| `name` | `string` | Optional | - |\n| `mtype` | [`Type1Enum`](#type-1) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "Name": null,\n  "Type": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Order\n\nA class that represents the Billbee data model of a single order\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelOrder`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `rebate_difference` | `float` | Optional | - |\n| `shipping_ids` | [`List of BillbeeInterfacesBillbeeAPIModelShipment`](#billbee-interfaces-billbee-api-model-shipment) | Optional | The shipments of the order |\n| `accept_loss_of_return_right` | `bool` | Optional | Customer accepts loss due to withdrawal |\n| `id` | `string` | Optional | Id of the order in the external system (marketplace) |\n| `order_number` | `string` | Optional | Order number of the order in the external system (marketplace) |\n| `state` | [`StateEnum`](#state) | Optional | The current state of the order |\n| `vat_mode` | [`VatModeEnum`](#vat-mode) | Optional | The vat mode of the order |\n| `created_at` | `datetime` | Optional | The date on which the order was created |\n| `shipped_at` | `datetime` | Optional | The date on which the order was shipped |\n| `confirmed_at` | `datetime` | Optional | The date on which the order was confirmed |\n| `payed_at` | `datetime` | Optional | The date on which the order was paid |\n| `seller_comment` | `string` | Optional | An internal seller comment |\n| `comments` | [`List of BillbeeInterfacesBillbeeAPIModelCommentApiModel`](#billbee-interfaces-billbee-api-model-comment-api-model) | Optional | All messages / comments of the order |\n| `invoice_number_prefix` | `string` | Optional | The prefix of the invoice number |\n| `invoice_number_postfix` | `string` | Optional | The postfix of the invoice number |\n| `invoice_number` | `int` | Optional | The invoice number |\n| `invoice_date` | `datetime` | Optional | The date on which the invoice was created |\n| `invoice_address` | [`BillbeeInterfacesBillbeeAPIModelOrderAddressApiModel`](#billbee-interfaces-billbee-api-model-order-address-api-model) | Optional | - |\n| `shipping_address` | [`BillbeeInterfacesBillbeeAPIModelOrderAddressApiModel`](#billbee-interfaces-billbee-api-model-order-address-api-model) | Optional | - |\n| `payment_method` | [`PaymentMethodEnum`](#payment-method) | Optional | The payment method |\n| `shipping_cost` | `float` | Optional | The shipping cost |\n| `total_cost` | `float` | Optional | The total cost excluding shipping cost |\n| `adjustment_cost` | `float` | Optional | - |\n| `adjustment_reason` | `string` | Optional | - |\n| `order_items` | [`List of BillbeeInterfacesBillbeeAPIModelOrderItem`](#billbee-interfaces-billbee-api-model-order-item) | Optional | The list of items purchased like shirt, pant, toys etc |\n| `currency` | `string` | Optional | The three letter currency code. |\n| `seller` | [`BillbeeInterfacesBillbeeAPIModelOrderUser`](#billbee-interfaces-billbee-api-model-order-user) | Optional | - |\n| `buyer` | [`BillbeeInterfacesBillbeeAPIModelOrderUser`](#billbee-interfaces-billbee-api-model-order-user) | Optional | - |\n| `updated_at` | `datetime` | Optional | The date on which the order was last updated |\n| `tax_rate_1` | `float` | Optional | The regular tax rate |\n| `tax_rate_2` | `float` | Optional | The reduced tax rate |\n| `bill_bee_order_id` | `long\\|int` | Optional | The Order.Id from the Billbee database |\n| `bill_bee_parent_order_id` | `long\\|int` | Optional | The Id of the parent order in the Billbee database |\n| `vat_id` | `string` | Optional | The customers vat id |\n| `tags` | `List of string` | Optional | The Tags of the order |\n| `ship_weight_kg` | `float` | Optional | The total weight of the shipment(s) |\n| `language_code` | `string` | Optional | The two-letter language code of the customer |\n| `paid_amount` | `float` | Optional | - |\n| `shipping_profile_id` | `string` | Optional | Internal Id for the shipping profile for that order |\n| `shipping_provider_id` | `long\\|int` | Optional | Internal Id for the used shipping provider |\n| `shipping_provider_product_id` | `long\\|int` | Optional | Internal Id for the used shipping product |\n| `shipping_provider_name` | `string` | Optional | The Name for of used shipping provider |\n| `shipping_provider_product_name` | `string` | Optional | The Name of the used shipping product |\n| `shipping_profile_name` | `string` | Optional | Display Name of Shipping profile, if available |\n| `payment_instruction` | `string` | Optional | A textfield optionaly filled with a payment instruction text for printout on the invoice (z.B. Ebay Kauf auf Rechnung) |\n| `is_cancelation_for` | `string` | Optional | An optional Order Id (externalid) for an order if this is a cancel order (shopify only at the moment) |\n| `payment_transaction_id` | `string` | Optional | The id of the payment transaction. For example the transaction id of PayPal payment. Should not be used any more. Please use \'Payments\' instead. |\n| `distribution_center` | `string` | Optional | An optional code for the distribution center delivering this order |\n| `delivery_source_country_code` | `string` | Optional | An optional Country ISO2 Code of the country where order is shipped from (FBA) |\n| `custom_invoice_note` | `string` | Optional | An optional multiline text which is printed on the invoice |\n| `customer_number` | `string` | Optional | The customer number (not to be confused with the id of the customer) |\n| `payment_reference` | `string` | Optional | A payment reference. Should not be used any more. Please use \'Payments\' instead. |\n| `shipping_services` | [`List of BillbeeInterfacesShippingProductService`](#billbee-interfaces-shipping-product-service) | Optional | Additional services for the shipment |\n| `customer` | [`BillbeeInterfacesBillbeeAPIModelCustomerApiModel`](#billbee-interfaces-billbee-api-model-customer-api-model) | Optional | - |\n| `history` | [`List of BillbeeInterfacesOrderHistoryEntry`](#billbee-interfaces-order-history-entry) | Optional | - |\n| `payments` | [`List of BillbeeInterfacesBillbeeAPIModelsOrderPayment`](#billbee-interfaces-billbee-api-models-order-payment) | Optional | - |\n| `last_modified_at` | `datetime` | Optional | Date of the last update, the order got |\n\n##### Example (as JSON)\n\n```json\n{\n  "RebateDifference": null,\n  "ShippingIds": null,\n  "AcceptLossOfReturnRight": null,\n  "Id": null,\n  "OrderNumber": null,\n  "State": null,\n  "VatMode": null,\n  "CreatedAt": null,\n  "ShippedAt": null,\n  "ConfirmedAt": null,\n  "PayedAt": null,\n  "SellerComment": null,\n  "Comments": null,\n  "InvoiceNumberPrefix": null,\n  "InvoiceNumberPostfix": null,\n  "InvoiceNumber": null,\n  "InvoiceDate": null,\n  "InvoiceAddress": null,\n  "ShippingAddress": null,\n  "PaymentMethod": null,\n  "ShippingCost": null,\n  "TotalCost": null,\n  "AdjustmentCost": null,\n  "AdjustmentReason": null,\n  "OrderItems": null,\n  "Currency": null,\n  "Seller": null,\n  "Buyer": null,\n  "UpdatedAt": null,\n  "TaxRate1": null,\n  "TaxRate2": null,\n  "BillBeeOrderId": null,\n  "BillBeeParentOrderId": null,\n  "VatId": null,\n  "Tags": null,\n  "ShipWeightKg": null,\n  "LanguageCode": null,\n  "PaidAmount": null,\n  "ShippingProfileId": null,\n  "ShippingProviderId": null,\n  "ShippingProviderProductId": null,\n  "ShippingProviderName": null,\n  "ShippingProviderProductName": null,\n  "ShippingProfileName": null,\n  "PaymentInstruction": null,\n  "IsCancelationFor": null,\n  "PaymentTransactionId": null,\n  "DistributionCenter": null,\n  "DeliverySourceCountryCode": null,\n  "CustomInvoiceNote": null,\n  "CustomerNumber": null,\n  "PaymentReference": null,\n  "ShippingServices": null,\n  "Customer": null,\n  "History": null,\n  "Payments": null,\n  "LastModifiedAt": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Shipment\n\nRepresents a single shipment.\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelShipment`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `billbee_id` | `long\\|int` | Optional | The billbee internal id of the shipment |\n| `shipping_id` | `string` | Optional | The id of this shipment |\n| `shipper` | `string` | Optional | The name of the shipping provider |\n| `created` | `datetime` | Optional | The creation date |\n| `tracking_url` | `string` | Optional | The url to track this shipment |\n| `shipping_provider_id` | `long\\|int` | Optional | The id of the used shipping provider |\n| `shipping_provider_product_id` | `long\\|int` | Optional | The id of the used shipping provider product |\n| `shipping_carrier` | `int` | Optional | The carrier used to ship the parcel with |\n\n##### Example (as JSON)\n\n```json\n{\n  "BillbeeId": null,\n  "ShippingId": null,\n  "Shipper": null,\n  "Created": null,\n  "TrackingUrl": null,\n  "ShippingProviderId": null,\n  "ShippingProviderProductId": null,\n  "ShippingCarrier": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Comment Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelCommentApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | - |\n| `from_customer` | `bool` | Optional | - |\n| `text` | `string` | Optional | - |\n| `name` | `string` | Optional | - |\n| `created` | `datetime` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "FromCustomer": null,\n  "Text": null,\n  "Name": null,\n  "Created": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Order Address Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelOrderAddressApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `billbee_id` | `long\\|int` | Optional | - |\n| `first_name` | `string` | Optional | - |\n| `last_name` | `string` | Optional | - |\n| `company` | `string` | Optional | - |\n| `name_addition` | `string` | Optional | - |\n| `street` | `string` | Optional | - |\n| `house_number` | `string` | Optional | - |\n| `zip` | `string` | Optional | - |\n| `city` | `string` | Optional | - |\n| `country_iso_2` | `string` | Optional | - |\n| `country` | `string` | Optional | - |\n| `line_2` | `string` | Optional | - |\n| `email` | `string` | Optional | - |\n| `state` | `string` | Optional | - |\n| `phone` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "BillbeeId": null,\n  "FirstName": null,\n  "LastName": null,\n  "Company": null,\n  "NameAddition": null,\n  "Street": null,\n  "HouseNumber": null,\n  "Zip": null,\n  "City": null,\n  "CountryISO2": null,\n  "Country": null,\n  "Line2": null,\n  "Email": null,\n  "State": null,\n  "Phone": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Order Item\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelOrderItem`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `billbee_id` | `long\\|int` | Optional | The billbee id of this item |\n| `transaction_id` | `string` | Optional | Id of the individual transaction. Only required by Ebay to detect aggregated orders |\n| `product` | [`BillbeeInterfacesBillbeeAPIModelSoldProduct`](#billbee-interfaces-billbee-api-model-sold-product) | Optional | - |\n| `quantity` | `float` | Optional | The sold quantity |\n| `total_price` | `float` | Optional | The total price (unit price * quantity) |\n| `tax_amount` | `float` | Optional | The tax amount in the total price |\n| `tax_index` | `int` | Optional | The tax index. |\n| `discount` | `float` | Optional | Sets the discount in percent |\n| `attributes` | [`List of BillbeeInterfacesBillbeeAPIModelOrderItemAttribute`](#billbee-interfaces-billbee-api-model-order-item-attribute) | Optional | A list of product attributes for this position |\n| `get_price_from_article_if_any` | `bool` | Optional | If true, the price will be overwritten by the known article price in billbee if available |\n| `is_coupon` | `bool` | Optional | Determines if it is a coupon, which is interpreted as tax-free payment |\n| `shipping_profile_id` | `string` | Optional | Determines if it is a coupon, which is interpreted as tax-free payment |\n| `dont_adjust_stock` | `bool` | Optional | If true, the import of this order won\'t adjust the stock level at billbee. |\n| `unrebated_total_price` | `float` | Optional | Is just used for the billbee api |\n| `serial_number` | `string` | Optional | Contains the used serial number |\n\n##### Example (as JSON)\n\n```json\n{\n  "BillbeeId": null,\n  "TransactionId": null,\n  "Product": null,\n  "Quantity": null,\n  "TotalPrice": null,\n  "TaxAmount": null,\n  "TaxIndex": null,\n  "Discount": null,\n  "Attributes": null,\n  "GetPriceFromArticleIfAny": null,\n  "IsCoupon": null,\n  "ShippingProfileId": null,\n  "DontAdjustStock": null,\n  "UnrebatedTotalPrice": null,\n  "SerialNumber": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Order User\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelOrderUser`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `platform` | `string` | Optional | The name of the platform from which this customer originated |\n| `billbee_shop_name` | `string` | Optional | The name of shop connection in Billbee |\n| `billbee_shop_id` | `long\\|int` | Optional | The Billbee internal id of the shop connection |\n| `id` | `string` | Optional | The Billbee internal id of this customer |\n| `nick` | `string` | Optional | The nick name (if available) from the original system |\n| `first_name` | `string` | Optional | The first name of this customer |\n| `last_name` | `string` | Optional | The last name of this customer |\n| `full_name` | `string` | Optional | The full name ("firstname lastname") of this customer |\n| `email` | `string` | Optional | The email address of this customer |\n\n##### Example (as JSON)\n\n```json\n{\n  "Platform": null,\n  "BillbeeShopName": null,\n  "BillbeeShopId": null,\n  "Id": null,\n  "Nick": null,\n  "FirstName": null,\n  "LastName": null,\n  "FullName": null,\n  "Email": null\n}\n```\n\n#### Billbee Interfaces Shipping Product Service\n\n##### Class Name\n\n`BillbeeInterfacesShippingProductService`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `display_name` | `string` | Optional | - |\n| `display_value` | `string` | Optional | - |\n| `requires_user_input` | `bool` | Optional | - |\n| `service_name` | `string` | Optional | - |\n| `type_name` | `string` | Optional | - |\n| `possible_value_lists` | [`List of SystemCollectionsGenericKeyValuePairSystemStringSystemCollectionsGenericListSystemCollectionsGenericKeyValuePairSystemInt32SystemString`](#system-collections-generic-key-value-pair-system-string-system-collections-generic-list-system-collections-generic-key-value-pair-system-int-32-system-string) | Optional | - |\n| `can_be_configured` | `bool` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "DisplayName": null,\n  "DisplayValue": null,\n  "RequiresUserInput": null,\n  "ServiceName": null,\n  "typeName": null,\n  "PossibleValueLists": null,\n  "CanBeConfigured": null\n}\n```\n\n#### Billbee Interfaces Order History Entry\n\n##### Class Name\n\n`BillbeeInterfacesOrderHistoryEntry`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `created` | `datetime` | Optional | - |\n| `event_type_name` | `string` | Optional | - |\n| `text` | `string` | Optional | - |\n| `employee_name` | `string` | Optional | - |\n| `type_id` | `int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Created": null,\n  "EventTypeName": null,\n  "Text": null,\n  "EmployeeName": null,\n  "TypeId": null\n}\n```\n\n#### Billbee Interfaces Billbee API Models Order Payment\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelsOrderPayment`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `billbee_id` | `long\\|int` | Optional | - |\n| `transaction_id` | `string` | Optional | - |\n| `pay_date` | `datetime` | Optional | - |\n| `payment_type` | `int` | Optional | - |\n| `source_technology` | `string` | Optional | - |\n| `source_text` | `string` | Optional | - |\n| `pay_value` | `float` | Optional | - |\n| `purpose` | `string` | Optional | - |\n| `name` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "BillbeeId": null,\n  "TransactionId": null,\n  "PayDate": null,\n  "PaymentType": null,\n  "SourceTechnology": null,\n  "SourceText": null,\n  "PayValue": null,\n  "Purpose": null,\n  "Name": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Sold Product\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelSoldProduct`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `old_id` | `string` | Optional | This is for migration scenarios when the internal id of a product changes<br>I.E. Etsy when switching to the new inventory management, the ids for variants will change. |\n| `id` | `string` | Optional | The id of this product in the external system |\n| `title` | `string` | Optional | The name of this product |\n| `weight` | `int` | Optional | Weight of one item in gram |\n| `sku` | `string` | Optional | The SKU of this product |\n| `sku_or_id` | `string` | Optional | The SKU of this product or the id if the SKU is empty |\n| `is_digital` | `bool` | Optional | True if the product is a digital good (download etc.), false if not |\n| `images` | [`List of BillbeeInterfacesBillbeeAPIModelProductImage`](#billbee-interfaces-billbee-api-model-product-image) | Optional | The images of this product |\n| `ean` | `string` | Optional | The EAN / GTIN of this product |\n| `platform_data` | `string` | Optional | Optional platform specific Data as serialized JSON Object for the product |\n| `taric_code` | `string` | Optional | The TARIC code |\n| `country_of_origin` | `string` | Optional | The country where this article was made |\n| `billbee_id` | `long\\|int` | Optional | The Billbee internal id of the linked product |\n\n##### Example (as JSON)\n\n```json\n{\n  "OldId": null,\n  "Id": null,\n  "Title": null,\n  "Weight": null,\n  "SKU": null,\n  "SkuOrId": null,\n  "IsDigital": null,\n  "Images": null,\n  "EAN": null,\n  "PlatformData": null,\n  "TARICCode": null,\n  "CountryOfOrigin": null,\n  "BillbeeId": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Order Item Attribute\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelOrderItemAttribute`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `string` | Optional | The internal id of this attribute |\n| `name` | `string` | Optional | The attribute name. E.g. color |\n| `value` | `string` | Optional | The attribute value. E.g. red |\n| `price` | `float` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "Name": null,\n  "Value": null,\n  "Price": null\n}\n```\n\n#### System Collections Generic Key Value Pair System String System Collections Generic List System Collections Generic Key Value Pair System Int 32 System String\n\n##### Class Name\n\n`SystemCollectionsGenericKeyValuePairSystemStringSystemCollectionsGenericListSystemCollectionsGenericKeyValuePairSystemInt32SystemString`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `key` | `string` | Optional | - |\n| `value` | [`List of SystemCollectionsGenericKeyValuePairSystemInt32SystemString`](#system-collections-generic-key-value-pair-system-int-32-system-string) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "key": null,\n  "value": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Product Image\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelProductImage`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `url` | `string` | Optional | The url to the image |\n| `is_default_image` | `bool` | Optional | True if the image is / should be the default image |\n| `position` | `int` | Optional | The order of this image |\n| `external_id` | `string` | Optional | The id of this image from the original system |\n\n##### Example (as JSON)\n\n```json\n{\n  "Url": null,\n  "IsDefaultImage": null,\n  "Position": null,\n  "ExternalId": null\n}\n```\n\n#### System Collections Generic Key Value Pair System Int 32 System String\n\n##### Class Name\n\n`SystemCollectionsGenericKeyValuePairSystemInt32SystemString`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `key` | `int` | Optional | - |\n| `value` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "key": null,\n  "value": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Billbee Interfaces Billbee API Model Order\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultBillbeeInterfacesBillbeeAPIModelOrder`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`BillbeeInterfacesBillbeeAPIModelOrder`](#billbee-interfaces-billbee-api-model-order) | Optional | A class that represents the Billbee data model of a single order |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Order\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelOrder`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `paging` | [`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelOrder`](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-order) | Optional | - |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelOrder`](#billbee-interfaces-billbee-api-model-order) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Paging": null,\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Order\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelOrder`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Optional | - |\n| `total_pages` | `int` | Optional | - |\n| `total_rows` | `int` | Optional | - |\n| `page_size` | `int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Page": null,\n  "TotalPages": null,\n  "TotalRows": null,\n  "PageSize": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Order Tag Create\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiOrderTagCreate`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `tags` | `List of string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Tags": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Order State Update\n\nSpecifies the parameters used to set the new state of an order\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiOrderStateUpdate`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `new_state_id` | [`NewStateIdEnum`](#new-state-id) | Optional | The new state to set |\n\n##### Example (as JSON)\n\n```json\n{\n  "NewStateId": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Add Shipment to Order Model\n\nData of the shipment to be created\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiAddShipmentToOrderModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `shipping_id` | `string` | Optional | The id of the shipment (Sendungsnummer/trackingid) |\n| `order_id` | `string` | Optional | Optional a differing order number of the shipment if available |\n| `comment` | `string` | Optional | Optional a text stored with the shipment |\n| `shipping_provider_id` | `long\\|int` | Optional | Optional the id of a shipping provider existing in the billbee account that should be assigned to the shipment |\n| `shipping_provider_product_id` | `long\\|int` | Optional | Optional the id of a shipping provider product that should be assigend to the shipment |\n| `carrier_id` | `int` | Optional | Optional the id of a shipping carrier that should be assigend to the shipment<br>Will override the carrier from the shipment product.<br>Please use the integer value from this Enumeration:<br>{Billbee.Interfaces.Shipping.Enums.ShippingCarrier} |\n\n##### Example (as JSON)\n\n```json\n{\n  "ShippingId": null,\n  "OrderId": null,\n  "Comment": null,\n  "ShippingProviderId": null,\n  "ShippingProviderProductId": null,\n  "CarrierId": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Order Api Controller Send Message Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiOrderApiControllerSendMessageModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `send_mode` | [`SendModeEnum`](#send-mode) | Optional | - |\n| `subject` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `body` | [`List of BillbeeInterfacesOrderMultiLanguageString`](#billbee-interfaces-order-multi-language-string) | Optional | - |\n| `alternative_mail` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "SendMode": null,\n  "Subject": null,\n  "Body": null,\n  "AlternativeMail": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Order Api Controller Trigger Event Container\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiOrderApiControllerTriggerEventContainer`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `name` | `string` | Optional | Name of the event |\n| `delay_in_minutes` | `int` | Optional | The delay in minutes until the rule is executed |\n\n##### Example (as JSON)\n\n```json\n{\n  "Name": null,\n  "DelayInMinutes": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Order Api Controller Parse Text Container\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiOrderApiControllerParseTextContainer`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `text_to_parse` | `string` | Optional | The text to parse and replace the placeholders in. |\n| `is_html` | `bool` | Optional | If true, the string will be handled as html. |\n| `language` | `string` | Optional | The ISO 639-1 code of the target language. Using default if not set. |\n| `trim` | `bool` | Optional | If true, then the placeholder values are trimmed after usage. |\n\n##### Example (as JSON)\n\n```json\n{\n  "TextToParse": null,\n  "IsHtml": null,\n  "Language": null,\n  "Trim": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Search Controller Search Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiSearchControllerSearchModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `mtype` | `List of string` | Optional | - |\n| `term` | `string` | Optional | - |\n| `search_mode` | [`SearchModeEnum`](#search-mode) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Type": null,\n  "Term": null,\n  "SearchMode": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Rechnungsdruck Web App Controllers Api Search Controller Search Results Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultRechnungsdruckWebAppControllersApiSearchControllerSearchResultsModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`RechnungsdruckWebAppControllersApiSearchControllerSearchResultsModel`](#rechnungsdruck-web-app-controllers-api-search-controller-search-results-model) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Search Controller Search Results Model\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiSearchControllerSearchResultsModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `products` | [`List of RechnungsdruckWebAppControllersApiSearchControllerProductResult`](#rechnungsdruck-web-app-controllers-api-search-controller-product-result) | Optional | - |\n| `orders` | [`List of RechnungsdruckWebAppControllersApiSearchControllerOrderResult`](#rechnungsdruck-web-app-controllers-api-search-controller-order-result) | Optional | - |\n| `customers` | [`List of RechnungsdruckWebAppControllersApiSearchControllerCustomerResult`](#rechnungsdruck-web-app-controllers-api-search-controller-customer-result) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Products": null,\n  "Orders": null,\n  "Customers": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Search Controller Product Result\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiSearchControllerProductResult`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | - |\n| `short_text` | `string` | Optional | - |\n| `sku` | `string` | Optional | - |\n| `tags` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "ShortText": null,\n  "SKU": null,\n  "Tags": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Search Controller Order Result\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiSearchControllerOrderResult`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | - |\n| `external_reference` | `string` | Optional | - |\n| `buyer_name` | `string` | Optional | - |\n| `invoice_number` | `string` | Optional | - |\n| `customer_name` | `string` | Optional | - |\n| `article_texts` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "ExternalReference": null,\n  "BuyerName": null,\n  "InvoiceNumber": null,\n  "CustomerName": null,\n  "ArticleTexts": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Search Controller Customer Result\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiSearchControllerCustomerResult`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `long\\|int` | Optional | - |\n| `name` | `string` | Optional | - |\n| `addresses` | `string` | Optional | - |\n| `number` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "Name": null,\n  "Addresses": null,\n  "Number": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Create Shipment Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelCreateShipmentApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `provider_name` | `string` | Optional | The name of the provider as specified in the billbee account |\n| `product_code` | `string` | Optional | The productcode to be used when creating the shipment. Values depends on the carrier used |\n| `printer_name` | `string` | Optional | The name of a connected Cloudprinter to sent the label to |\n| `printer_id_for_export_docs` | `long\\|int` | Optional | The id of a connected Cloudprinter to sent the export docs to |\n| `services` | [`List of BillbeeInterfacesShippingProductService`](#billbee-interfaces-shipping-product-service) | Optional | A list of services to be used when creating the shipment |\n| `receiver_address` | [`BillbeeInterfacesBillbeeAPIModelShipmentAddressApiModel`](#billbee-interfaces-billbee-api-model-shipment-address-api-model) | Optional | - |\n| `client_reference` | `string` | Optional | Optional specify a text to be included on the label. Not possible with all carriers |\n| `customer_number` | `string` | Optional | Not used anymore |\n| `weight_in_gram` | `float` | Optional | Optional specify the weight in gram of the shipment |\n| `order_sum` | `float` | Optional | The value of the shipments content |\n| `order_currency_code` | `string` | Optional | The Currency if the ordersum |\n| `content` | `string` | Optional | Optional specify a text describing the content of the shipment. Used for export shipments |\n| `ship_date` | `datetime` | Optional | Optional overwrite the shipdate to be transferred to the carrier |\n| `shipping_carrier` | [`ShippingCarrierEnum`](#shipping-carrier) | Optional | - |\n| `dimension` | [`BillbeeInterfacesShippingShipmentDataDimensions`](#billbee-interfaces-shipping-shipment-data-dimensions) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ProviderName": null,\n  "ProductCode": null,\n  "PrinterName": null,\n  "PrinterIdForExportDocs": null,\n  "Services": null,\n  "ReceiverAddress": null,\n  "ClientReference": null,\n  "CustomerNumber": null,\n  "WeightInGram": null,\n  "OrderSum": null,\n  "OrderCurrencyCode": null,\n  "Content": null,\n  "ShipDate": null,\n  "shippingCarrier": null,\n  "Dimension": null\n}\n```\n\n#### Billbee Interfaces Billbee API Model Shipment Address Api Model\n\n##### Class Name\n\n`BillbeeInterfacesBillbeeAPIModelShipmentAddressApiModel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `company` | `string` | Optional | - |\n| `first_name` | `string` | Optional | - |\n| `last_name` | `string` | Optional | - |\n| `name_2` | `string` | Optional | - |\n| `street` | `string` | Optional | - |\n| `housenumber` | `string` | Optional | - |\n| `zip` | `string` | Optional | - |\n| `city` | `string` | Optional | - |\n| `country_code` | `string` | Optional | - |\n| `country_code_iso_3` | `string` | Optional | - |\n| `email` | `string` | Optional | - |\n| `telephone` | `string` | Optional | - |\n| `address_addition` | `string` | Optional | - |\n| `is_export_country` | `bool` | Optional | - |\n| `state` | `string` | Optional | - |\n| `full_name` | `string` | Optional | - |\n| `full_street` | `string` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Company": null,\n  "FirstName": null,\n  "LastName": null,\n  "Name2": null,\n  "Street": null,\n  "Housenumber": null,\n  "Zip": null,\n  "City": null,\n  "CountryCode": null,\n  "CountryCodeISO3": null,\n  "Email": null,\n  "Telephone": null,\n  "AddressAddition": null,\n  "IsExportCountry": null,\n  "State": null,\n  "FullName": null,\n  "FullStreet": null\n}\n```\n\n#### Billbee Interfaces Shipping Shipment Data Dimensions\n\n##### Class Name\n\n`BillbeeInterfacesShippingShipmentDataDimensions`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `length` | `float` | Optional | - |\n| `width` | `float` | Optional | - |\n| `height` | `float` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "length": null,\n  "width": null,\n  "height": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Shipment With Label\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiShipmentWithLabel`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `order_id` | `long\\|int` | Optional | The Billbee internal id of the order to ship |\n| `provider_id` | `long\\|int` | Optional | The id of the provider. You can query all providers with the shippingproviders endpoint |\n| `product_id` | `long\\|int` | Optional | the id of the shipping provider product to be used |\n| `change_state_to_send` | `bool` | Optional | Optional parameter to automatically change the orderstate to sent after creating the shipment |\n| `printer_name` | `string` | Optional | Optional the name of a connected cloudprinter to send the label to |\n| `weight_in_gram` | `int` | Optional | Optional the shipments weight in gram to override the calculated weight |\n| `ship_date` | `datetime` | Optional | Optional specify the shipdate to be transmitted to the carrier |\n| `client_reference` | `string` | Optional | Optional specify a reference text to be included on the label. Works not with all carriers |\n| `dimension` | [`BillbeeInterfacesShippingShipmentDataDimensions`](#billbee-interfaces-shipping-shipment-data-dimensions) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "OrderId": null,\n  "ProviderId": null,\n  "ProductId": null,\n  "ChangeStateToSend": null,\n  "PrinterName": null,\n  "WeightInGram": null,\n  "ShipDate": null,\n  "ClientReference": null,\n  "Dimension": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Result Rechnungsdruck Web App Controllers Api Shipment With Label Result\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiResultRechnungsdruckWebAppControllersApiShipmentWithLabelResult`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`RechnungsdruckWebAppControllersApiShipmentWithLabelResult`](#rechnungsdruck-web-app-controllers-api-shipment-with-label-result) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Shipment With Label Result\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiShipmentWithLabelResult`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `order_id` | `long\\|int` | Optional | - |\n| `order_reference` | `string` | Optional | - |\n| `shipping_id` | `string` | Optional | - |\n| `tracking_url` | `string` | Optional | - |\n| `label_data_pdf` | `string` | Optional | - |\n| `export_docs_pdf` | `string` | Optional | - |\n| `carrier` | `string` | Optional | - |\n| `shipping_date` | `datetime` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "OrderId": null,\n  "OrderReference": null,\n  "ShippingId": null,\n  "TrackingUrl": null,\n  "LabelDataPdf": null,\n  "ExportDocsPdf": null,\n  "Carrier": null,\n  "ShippingDate": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result System Collections Generic List Billbee Interfaces Billbee API Model Shipment\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelShipment`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `paging` | [`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelShipment`](#rechnungsdruck-web-app-controllers-api-api-paged-result-paging-information-system-collections-generic-list-billbee-interfaces-billbee-api-model-shipment) | Optional | - |\n| `error_message` | `string` | Optional | - |\n| `error_code` | [`ErrorCodeEnum`](#error-code) | Optional | - |\n| `data` | [`List of BillbeeInterfacesBillbeeAPIModelShipment`](#billbee-interfaces-billbee-api-model-shipment) | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Paging": null,\n  "ErrorMessage": null,\n  "ErrorCode": null,\n  "Data": null\n}\n```\n\n#### Rechnungsdruck Web App Controllers Api Api Paged Result Paging Information System Collections Generic List Billbee Interfaces Billbee API Model Shipment\n\n##### Class Name\n\n`RechnungsdruckWebAppControllersApiApiPagedResultPagingInformationSystemCollectionsGenericListBillbeeInterfacesBillbeeAPIModelShipment`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `page` | `int` | Optional | - |\n| `total_pages` | `int` | Optional | - |\n| `total_rows` | `int` | Optional | - |\n| `page_size` | `int` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Page": null,\n  "TotalPages": null,\n  "TotalRows": null,\n  "PageSize": null\n}\n```\n\n#### Microsoft Asp Net Web Hooks Web Hook\n\n##### Class Name\n\n`MicrosoftAspNetWebHooksWebHook`\n\n##### Fields\n\n| Name | Type | Tags | Description |\n|  --- | --- | --- | --- |\n| `id` | `string` | Optional | - |\n| `web_hook_uri` | `string` | Required | - |\n| `secret` | `string` | Optional | - |\n| `description` | `string` | Optional | - |\n| `is_paused` | `bool` | Optional | - |\n| `filters` | `List of string` | Optional | - |\n| `headers` | `dict` | Optional | - |\n| `properties` | `dict` | Optional | - |\n\n##### Example (as JSON)\n\n```json\n{\n  "Id": null,\n  "WebHookUri": "WebHookUri0",\n  "Secret": null,\n  "Description": null,\n  "IsPaused": null,\n  "Filters": null,\n  "Headers": null,\n  "Properties": null\n}\n```\n\n### Enumerations\n\n* [Address Type](#address-type)\n* [Article Title Source](#article-title-source)\n* [Default Vat Mode](#default-vat-mode)\n* [Error Code](#error-code)\n* [New State Id](#new-state-id)\n* [Payment Method](#payment-method)\n* [Search Mode](#search-mode)\n* [Send Mode](#send-mode)\n* [Shipping Carrier](#shipping-carrier)\n* [State](#state)\n* [Type](#type)\n* [Type 1](#type-1)\n* [Vat Mode](#vat-mode)\n\n#### Address Type\n\nThe type of the address\n\n##### Class Name\n\n`AddressTypeEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_1` |\n| `ENUM_2` |\n\n#### Article Title Source\n\n##### Class Name\n\n`ArticleTitleSourceEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n\n#### Default Vat Mode\n\nOptionally specify the default vat mode of the user\n\n##### Class Name\n\n`DefaultVatModeEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n| `ENUM_5` |\n\n#### Error Code\n\n##### Class Name\n\n`ErrorCodeEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n| `ENUM_5` |\n| `ENUM_6` |\n| `ENUM_7` |\n| `ENUM_8` |\n| `ENUM_9` |\n| `ENUM_10` |\n| `ENUM_11` |\n| `ENUM_12` |\n| `ENUM_13` |\n| `ENUM_14` |\n| `ENUM_15` |\n| `ENUM_16` |\n| `ENUM_17` |\n| `ENUM_18` |\n| `ENUM_19` |\n| `ENUM_20` |\n| `ENUM_21` |\n| `ENUM_22` |\n| `ENUM_23` |\n| `ENUM_24` |\n| `ENUM_25` |\n| `ENUM_26` |\n| `ENUM_27` |\n\n#### New State Id\n\nThe new state to set\n\n##### Class Name\n\n`NewStateIdEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n| `ENUM_5` |\n| `ENUM_6` |\n| `ENUM_7` |\n| `ENUM_8` |\n| `ENUM_9` |\n| `ENUM_10` |\n| `ENUM_11` |\n| `ENUM_12` |\n| `ENUM_13` |\n| `ENUM_14` |\n| `ENUM_15` |\n| `ENUM_16` |\n\n#### Payment Method\n\nThe payment method\n\n##### Class Name\n\n`PaymentMethodEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n| `ENUM_6` |\n| `ENUM_19` |\n| `ENUM_20` |\n| `ENUM_21` |\n| `ENUM_22` |\n| `ENUM_23` |\n| `ENUM_24` |\n| `ENUM_25` |\n| `ENUM_26` |\n| `ENUM_27` |\n| `ENUM_28` |\n| `ENUM_29` |\n| `ENUM_30` |\n| `ENUM_31` |\n| `ENUM_32` |\n| `ENUM_33` |\n| `ENUM_34` |\n| `ENUM_35` |\n| `ENUM_36` |\n| `ENUM_37` |\n| `ENUM_38` |\n| `ENUM_39` |\n| `ENUM_40` |\n| `ENUM_41` |\n| `ENUM_42` |\n| `ENUM_43` |\n| `ENUM_44` |\n| `ENUM_45` |\n| `ENUM_46` |\n| `ENUM_47` |\n| `ENUM_48` |\n| `ENUM_49` |\n| `ENUM_50` |\n| `ENUM_51` |\n| `ENUM_52` |\n| `ENUM_53` |\n| `ENUM_54` |\n| `ENUM_55` |\n| `ENUM_56` |\n| `ENUM_57` |\n| `ENUM_58` |\n| `ENUM_59` |\n| `ENUM_60` |\n| `ENUM_61` |\n| `ENUM_62` |\n| `ENUM_63` |\n| `ENUM_64` |\n| `ENUM_65` |\n| `ENUM_66` |\n| `ENUM_67` |\n| `ENUM_68` |\n| `ENUM_69` |\n| `ENUM_70` |\n| `ENUM_71` |\n| `ENUM_72` |\n| `ENUM_73` |\n| `ENUM_74` |\n| `ENUM_75` |\n| `ENUM_76` |\n| `ENUM_77` |\n| `ENUM_78` |\n| `ENUM_79` |\n| `ENUM_80` |\n| `ENUM_81` |\n| `ENUM_82` |\n| `ENUM_83` |\n| `ENUM_84` |\n| `ENUM_85` |\n| `ENUM_86` |\n| `ENUM_87` |\n| `ENUM_88` |\n| `ENUM_89` |\n| `ENUM_90` |\n| `ENUM_91` |\n| `ENUM_92` |\n| `ENUM_93` |\n| `ENUM_94` |\n| `ENUM_95` |\n| `ENUM_96` |\n| `ENUM_97` |\n| `ENUM_98` |\n| `ENUM_99` |\n| `ENUM_100` |\n| `ENUM_101` |\n| `ENUM_102` |\n| `ENUM_103` |\n| `ENUM_104` |\n| `ENUM_105` |\n| `ENUM_106` |\n| `ENUM_107` |\n| `ENUM_108` |\n| `ENUM_109` |\n| `ENUM_110` |\n| `ENUM_111` |\n| `ENUM_112` |\n\n#### Search Mode\n\n##### Class Name\n\n`SearchModeEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n\n#### Send Mode\n\n##### Class Name\n\n`SendModeEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n\n#### Shipping Carrier\n\n##### Class Name\n\n`ShippingCarrierEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n| `ENUM_5` |\n| `ENUM_6` |\n| `ENUM_7` |\n| `ENUM_8` |\n| `ENUM_9` |\n| `ENUM_10` |\n| `ENUM_11` |\n| `ENUM_12` |\n| `ENUM_13` |\n| `ENUM_14` |\n| `ENUM_15` |\n| `ENUM_16` |\n| `ENUM_17` |\n\n#### State\n\nThe current state of the order\n\n##### Class Name\n\n`StateEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n| `ENUM_5` |\n| `ENUM_6` |\n| `ENUM_7` |\n| `ENUM_8` |\n| `ENUM_9` |\n| `ENUM_10` |\n| `ENUM_11` |\n| `ENUM_12` |\n| `ENUM_13` |\n| `ENUM_14` |\n| `ENUM_15` |\n| `ENUM_16` |\n\n#### Type\n\n##### Class Name\n\n`TypeEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n\n#### Type 1\n\n##### Class Name\n\n`Type1Enum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n| `ENUM_5` |\n\n#### Vat Mode\n\nThe vat mode of the order\n\n##### Class Name\n\n`VatModeEnum`\n\n##### Fields\n\n| Name |\n|  --- |\n| `ENUM_0` |\n| `ENUM_1` |\n| `ENUM_2` |\n| `ENUM_3` |\n| `ENUM_4` |\n| `ENUM_5` |\n\n## Utility Classes Documentation\n\n### ApiHelper\n\nA utility class for processing API Calls. Also contains classes for supporting standard datetime formats.\n\n#### Methods\n\n| Name | Description |\n|  --- | --- |\n| json_deserialize | Deserializes a JSON string to a Python dictionary. |\n\n#### Classes\n\n| Name | Description |\n|  --- | --- |\n| HttpDateTime | A wrapper for datetime to support HTTP date format. |\n| UnixDateTime | A wrapper for datetime to support Unix date format. |\n| RFC3339DateTime | A wrapper for datetime to support RFC3339 format. |\n\n## Common Code Documentation\n\n### HttpResponse\n\nHttp response received.\n\n#### Parameters\n\n| Name | Type | Description |\n|  --- | --- | --- |\n| status_code | int | The status code returned by the server. |\n| reason_phrase | str | The reason phrase returned by the server. |\n| headers | dict | Response headers. |\n| text | str | Response body. |\n| request | HttpRequest | The request that resulted in this response. |\n\n### HttpRequest\n\nRepresents a single Http Request.\n\n#### Parameters\n\n| Name | Type | Tag | Description |\n|  --- | --- | --- | --- |\n| http_method | HttpMethodEnum |  | The HTTP method of the request. |\n| query_url | str |  | The endpoint URL for the API request. |\n| headers | dict | optional | Request headers. |\n| query_parameters | dict | optional | Query parameters to add in the URL. |\n| parameters | dict &#124; str | optional | Request body, either as a serialized string or else a list of parameters to form encode. |\n| files | dict | optional | Files to be sent with the request. |\n\n',
    'author': 'MrP01',
    'author_email': 'peter@waldert.at',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://www.billbee.io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
