# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['jsoncomparison']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'jsoncomparison',
    'version': '1.1.0',
    'description': 'json compare utility',
    'long_description': '## The JSON Comparison package\n\n[![Build Status](https://travis-ci.com/rugleb/JsonCompare.svg?branch=master)](https://travis-ci.com/rugleb/JsonCompare)\n[![codecov](https://codecov.io/gh/rugleb/JsonCompare/branch/master/graph/badge.svg)](https://codecov.io/gh/rugleb/JsonCompare)\n[![Python 3.6+](https://img.shields.io/badge/python-3.6+-green.svg)](https://www.python.org/downloads/release/python-360/)\n[![PyPI version](https://badge.fury.io/py/jsoncomparison.svg)](https://badge.fury.io/py/jsoncomparison)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n\nThis package is designed to compare two objects with a JSON-like structure and data types.\n\n### Install\n\n```\npip install -U pip jsoncomparison\n```\n\n### Usage\n\nFirst you need to define two variables: `expected` & `actual`.\nThink of them as the same variables that you use in tests.\n\nExpected - the original data object that you want to see.\nActual - the given data object.\n\nThen we will transfer these objects to check and identify the difference between them:\n\n```python\nfrom jsoncomparison import Compare, NO_DIFF\n\n\nexpected = {\n    "project": {\n        "name": "jsoncomparison",\n        "version": "0.1",\n        "license": "MIT",\n        "language": {\n            "name": "python",\n            "versions": [\n                3.5,\n                3.6\n            ]\n        }\n    },\n    "os": "linux"\n}\n\nactual = {\n    "project": {\n        "name": "jsoncomparison",\n        "version": 0.1,\n        "license": "Apache 2.0",\n        "language": {\n            "name": "python",\n            "versions": [\n                3.6\n            ]\n        }\n    }\n}\n\ndiff = Compare().check(expected, actual)\nassert diff != NO_DIFF\n```\n\nThe `check` method returns a dictionary of differences between `expected` and `actual` objects:\n\n```json\n{\n    "project": {\n        "version": {\n            "_message": "Types not equal. Expected: <str>, received: <float>",\n            "_expected": "str",\n            "_received": "float"\n        },\n        "license": {\n            "_message": "Values not equal. Expected: <MIT>, received: <Apache 2.0>",\n            "_expected": "MIT",\n            "_received": "Apache 2.0"\n        },\n        "language": {\n            "versions": {\n                "_length": {\n                    "_message": "Lengths not equal. Expected <2>, received: <1>",\n                    "_expected": 2,\n                    "_received": 1\n                },\n                "_content": {\n                    "0": {\n                        "_message": "Value not found. Expected <3.5>",\n                        "_expected": 3.5,\n                        "_received": null\n                    }\n                }\n            }\n        }\n    },\n    "os": {\n        "_message": "Key does not exists. Expected: <os>",\n        "_expected": "os",\n        "_received": null\n    }\n}\n```\n\n### Configuration\n\nThe default configuration can be overridden by passing the config dictionary to the Compare class constructor:\n\n```python\nfrom jsoncomparison import Compare\n\nconfig = {\n    "output": {\n        "console": False,\n        "file": {\n            "allow_nan": True,\n            "ensure_ascii": True,\n            "indent": 4,\n            "name": None,\n            "skipkeys": True,\n        },\n    },\n    "types": {\n        "float": {\n            "allow_round": 2,\n        },\n        "list": {\n            "check_length": True,\n        }\n    }\n}\n\ncmp = Compare(config)\n```\n\n### Output\n\nBy default, the configuration does not allow printing the comparison result to the console,\nbut at the same time writes the results to a file.\n\n\nThese settings can be changed in your class config:\n\n```py\nconfig = {\n    "output": {\n        "console": True,\n        "file": {}\n    }\n}\n```\n\n### Ignore rules\n\nWhat if you do not want to compare some values and keys of objects from your JSON?  \nIn this case, you can define exception rules and pass them to the class constructor.\n\nLet\'s go back to the example above:\n\n```python\nfrom jsoncomparison import Compare, NO_DIFF\n\n\nexpected = {\n    # ...\n}\n\nactual = {\n    # ...\n}\n\nrules = {\n    "project": {\n        "version": "*",\n        "license": "*",\n        "language": {\n            "versions": {\n                "_values": [\n                    3.5\n                ]\n            }\n        }\n    },\n    "os": "*",\n}\n\ndiff = Compare(rules=rules).check(expected, actual)\nassert diff == NO_DIFF\n```\n\nNow that we have added exceptions to the missing values,\nthe comparison test has been successfully passed!\n\n### Links\n\nYou can see a more complex comparison example that I used to test the correct operation of an application:\n[link](https://github.com/rugleb/jsoncomparison/blob/master/tests/test_compare.py#L84).\n',
    'author': 'Gleb Karpushkin',
    'author_email': 'rugleb@gmail.com',
    'maintainer': 'Gleb Karpushkin',
    'maintainer_email': 'rugleb@gmail.com',
    'url': 'https://pypi.org/project/jsoncomparison',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
