# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['paperoni']

package_data = \
{'': ['*'], 'paperoni': ['commands/*']}

install_requires = \
['beautifulsoup4>=4.9.1,<5.0.0',
 'blessed>=1.17.9,<2.0.0',
 'coleo>=0.1.7,<0.2.0',
 'hrepr>=0.2.4,<0.3.0',
 'requests>=2.24.0,<3.0.0',
 'tqdm>=4.48.2,<5.0.0']

entry_points = \
{'console_scripts': ['paperoni = paperoni.__main__:main'],
 'paperoni.command': ['bibtex = '
                      'paperoni.commands.command_bibtex:command_bibtex',
                      'collect = '
                      'paperoni.commands.command_collect:command_collect',
                      'config = '
                      'paperoni.commands.command_config:command_config',
                      'html = paperoni.commands.command_html:command_html',
                      'researcher = '
                      'paperoni.commands.command_researcher:command_researcher',
                      'search = '
                      'paperoni.commands.command_search:command_search']}

setup_kwargs = {
    'name': 'paperoni',
    'version': '0.1.4',
    'description': 'Search for scientific papers',
    'long_description': '\n# Paperoni\n\n* **Search** for scientific papers on the command line\n* **Download PDFs**\n* Generate **BibTeX** entries\n* Generate **HTML** for your publications!\n* Build **collections** of papers\n\n<!-- <img src="./media/screenshot.png"> -->\n\n![Screenshot](https://github.com/mila-iqia/paperoni/raw/master/media/screenshot.png)\n\nPaperoni uses the Microsoft Academic Knowledge API, which requires an API key:\n\n**[Get a Microsoft Academic Knowledge API key](https://msr-apis.portal.azure-api.net/products/project-academic-knowledge)** (free tier: 10,000 queries per month)\n\n\n## Install\n\n```bash\npip install paperoni\n\n# This will prompt for the API key\npaperoni config\n```\n\n## Overview\n\nThis is a curated set of examples of things you can do with Paperoni. Most of the flags shown below can be combined.\n\n```bash\n# Search for papers from an author\npaperoni search -a alan turing\n\n# Search for papers with a certain title\npaperoni search -t the chemical basis of morphogenesis\n\n# Search for the most cited AI papers of 2017\npaperoni search -k artificial_intelligence -y 2017 --cited\n\n# Collect papers into a file\npaperoni collect -c my-papers.json -a olivier breuleux -y 2018\n\n# Dump BibTeX for all papers in a collection\npaperoni bibtex -c my-papers.json >> papers.bib\n\n# Output a webpage\npaperoni html -c my-papers.json\n\n# Collect info about a researcher into a file (interactive)\npaperoni researcher -r researchers.json -a olivier breuleux\n\n# Search for papers from researchers with status "phd"\npaperoni search -r researchers.json --status phd\n```\n\n## Search\n\nThe `paperoni search` command allows you to search for papers:\n\n```\n$ paperoni search -h\nusage: paperoni search [-h] [--author [VALUE [VALUE ...]]] [--cited]\n                       [--collection VALUE] [--command VALUE] [--end VALUE]\n                       [--group] [--institution [VALUE [VALUE ...]]]\n                       [--key VALUE] [--keywords [VALUE [VALUE ...]]]\n                       [--limit NUM] [--offset NUM] [--recent]\n                       [--researchers VALUE] [--start VALUE]\n                       [--status [VALUE [VALUE ...]]] [--symposium]\n                       [--no-symposium] [--title [VALUE [VALUE ...]]]\n                       [--venue VALUE] [--verbose]\n                       [--words [VALUE [VALUE ...]]] [--workshop]\n                       [--no-workshop] [--year NUM]\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --collection VALUE, -c VALUE\n                        File containing the collection\n  --command VALUE       Command to run on every paper\n  --group               Group multiple versions of the same paper\n  --key VALUE           Microsoft Cognitive API key\n  --researchers VALUE, -r VALUE\n                        Researchers file (JSON)\n  --verbose, -v         Verbose output\n\nsearch:\n  --author [VALUE [VALUE ...]], -a [VALUE [VALUE ...]]\n                        Search for an author\n  --cited               Sort by most cited\n  --end VALUE           End date (yyyy-mm-dd or yyyy)\n  --institution [VALUE [VALUE ...]], -i [VALUE [VALUE ...]]\n                        Search papers from institution\n  --keywords [VALUE [VALUE ...]], -k [VALUE [VALUE ...]]\n                        Search for keywords\n  --limit NUM           Number of papers to fetch (default: 100)\n  --offset NUM          Search offset\n  --recent              Sort by most recent\n  --start VALUE         Start date (yyyy-mm-dd or yyyy)\n  --status [VALUE [VALUE ...]]\n                        Researcher status(es) to filter for\n  --symposium           List symposiums\n  --no-symposium        Do not list symposiums\n  --title [VALUE [VALUE ...]], -t [VALUE [VALUE ...]]\n                        Search words in the title\n  --venue VALUE         Search papers from a specific conference or journal\n  --words [VALUE [VALUE ...]], -w [VALUE [VALUE ...]]\n                        Search words in the title or abstract\n  --workshop            List workshops\n  --no-workshop         Do not list workshops\n  --year NUM, -y NUM    Year\n```\n\nThe interface will list each result interactively, allowing you to perform actions:\n\n* `l` to show more information about the paper: abstract, affiliations, all links\n* `b` to print out a BibTeX entry for the paper (see also `paperoni bibtex`)\n* `p` to save the PDF in the current directory, if a PDF is available (and doesn\'t require authentication or captchas)\n\n## Generate BibTeX\n\nWith `paperoni bibtex` you can generate bibtex entries from a search or a collection. Each entry will have a reference name generated from the first author, year, longest word in the title and a small hash number.\n\n## Generate HTML\n\nWith `paperoni html` you can generate HTML from a search or a collection.\n\n* Use `--template my-template.html` to use the specified file as a template. The publications will be inserted in the element with id `paperoni-papers`, e.g. `<div id="paperoni-papers">PAPERS GO HERE</div>`. You can also specify a different id using the following syntax: `--template my-template.html#mypapers`.\n* Use `--template none` if you don\'t want to use a template at all an only want the raw HTML.\n* Use `--inject file.html` to insert the papers into the elemnet with id `paperoni-papers` *directly* into `file.html`. **This will modify the file.** Any previous contents of that div will be erased. For safety, `paperoni` will create a backup file, with a `.bk` extension, unless you pass `--no-backup`.\n\n`paperoni html` includes *the full search interface*. You don\'t need to pass a collection if you want to search directly on the web.\n\nIf you have a researchers file, you can pass it with `-r` and paperoni can generate bio links for any researchers in the set.\n\nYou can see all the options with `paperoni html -h`.\n\n## Collections\n\nIt is possible to save papers into collections using:\n\n```bash\n# Assuming you want your collection to be in my-collection.json\npaperoni collect -c my-collection.json\n```\n\nThe options are the same as `search`, but you can sort through the search results to add papers to the collection or reject them. Papers that were already added or rejected will be ignored, so the collection can be built incrementally by reusing the same search string and going through any new papers.\n\n`paperoni search -c my-collection.json` will search for papers in the collection. The search algorithm may be slightly different since it is a purely local search.\n\n## Researchers database\n\nFor more advanced uses you can create a researchers file that contains one or more people.\n\nThe purpose of `paperoni researcher` is to properly identify authors, better than simply searching by name, because an individual may have several homonyms. Multiple authors can also be grouped so that you can search papers from all of them, so this can be useful to collect all of your organization\'s publications: simply register all of your researchers. You can even log their start/end dates, which will only collect publications from that time period.\n\n`paperoni researcher -r researchers.json -a author name` will guide you interactively.\n\n1. **`Find ids`**: You will be asked whether certain papers are from the author or not, to weed out homonyms.\n2. **`Set a property`**: You can set arbitrary properties for the researcher. Note that `paperoni html` recognizes the `bio` property. Erase a property by entering `null`.\n3. **`Add a role`**: You can optionally assign one or more "roles". A "role" is some arbitrary tag with optional start and end dates that can be assigned to a researcher.\n\nThen you can write for example `paperoni collect -c org.json -r researchers.json --status xyz` to collect papers from researchers when they had a given status. The `-r` flag is also compatible with `paperoni search`.\n\n## FAQ\n\n**I cannot find a paper.**\n\nPaperoni uses [Microsoft Academic](https://academic.microsoft.com/home) to find papers. First check if you can find the paper there. If it is indeed there, this is a bug with Paperoni and you should file an issue.\n\nIf it isn\'t, the most likely reason is that the paper is too recent and has not yet been indexed. It can sometimes take a few weeks before a paper is indexed.\n\n**The PDF download does not work.**\n\nTry the `l` command, which will list all links pertaining to the paper, organized by type. Try opening them in your browser, it might work better.\n\n**Can I manually enter a new paper into a collection?**\n\nAssuming you mean a paper that is not indexed in the database, the answer, unfortunately, is no. \n\n<!-- Feel free to request this feature by voting on issue [TODO: issue]. -->\n\n\n**Can I remove a paper from a collection?**\n\nYes, search for the paper(s) to remove, passing the collection using the `-c` argument, and use the `r` interactive command to remove it.\n\n```\n$ paperoni search -c collection.json -t the paper title\n================================================================================\nTitle: The paper title\nAuthors: Some Guy\nDate: 2020-02-02\n================================================================================\nEnter a command (h or ? for help): b/p/[s]/r/q/l/h r\nRemoved \'The paper title\' from the collection.\n```\n\nYou can use `--command r` to do this non-interactively.\n\n\n## Programmatic API\n\n**The API is very beta and unstable. It is liable to change arbitrarily.**\n\nIf you want to perform some custom operations like generating HTML exactly the way you want it, write some reference format other than BibTeX, or whatever else, here\'s some code to get you started. The following will search for papers by Alan Turing and will print out the titles and abstracts:\n\n\n```python\nimport coleo\n\n# You need these to wrap collection or researchers, if you want to provide\n# them outside of the command line.\nfrom paperoni.io import PapersFile, ResearchersFile\n\n# There is also search(), the difference is that search() does not define\n# CLI arguments for collection and researchers but takes them as inputs\n# instead\nfrom paperoni.commands.searchutils import search_ext\n\ndef main():\n    papers = search_ext()\n    for paper in papers:\n        print(paper.title)\n        print(paper.abstract)\n        print("====")\n\nif __name__ == "__main__":\n    with coleo.setvars(\n        author="alan turing",\n        # collection=PapersFile("alan.json"),\n        # researchers=ResearchersFile("rsch.json"),\n    ):\n        coleo.auto_cli(main, print_result=False)\n```\n\n* `coleo.auto_cli` will expose all the search flags like `--title` and whatnot that are defined inside `search_ext`, so you actually get all that for free.\n* `coleo.setvars` lets you set any of the options programmatically, but some of them, like `collection` or `researchers`, you will need to wrap yourself (see the commented lines).\n* The API for the Paper object is kind of bad and in flux so I\'m not going to document it right now, but if you dump `paper.data` you can see all the raw data and work from there.\n\nCheck out [coleo](https://github.com/breuleux/coleo) if you want to define extra command line arguments in `main()`, it\'s quite easy.\n\nFuture versions of `paperoni` might break the API, so make sure to pin the version you\'re using.\n\n## Plugins\n\n**The API is very beta and unstable. It is liable to change arbitrarily.**\n\nYou can add new commands to paperoni by registering them in the `paperoni.command` entry point. Command line options must be defined using [coleo](https://github.com/breuleux/coleo). If you are using [poetry](https://python-poetry.org/):\n\n**pyproject.toml**\n\n```toml\n[tool.poetry.plugins."paperoni.command"]\nshowprop = "my_paperoni:showprop"\n```\n\n**my_paperoni/\\_\\_init__.py**\n\n```python\nfrom coleo import Argument, default, tooled\nfrom paperoni.commands.searchutils import search\n\n\n@tooled\ndef showprop():\n\n    # Name of the property to display\n    # [alias: -p]\n    prop: Argument & str = default("title")\n\n    # This will add all the search options\n    papers = search()\n\n    for paper in papers:\n        if prop == "title":\n            print(paper.title)\n        elif prop == "venue":\n            print(paper.venue)\n        ...\n```\n\nInstall the plugin:\n\n```bash\n# If the plugin is accessible through pip\npip install my_paperoni\n\n# If this is a local project:\npoetry install\n```\n\nUse the plugin:\n\n```bash\npaperoni showprop -p venue -a alan turing --limit 10\n```\n\nFuture versions of `paperoni` might break the API, so make sure to pin the version you\'re using.\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mila-iqia/paperoni',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
