"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const aws_secret_helper_1 = require("../aws-secret-helper");
const github_helper_1 = require("../github-helper");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManager({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.getSecretValue({ SecretId: event.ResourceProperties.githubTokenSecret });
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    console.log('Create new ActionEnvironmentSecret with props ' + JSON.stringify(event.ResourceProperties));
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log(`Update ActionEnvironmentSecret ${props.repositorySecretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onDelete = async (event, octokit) => {
    console.log('Delete ActionEnvironmentSecret ' + event.ResourceProperties.repositorySecretName);
    await deleteEnvironmentSecret(event, octokit);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const createOrUpdateEnvironmentSecret = async (event, octokit, smClient) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name, environment: environment_name, sourceSecretArn: secretId, sourceSecretJsonField, } = event.ResourceProperties;
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = await aws_secret_helper_1.getSecretString(secretId, smClient, sourceSecretJsonField);
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const repository_id = await getRepositoryId(event, octokit, owner);
    const secretResponse = await octokit.request('PUT /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteEnvironmentSecret = async (event, octokit) => {
    const { environment: environment_name, repositorySecretName: secret_name, repositoryOwner } = event.ResourceProperties;
    const repository_id = await getRepositoryId(event, octokit, repositoryOwner);
    const deleteSecretResponse = await octokit.request('DELETE /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const getRepositoryId = async (event, octokit, repositoryOwner) => {
    const { repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}', {
        owner,
        repo,
    });
    return data.id;
};
const buildPhysicalResourceId = async (event, octokit) => {
    const { environment, repositorySecretName: secret, repositoryOwner, repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    return [environment, secret, owner, repo].join('/');
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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