"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const client_ssm_1 = require("@aws-sdk/client-ssm");
const core_1 = require("@octokit/core");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    const clientConfig = { region: event.ResourceProperties.awsRegion };
    const ssmClient = new client_ssm_1.SSM(clientConfig);
    const smClient = new client_secrets_manager_1.SecretsManager(clientConfig);
    const githubTokenSecret = await smClient.getSecretValue(({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, ssmClient);
        case 'Update':
            return onUpdate(event, octokit, ssmClient);
        case 'Delete':
            return onDelete(event, octokit, ssmClient);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, ssmClient) => {
    const { createRequestEndpoint, createRequestPayload, responseBodySSMParameterName, } = event.ResourceProperties;
    const requestResult = await executeRequest(octokit, createRequestEndpoint, createRequestPayload);
    console.log('Created GitHub Resource');
    const PhysicalResourceId = await extractIdFromRequestResult(event, requestResult);
    await writeResponseBodyToSSMParameterIfNeeded(ssmClient, requestResult.data, responseBodySSMParameterName);
    return { PhysicalResourceId };
};
const onUpdate = async (event, octokit, ssmClient) => {
    const { updateRequestEndpoint, updateRequestPayload, createRequestResultParameter, responseBodySSMParameterName, } = event.ResourceProperties;
    const PhysicalResourceId = event.PhysicalResourceId;
    if (!updateRequestEndpoint) {
        console.log("No update request endpoint specified, so we'll just ignore the update request.");
        return { PhysicalResourceId };
    }
    const updateEndpoint = replaceResultParamterInEndpoint(updateRequestEndpoint, createRequestResultParameter, PhysicalResourceId);
    const requestResult = await executeRequest(octokit, updateEndpoint, updateRequestPayload);
    console.log(`Updated GitHub Resource with id '${event.PhysicalResourceId}'`);
    await writeResponseBodyToSSMParameterIfNeeded(ssmClient, requestResult.data, responseBodySSMParameterName);
    return { PhysicalResourceId };
};
const onDelete = async (event, octokit, ssmClient) => {
    const { deleteRequestEndpoint, deleteRequestPayload, createRequestResultParameter, responseBodySSMParameterName, } = event.ResourceProperties;
    const PhysicalResourceId = event.PhysicalResourceId;
    const deleteEndpoint = replaceResultParamterInEndpoint(deleteRequestEndpoint, createRequestResultParameter, PhysicalResourceId);
    const requestResult = await executeRequest(octokit, deleteEndpoint, deleteRequestPayload);
    const responseBody = requestResult.data;
    console.log(`Deleted GitHub Resource with id '${event.PhysicalResourceId}'`);
    await writeResponseBodyToSSMParameterIfNeeded(ssmClient, responseBody, responseBodySSMParameterName);
    return { PhysicalResourceId };
};
const executeRequest = async (octokit, endpoint, payload) => {
    const payloadToSend = payload ? JSON.parse(payload) : undefined;
    console.log(`Making a request to '${endpoint}'.`);
    const requestResult = await octokit.request(endpoint, payloadToSend);
    return requestResult;
};
const replaceResultParamterInEndpoint = (endpoint, createRequestResultParameter, createRequestResultValue) => createRequestResultParameter && createRequestResultValue ? endpoint.replace(':' + createRequestResultParameter, createRequestResultValue) : endpoint;
const toStringIfNotUndefined = (value) => value !== undefined ? String(value) : undefined;
const writeResponseBodyToSSMParameterIfNeeded = async (ssmClient, responseBody, parameterName) => {
    if (parameterName) {
        console.log(`Write the response body to the ssm parameter '${parameterName}'`);
        await ssmClient.putParameter({ Name: parameterName, Value: JSON.stringify(responseBody), Overwrite: true });
    }
};
const extractIdFromRequestResult = async (event, requestResult) => {
    const { createRequestResultParameter } = event.ResourceProperties;
    console.log('Going to extract the parameter from the create result (if needed)');
    const createRequestResultOriginalValue = createRequestResultParameter ?
        requestResult.data[createRequestResultParameter] : undefined;
    const createRequestResultValue = toStringIfNotUndefined(createRequestResultOriginalValue);
    if (!createRequestResultValue && createRequestResultParameter) {
        throw new Error("Could not find '" + createRequestResultParameter + "' in request result: " + JSON.stringify(requestResult));
    }
    return createRequestResultValue;
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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