/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as kms from '@aws-cdk/aws-kms';
import * as sqs from '@aws-cdk/aws-sqs';
import * as s3 from '@aws-cdk/aws-s3';
import { Construct } from '@aws-cdk/core';
/**
 * @summary The properties for the S3ToSqs class.
 */
export interface S3ToSqsProps {
    /**
     * Existing instance of S3 Bucket object, providing both this and `bucketProps` will cause an error.
     *
     * @default - None
     */
    readonly existingBucketObj?: s3.Bucket;
    /**
     * User provided props to override the default props for the S3 Bucket.
     *
     * @default - Default props are used
     */
    readonly bucketProps?: s3.BucketProps;
    /**
     * The S3 event types that will trigger the notification.
     *
     * @default - If not specified the s3.EventType.OBJECT_CREATED event will trigger the notification.
     */
    readonly s3EventTypes?: s3.EventType[];
    /**
     * S3 object key filter rules to determine which objects trigger this event.
     *
     * @default - If not specified no filter rules will be applied.
     */
    readonly s3EventFilters?: s3.NotificationKeyFilter[];
    /**
     * Existing instance of SQS queue object, Providing both this and queueProps will cause an error.
     *
     * @default - Default props are used
     */
    readonly existingQueueObj?: sqs.Queue;
    /**
     * Optional user provided properties.
     *
     * @default - Default props are used
     */
    readonly queueProps?: sqs.QueueProps;
    /**
     * Optional user provided properties for the dead letter queue.
     *
     * @default - Default props are used
     */
    readonly deadLetterQueueProps?: sqs.QueueProps;
    /**
     * Whether to deploy a secondary queue to be used as a dead letter queue.
     *
     * @default - true.
     */
    readonly deployDeadLetterQueue?: boolean;
    /**
     * The number of times a message can be unsuccessfully dequeued before being moved to the dead-letter queue.
     *
     * @default - required field if deployDeadLetterQueue=true.
     */
    readonly maxReceiveCount?: number;
    /**
     * Use a KMS Key, either managed by this CDK app, or imported.
     *
     * If importing an encryption key, it must be specified in
     * the encryptionKey property for this construct.
     *
     * @default - true (encryption enabled, managed by this CDK app).
     */
    readonly enableEncryptionWithCustomerManagedKey?: boolean;
    /**
     * An optional, imported encryption key to encrypt the SQS queue.
     *
     * @default - not specified.
     */
    readonly encryptionKey?: kms.Key;
    /**
     * Optional user-provided props to override the default props for the encryption key.
     *
     * @default - Default props are used.
     */
    readonly encryptionKeyProps?: kms.KeyProps;
}
/**
 * @summary The S3ToSqs class.
 */
export declare class S3ToSqs extends Construct {
    readonly sqsQueue: sqs.Queue;
    readonly deadLetterQueue?: sqs.DeadLetterQueue;
    readonly s3Bucket?: s3.Bucket;
    readonly s3LoggingBucket?: s3.Bucket;
    readonly encryptionKey?: kms.IKey;
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the S3ToSqs class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope: Construct, id: string, props: S3ToSqsProps);
}
