"""
Comprehensive tests for RedisStackMemoryLogger with 100% coverage.
"""

import threading
import time
from unittest.mock import AsyncMock, Mock, patch

import numpy as np
import pytest

from orka.memory.redisstack_logger import RedisStackMemoryLogger


class TestRedisStackLoggerInitialization:
    """Test RedisStackMemoryLogger initialization scenarios."""

    @patch("orka.memory.redisstack_logger.redis.Redis")
    @patch("orka.utils.bootstrap_memory_index.ensure_enhanced_memory_index")
    @patch.object(RedisStackMemoryLogger, "_create_redis_connection")
    def test_default_initialization(self, mock_create_connection, mock_ensure_index, mock_redis):
        """Test initialization with default parameters."""
        mock_ensure_index.return_value = True
        mock_create_connection.return_value = Mock()

        logger = RedisStackMemoryLogger()

        assert logger.redis_url == "redis://localhost:6379/0"  # Updated to use port 6379
        assert logger.index_name == "orka_enhanced_memory"
        assert logger.embedder is None
        assert logger.memory_decay_config is None

        # Check that the external ensure function was called with correct parameters
        mock_ensure_index.assert_called_once()
        call_args = mock_ensure_index.call_args
        assert call_args[1]["vector_dim"] == 384  # Default dimension
        assert call_args[1]["index_name"] == "orka_enhanced_memory"

    @patch("orka.memory.redisstack_logger.redis.Redis")
    @patch.object(RedisStackMemoryLogger, "_ensure_index")
    @patch.object(RedisStackMemoryLogger, "_create_redis_connection")
    def test_custom_initialization(self, mock_create_connection, mock_ensure, mock_redis):
        """Test initialization with custom parameters."""
        mock_create_connection.return_value = Mock()

        custom_config = {"short_term": 1.0, "long_term": 24.0}

        logger = RedisStackMemoryLogger(
            redis_url="redis://custom:6379/1",
            index_name="custom_index",
            memory_decay_config=custom_config,
        )

        assert logger.redis_url == "redis://custom:6379/1"
        assert logger.index_name == "custom_index"
        assert logger.memory_decay_config == custom_config
        mock_ensure.assert_called_once()

    @patch("orka.memory.redisstack_logger.redis.Redis")
    @patch.object(RedisStackMemoryLogger, "_ensure_index")
    @patch.object(RedisStackMemoryLogger, "_create_redis_connection")
    def test_redis_connection_failure(self, mock_create_connection, mock_ensure, mock_redis):
        """Test handling of Redis connection failure."""
        mock_create_connection.side_effect = Exception("Connection failed")

        with pytest.raises(Exception, match="Connection failed"):
            RedisStackMemoryLogger()

    @patch("orka.memory.redisstack_logger.redis.Redis")
    @patch.object(RedisStackMemoryLogger, "_ensure_index")
    @patch.object(RedisStackMemoryLogger, "_create_redis_connection")
    def test_thread_safe_client_creation(self, mock_create_connection, mock_ensure, mock_redis):
        """Test thread-safe client creation."""
        mock_create_connection.return_value = Mock()
        logger = RedisStackMemoryLogger()

        # Mock the _create_redis_connection method
        logger._create_redis_connection = Mock(return_value=mock_redis.return_value)

        # First call should create client
        client1 = logger._get_thread_safe_client()
        assert logger._create_redis_connection.called

        # Second call should return cached client
        logger._create_redis_connection.reset_mock()
        client2 = logger._get_thread_safe_client()
        assert not logger._create_redis_connection.called
        assert client1 is client2

    @patch("orka.memory.redisstack_logger.redis.Redis")
    @patch("orka.utils.bootstrap_memory_index.ensure_enhanced_memory_index")
    @patch.object(RedisStackMemoryLogger, "_create_redis_connection")
    def test_index_creation_with_embedder_dimensions(
        self,
        mock_create_connection,
        mock_ensure,
        mock_redis,
    ):
        """Test index creation with embedder dimensions."""
        mock_ensure.return_value = True
        mock_create_connection.return_value = Mock()
        mock_embedder = Mock()
        mock_embedder.embedding_dim = 512

        logger = RedisStackMemoryLogger(embedder=mock_embedder)

        mock_ensure.assert_called_once()
        call_args = mock_ensure.call_args
        assert call_args[1]["vector_dim"] == 512


class TestRedisStackLoggerMemoryOperations:
    """Test memory logging and manipulation operations."""

    def setup_method(self):
        """Set up test fixtures."""
        with (
            patch("orka.memory.redisstack_logger.redis.Redis"),
            patch.object(
                RedisStackMemoryLogger,
                "_ensure_index",
            ),
            patch.object(RedisStackMemoryLogger, "_create_redis_connection"),
        ):
            self.logger = RedisStackMemoryLogger()
            self.mock_redis_client = Mock()
            self.logger.redis_client = self.mock_redis_client
            self.logger._get_thread_safe_client = Mock(return_value=self.mock_redis_client)

    def test_log_memory_basic(self):
        """Test basic memory logging."""
        self.mock_redis_client.hset.return_value = 1

        with patch("time.time", return_value=1000.0):
            with patch("uuid.uuid4") as mock_uuid:
                # Mock the UUID to return a value that when str() and replace("-", "") gives "test123"
                mock_uuid.return_value.__str__ = Mock(return_value="test-1-2-3")
                # So str(uuid.uuid4()).replace("-", "") will return "test123"

                memory_key = self.logger.log_memory(
                    content="Test memory content",
                    node_id="test_node",
                    trace_id="test_trace",
                    metadata={"type": "test"},
                    importance_score=0.8,
                    memory_type="short_term",
                    expiry_hours=2.0,
                )

        # Memory key should use str(uuid4()).replace("-", "") format
        assert memory_key == "orka_memory:test123"
        assert self.mock_redis_client.hset.called
        assert self.mock_redis_client.expire.called

    def test_log_memory_with_embedder(self):
        """Test memory logging with vector embeddings."""
        mock_embedder = Mock()
        mock_embedder.embedding_dim = 384
        mock_vector = np.array([0.1, 0.2, 0.3] * 128, dtype=np.float32)

        self.logger.embedder = mock_embedder
        self.logger._embedding_lock = threading.Lock()
        self.logger._get_embedding_sync = Mock(return_value=mock_vector)
        self.mock_redis_client.hset.return_value = 1

        with patch("time.time", return_value=1000.0):
            with patch("uuid.uuid4") as mock_uuid:
                # Mock the UUID to return a value that when str() and replace("-", "") gives "test123"
                mock_uuid.return_value.__str__ = Mock(return_value="test-1-2-3")

                memory_key = self.logger.log_memory(
                    content="Test memory content",
                    node_id="test_node",
                    trace_id="test_trace",
                )

        assert memory_key == "orka_memory:test123"
        assert self.logger._get_embedding_sync.called
        assert self.mock_redis_client.hset.called

    def test_log_memory_expiry_handling(self):
        """Test memory logging with expiry time calculation."""
        self.mock_redis_client.hset.return_value = 1

        with patch("time.time", return_value=1000.0):
            with patch("uuid.uuid4") as mock_uuid:
                # Mock the UUID to return a value that when str() and replace("-", "") gives "test123"
                mock_uuid.return_value.__str__ = Mock(return_value="test-1-2-3")

                self.logger.log_memory(
                    content="Test content",
                    node_id="test_node",
                    trace_id="test_trace",
                    expiry_hours=1.0,
                )

        call_args = self.mock_redis_client.hset.call_args
        memory_data = call_args[1]["mapping"]
        assert "orka_expire_time" in memory_data
        assert memory_data["orka_expire_time"] == int((1000.0 + 3600) * 1000)

    def test_log_orchestration_event_with_extraction(self):
        """Test logging orchestration events with content extraction."""
        self.mock_redis_client.hset = Mock(return_value=1)
        self.mock_redis_client.expire = Mock(return_value=True)

        # Provide a proper memory decay config to avoid NoneType errors
        self.logger.memory_decay_config = {
            "enabled": True,
            "short_term_hours": 1.0,
            "long_term_hours": 24.0,
        }

        self.logger._classify_memory_category = Mock(return_value="log")
        self.logger.memory = []

        payload = {
            "agent_output": "Test agent output",
            "user_input": "Test user input",
        }

        with patch("time.time", return_value=1000.0), patch("uuid.uuid4") as mock_uuid:
            # Mock the UUID to return a value that when str() and replace("-", "") gives "testkey123"
            mock_uuid.return_value.__str__ = Mock(return_value="test-k-e-y-123")

            self.logger.log(
                agent_id="test_agent",
                event_type="agent_response",
                payload=payload,
                step=1,
                run_id="test_run",
                log_type="memory",
            )

        assert self.mock_redis_client.hset.called
        call_args = self.mock_redis_client.hset.call_args
        memory_data = call_args[1]["mapping"]
        assert "orka_expire_time" in memory_data
        # agent_response gets 0.5 importance, so expiry = 1.0 * (1.0 + 0.5) = 1.5 hours = 5400 seconds
        expected_expiry = int((1000.0 + 5400) * 1000)
        assert memory_data["orka_expire_time"] == expected_expiry

    def test_content_extraction_from_payload(self):
        """Test content extraction from different payload types."""
        # Test with agent_output
        payload1 = {"agent_output": "Test agent output"}
        content1 = self.logger._extract_content_from_payload(payload1, "agent_response")
        assert "Test agent output" in content1
        assert "Event: agent_response" in content1

        # Test with response
        payload2 = {"response": "Test response"}
        content2 = self.logger._extract_content_from_payload(payload2, "user_input")
        assert "Test response" in content2
        assert "Event: user_input" in content2

        # Test fallback to JSON
        payload3 = {"some_data": "test"}
        content3 = self.logger._extract_content_from_payload(payload3, "unknown")
        assert "Event: unknown" in content3
        assert "some_data" in content3

    def test_importance_score_calculation(self):
        """Test importance score calculation for different event types."""
        payload = {"agent_output": "test"}

        # Test high importance events
        score1 = self.logger._calculate_importance_score("agent.error", payload)
        assert score1 == 0.9

        # Test medium importance events - agent_response is not in importance_map, defaults to 0.5
        score2 = self.logger._calculate_importance_score("agent_response", payload)
        assert score2 == 0.5  # Default value, not 0.8

        # Test events in the importance_map
        score3 = self.logger._calculate_importance_score("agent.end", payload)
        assert score3 == 0.8

        # Test low importance events
        score4 = self.logger._calculate_importance_score("memory.retrieve", payload)
        assert score4 == 0.4

    def test_memory_type_determination(self):
        """Test memory type determination based on importance and event type."""
        # Test long-term for high importance
        memory_type1 = self.logger._determine_memory_type("agent.error", 0.9)
        assert memory_type1 == "long_term"

        # Test short-term for medium importance
        memory_type2 = self.logger._determine_memory_type("agent_response", 0.7)
        assert memory_type2 == "short_term"

    def test_expiry_hours_calculation(self):
        """Test expiry hours calculation with decay config."""
        # Test with agent-specific config
        agent_config = {"short_term_hours": 1.0, "long_term_hours": 24.0, "enabled": True}

        expiry1 = self.logger._calculate_expiry_hours("short_term", 0.5, agent_config)
        assert expiry1 == 1.0 * 1.5  # 1.0 * (1.0 + importance_score)

        expiry2 = self.logger._calculate_expiry_hours("long_term", 0.8, agent_config)
        assert expiry2 == 24.0 * 1.8  # 24.0 * (1.0 + importance_score)


class TestRedisStackLoggerSearch:
    """Test memory search functionality."""

    def setup_method(self):
        """Set up test fixtures."""
        with (
            patch("orka.memory.redisstack_logger.redis.Redis"),
            patch.object(RedisStackMemoryLogger, "_ensure_index"),
            patch.object(RedisStackMemoryLogger, "_create_redis_connection"),
        ):
            self.logger = RedisStackMemoryLogger()
            self.mock_redis_client = Mock()
            self.logger.redis_client = self.mock_redis_client
            self.logger._get_thread_safe_client = Mock(return_value=self.mock_redis_client)

    def test_vector_search_with_embedder(self):
        """Test vector-based memory search."""
        mock_embedder = Mock()
        mock_vector = np.array([0.1, 0.2, 0.3] * 128, dtype=np.float32)
        self.logger.embedder = mock_embedder
        self.logger._get_embedding_sync = Mock(return_value=mock_vector)

        mock_search_results = [{"key": "orka_memory:123", "score": 0.95}]

        mock_memory_data = {
            "content": "Test content",
            "node_id": "test_node",
            "trace_id": "test_trace",
            "importance_score": "0.8",
            "memory_type": "short_term",
            "metadata": '{"log_type": "memory"}',
        }

        self.mock_redis_client.hgetall.return_value = mock_memory_data
        self.logger._is_expired = Mock(return_value=False)

        with patch("orka.utils.bootstrap_memory_index.hybrid_vector_search") as mock_search:
            mock_search.return_value = mock_search_results

            results = self.logger.search_memories("test query")

            assert mock_search.called

    def test_fallback_text_search(self):
        """Test fallback text search when vector search fails."""
        self.logger.embedder = None

        # Mock search results
        mock_doc = Mock()
        mock_doc.id = "orka_memory:123"
        mock_search_results = Mock()
        mock_search_results.docs = [mock_doc]

        self.mock_redis_client.ft().search.return_value = mock_search_results
        mock_memory_data = {
            "content": "test content matching query",
            "node_id": "test_node",
            "trace_id": "test_trace",
            "importance_score": "0.8",
            "memory_type": "short_term",
            "metadata": '{"log_type": "memory"}',
        }
        self.mock_redis_client.hgetall.return_value = mock_memory_data
        self.logger._is_expired = Mock(return_value=False)
        self.logger._get_ttl_info = Mock(
            return_value={
                "ttl_seconds": 3600,
                "ttl_formatted": "1h",
                "expires_at": 123456789,
                "expires_at_formatted": "2023-01-01 12:00:00",
                "has_expiry": True,
            },
        )

        results = self.logger._fallback_text_search("test query", num_results=10)

        assert self.mock_redis_client.ft().search.called

    def test_search_with_filters(self):
        """Test memory search with various filters."""
        self.logger.embedder = None

        mock_doc = Mock()
        mock_doc.id = "orka_memory:123"
        mock_search_results = Mock()
        mock_search_results.docs = [mock_doc]

        self.mock_redis_client.ft().search.return_value = mock_search_results
        mock_memory_data = {
            "content": "test content",
            "node_id": "other_node",  # Should be filtered out
            "trace_id": "test_trace",
            "importance_score": "0.3",  # Should be filtered out
            "memory_type": "other_type",  # Should be filtered out
            "metadata": '{"log_type": "memory"}',
        }
        self.mock_redis_client.hgetall.return_value = mock_memory_data
        self.logger._is_expired = Mock(return_value=False)

        results = self.logger._fallback_text_search(
            "test",
            num_results=10,
            node_id="test_node",
            memory_type="short_term",
            min_importance=0.5,
        )

        # Should be empty due to filters
        assert len(results) == 0

    def test_safe_redis_value_retrieval(self):
        """Test safe retrieval of Redis values."""
        # Test string key
        data1 = {"key": "value"}
        result1 = self.logger._safe_get_redis_value(data1, "key")
        assert result1 == "value"

        # Test bytes key
        data2 = {b"key": b"value"}
        result2 = self.logger._safe_get_redis_value(data2, "key")
        assert result2 == "value"

        # Test default value
        data3 = {}
        result3 = self.logger._safe_get_redis_value(data3, "missing", "default")
        assert result3 == "default"

    def test_memory_expiration_check(self):
        """Test memory expiration checking."""
        current_time_ms = int(time.time() * 1000)

        # Test non-expired memory
        memory_data1 = {"orka_expire_time": str(current_time_ms + 3600000)}
        assert not self.logger._is_expired(memory_data1)

        # Test expired memory
        memory_data2 = {"orka_expire_time": str(current_time_ms - 3600000)}
        assert self.logger._is_expired(memory_data2)

        # Test memory without expiry
        memory_data3 = {}
        assert not self.logger._is_expired(memory_data3)

    def test_search_error_handling(self):
        """Test search error handling."""
        self.logger.embedder = Mock()
        self.logger._get_embedding_sync = Mock(side_effect=Exception("Embedding failed"))

        with patch.object(self.logger, "_fallback_text_search") as mock_fallback:
            mock_fallback.return_value = []

            results = self.logger.search_memories("test query")

            assert mock_fallback.called
            assert results == []


class TestRedisStackLoggerManagement:
    """Test memory management operations."""

    def setup_method(self):
        """Set up test fixtures."""
        with (
            patch("orka.memory.redisstack_logger.redis.Redis"),
            patch.object(RedisStackMemoryLogger, "_ensure_index"),
            patch.object(RedisStackMemoryLogger, "_create_redis_connection"),
        ):
            self.logger = RedisStackMemoryLogger()
            self.mock_redis_client = Mock()
            self.logger.redis_client = self.mock_redis_client

    def test_get_all_memories(self):
        """Test retrieving all memories."""
        self.mock_redis_client.keys.return_value = ["orka_memory:123", "orka_memory:456"]
        mock_memory_data = {
            "content": "test content",
            "node_id": "test_node",
            "trace_id": "test_trace",
            "importance_score": "0.8",
            "timestamp": "1000000",
            "metadata": "{}",
        }
        self.mock_redis_client.hgetall.return_value = mock_memory_data
        self.logger._is_expired = Mock(return_value=False)

        memories = self.logger.get_all_memories()

        assert self.mock_redis_client.keys.called
        assert len(memories) >= 0

    def test_delete_memory(self):
        """Test memory deletion."""
        self.mock_redis_client.delete.return_value = 1

        result = self.logger.delete_memory("orka_memory:123")

        assert result is True
        self.mock_redis_client.delete.assert_called_with("orka_memory:123")

    def test_clear_all_memories(self):
        """Test clearing all memories."""
        self.mock_redis_client.keys.return_value = ["orka_memory:123", "orka_memory:456"]
        self.mock_redis_client.delete.return_value = 2

        self.logger.clear_all_memories()

        assert self.mock_redis_client.keys.called

    def test_memory_statistics(self):
        """Test memory statistics generation."""
        self.mock_redis_client.keys.return_value = ["orka_memory:123", "orka_memory:456"]
        mock_memory_data = {
            "memory_type": "short_term",
            "importance_score": "0.8",
            "timestamp": "1000000",
            "metadata": '{"log_type": "memory"}',
        }
        self.mock_redis_client.hgetall.return_value = mock_memory_data
        self.logger._is_expired = Mock(return_value=False)
        self.logger._get_thread_safe_client = Mock(return_value=self.mock_redis_client)

        with patch("time.time", return_value=1001.0):
            stats = self.logger.get_memory_stats()

            # Check for actual field names from implementation
            assert "total_entries" in stats
            assert "active_entries" in stats
            assert "backend" in stats

    def test_cleanup_expired_memories(self):
        """Test cleanup of expired memories."""
        current_time_ms = int(time.time() * 1000)

        self.mock_redis_client.keys.return_value = ["orka_memory:123", "orka_memory:456"]

        def mock_hgetall(key):
            if key == "orka_memory:123":
                return {"orka_expire_time": str(current_time_ms - 3600000)}
            else:
                return {"orka_expire_time": str(current_time_ms + 3600000)}

        self.mock_redis_client.hgetall.side_effect = mock_hgetall
        self.mock_redis_client.delete.return_value = 1

        result = self.logger.cleanup_expired_memories()

        assert "total_checked" in result
        assert "expired_found" in result
        assert "cleaned" in result

    def test_cleanup_expired_memories_dry_run(self):
        """Test dry run cleanup of expired memories."""
        current_time_ms = int(time.time() * 1000)

        self.mock_redis_client.keys.return_value = ["orka_memory:123"]
        self.mock_redis_client.hgetall.return_value = {
            "orka_expire_time": str(current_time_ms - 3600000),
        }

        result = self.logger.cleanup_expired_memories(dry_run=True)

        assert result["cleaned"] == 0
        assert not self.mock_redis_client.delete.called

    def test_get_recent_stored_memories(self):
        """Test retrieving recent stored memories."""
        self.mock_redis_client.keys.return_value = ["orka_memory:123", "orka_memory:456"]
        mock_memory_data = {
            "content": "test content",
            "timestamp": "1000000",
            "metadata": '{"log_type": "memory"}',
        }
        self.mock_redis_client.hgetall.return_value = mock_memory_data
        self.logger._is_expired = Mock(return_value=False)
        self.logger._get_ttl_info = Mock(
            return_value={
                "ttl_seconds": 3600,
                "ttl_formatted": "1h",
                "expires_at": 123456789,
                "expires_at_formatted": "2023-01-01 12:00:00",
                "has_expiry": True,
            },
        )

        memories = self.logger.get_recent_stored_memories(count=5)

        assert isinstance(memories, list)

    def test_tail_recent_entries(self):
        """Test tailing recent log entries."""
        with patch.object(self.logger, "get_all_memories") as mock_get_all:
            mock_get_all.return_value = [
                {"timestamp": 1000000, "content": "test"},
                {"timestamp": 2000000, "content": "test2"},
            ]

            entries = self.logger.tail(count=10)

            assert isinstance(entries, list)
            assert len(entries) <= 10


class TestRedisStackLoggerRedisOperations:
    """Test Redis operation wrappers."""

    def setup_method(self):
        """Set up test fixtures."""
        with (
            patch("orka.memory.redisstack_logger.redis.Redis"),
            patch.object(RedisStackMemoryLogger, "_ensure_index"),
            patch.object(RedisStackMemoryLogger, "_create_redis_connection"),
        ):
            self.logger = RedisStackMemoryLogger()
            self.mock_redis_client = Mock()
            self.logger.redis_client = self.mock_redis_client
            self.logger._get_thread_safe_client = Mock(return_value=self.mock_redis_client)

    def test_hset_wrapper(self):
        """Test HSET operation wrapper."""
        self.mock_redis_client.hset.return_value = 1

        result = self.logger.hset("test_key", "field", "value")

        assert result == 1
        self.mock_redis_client.hset.assert_called_with("test_key", "field", "value")

    def test_hget_wrapper(self):
        """Test HGET operation wrapper."""
        self.mock_redis_client.hget.return_value = "value"

        result = self.logger.hget("test_key", "field")

        assert result == "value"
        self.mock_redis_client.hget.assert_called_with("test_key", "field")

    def test_hkeys_wrapper(self):
        """Test HKEYS operation wrapper."""
        self.mock_redis_client.hkeys.return_value = ["field1", "field2"]

        result = self.logger.hkeys("test_key")

        assert result == ["field1", "field2"]
        self.mock_redis_client.hkeys.assert_called_with("test_key")

    def test_hdel_wrapper(self):
        """Test HDEL operation wrapper."""
        self.mock_redis_client.hdel.return_value = 1

        result = self.logger.hdel("test_key", "field1", "field2")

        assert result == 1
        self.mock_redis_client.hdel.assert_called_with("test_key", "field1", "field2")

    def test_smembers_wrapper(self):
        """Test SMEMBERS operation wrapper."""
        self.mock_redis_client.smembers.return_value = {"member1", "member2"}

        result = self.logger.smembers("test_set")

        assert isinstance(result, list)
        self.mock_redis_client.smembers.assert_called_with("test_set")

    def test_sadd_wrapper(self):
        """Test SADD operation wrapper."""
        self.mock_redis_client.sadd.return_value = 2

        result = self.logger.sadd("test_set", "member1", "member2")

        assert result == 2
        self.mock_redis_client.sadd.assert_called_with("test_set", "member1", "member2")

    def test_srem_wrapper(self):
        """Test SREM operation wrapper."""
        self.mock_redis_client.srem.return_value = 1

        result = self.logger.srem("test_set", "member1")

        assert result == 1
        self.mock_redis_client.srem.assert_called_with("test_set", "member1")

    def test_get_wrapper(self):
        """Test GET operation wrapper."""
        self.mock_redis_client.get.return_value = "value"

        result = self.logger.get("test_key")

        assert result == "value"
        self.mock_redis_client.get.assert_called_with("test_key")

    def test_set_wrapper(self):
        """Test SET operation wrapper."""
        self.mock_redis_client.set.return_value = True

        result = self.logger.set("test_key", "value")

        assert result is True
        self.mock_redis_client.set.assert_called_with("test_key", "value")

    def test_delete_wrapper(self):
        """Test DELETE operation wrapper."""
        self.mock_redis_client.delete.return_value = 1

        result = self.logger.delete("test_key1", "test_key2")

        assert result == 1
        self.mock_redis_client.delete.assert_called_with("test_key1", "test_key2")


class TestRedisStackLoggerAdvanced:
    """Test advanced RedisStack logger functionality."""

    def setup_method(self):
        """Set up test fixtures."""
        with (
            patch("orka.memory.redisstack_logger.redis.Redis"),
            patch.object(RedisStackMemoryLogger, "_ensure_index"),
            patch.object(RedisStackMemoryLogger, "_create_redis_connection"),
        ):
            self.logger = RedisStackMemoryLogger()
            self.mock_redis_client = Mock()
            self.logger.redis_client = self.mock_redis_client
            self.logger._get_thread_safe_client = Mock(return_value=self.mock_redis_client)

    async def test_embedding_generation_success(self):
        """Test successful embedding generation."""
        # Mock embedder
        mock_embedder = AsyncMock()
        mock_embedder.get_embedding.return_value = [0.1, 0.2, 0.3]

        # Create logger instance with mock embedder
        logger = RedisStackMemoryLogger(
            host="localhost",
            port=6379,
            embedder=mock_embedder,
        )

        # Test embedding generation
        text = "Test text"
        embedding = await logger._generate_embedding(text)

        # Verify
        assert embedding == [0.1, 0.2, 0.3]
        mock_embedder.get_embedding.assert_called_once_with(text)

        # Clean up
        await logger.close()
        await mock_embedder.close()

    def test_embedding_generation_failure(self):
        """Test embedding generation failure and fallback."""
        mock_embedder = Mock()
        mock_embedder.embedding_dim = 384

        self.logger.embedder = mock_embedder
        self.logger._embedding_lock = threading.Lock()

        with patch("asyncio.get_running_loop", side_effect=RuntimeError("No running loop")):
            with patch("asyncio.run", side_effect=Exception("Embedding failed")):
                result = self.logger._get_embedding_sync("test text")

                assert isinstance(result, np.ndarray)
                assert result.shape == (384,)
                assert np.all(result == 0)

    def test_performance_metrics_collection(self):
        """Test performance metrics collection."""
        # Mock index info
        mock_index_info = {
            "num_docs": 100,
            "indexing": False,
            "percent_indexed": 100,
        }
        self.mock_redis_client.ft().info.return_value = mock_index_info
        self.mock_redis_client.keys.return_value = ["orka_memory:123", "orka_memory:456"]

        with patch.object(self.logger, "get_recent_stored_memories") as mock_recent:
            mock_recent.return_value = [
                {"importance_score": 0.8, "memory_type": "short_term"},
                {"importance_score": 0.9, "memory_type": "long_term"},
            ]

            metrics = self.logger.get_performance_metrics()

            assert "vector_search_enabled" in metrics
            assert "index_status" in metrics
            assert "memory_quality" in metrics

    def test_index_creation_wrapper(self):
        """Test index creation wrapper method."""
        with patch("orka.utils.bootstrap_memory_index.ensure_enhanced_memory_index") as mock_ensure:
            mock_ensure.return_value = True

            result = self.logger.ensure_index()

            assert result is True
            mock_ensure.assert_called_once()

    def test_async_context_embedding_fallback(self):
        """Test embedding fallback in async context."""
        mock_embedder = Mock()
        mock_embedder.embedding_dim = 384
        expected_vector = np.array([0.1, 0.2, 0.3] * 128, dtype=np.float32)
        mock_embedder._fallback_encode.return_value = expected_vector

        self.logger.embedder = mock_embedder
        self.logger._embedding_lock = threading.Lock()

        mock_loop = Mock()
        with patch("asyncio.get_running_loop", return_value=mock_loop):
            result = self.logger._get_embedding_sync("test text")

            assert np.array_equal(result, expected_vector)
            mock_embedder._fallback_encode.assert_called_once_with("test text")

    def test_thread_safety_stress_test(self):
        """Test thread safety under concurrent access."""
        self.logger._create_redis_connection = Mock(return_value=self.mock_redis_client)

        def worker():
            client = self.logger._get_thread_safe_client()
            return client

        import concurrent.futures

        with concurrent.futures.ThreadPoolExecutor(max_workers=5) as executor:
            futures = [executor.submit(worker) for _ in range(10)]
            results = [f.result() for f in futures]

        assert len(results) == 10

    def test_ttl_info_calculation(self):
        """Test TTL information calculation."""
        current_time_ms = int(time.time() * 1000)

        memory_data = {
            "orka_expire_time": str(current_time_ms + 3600000),
            "timestamp": str(current_time_ms - 1800000),
        }

        ttl_info = self.logger._get_ttl_info("test_key", memory_data, current_time_ms)

        # Check actual field names from implementation
        assert "has_expiry" in ttl_info
        assert "ttl_seconds" in ttl_info
        assert "ttl_formatted" in ttl_info
        assert "expires_at" in ttl_info
        assert "expires_at_formatted" in ttl_info
        assert ttl_info["has_expiry"] is True

    def test_ttl_info_no_expiry(self):
        """Test TTL info for memory without expiry."""
        current_time_ms = int(time.time() * 1000)

        memory_data = {"timestamp": str(current_time_ms - 1800000)}

        ttl_info = self.logger._get_ttl_info("test_key", memory_data, current_time_ms)

        assert ttl_info["has_expiry"] is False
        assert ttl_info["ttl_formatted"] == "Never"
        assert ttl_info["expires_at"] is None

    def test_close_cleanup(self):
        """Test proper cleanup on close."""
        self.logger.close()
        # Should not raise exception

    def test_redis_property_backward_compatibility(self):
        """Test backward compatibility redis property."""
        assert self.logger.redis == self.logger.redis_client


class TestRedisStackLoggerEdgeCases:
    """Test edge cases and error conditions."""

    def setup_method(self):
        """Set up test fixtures."""
        with (
            patch("orka.memory.redisstack_logger.redis.Redis"),
            patch.object(RedisStackMemoryLogger, "_ensure_index"),
            patch.object(RedisStackMemoryLogger, "_create_redis_connection"),
        ):
            self.logger = RedisStackMemoryLogger()
            self.mock_redis_client = Mock()
            self.logger.redis_client = self.mock_redis_client
            self.logger._get_thread_safe_client = Mock(return_value=self.mock_redis_client)

    def test_log_memory_exception_handling(self):
        """Test exception handling in log_memory."""
        self.mock_redis_client.hset.side_effect = Exception("Redis error")

        with pytest.raises(Exception, match="Redis error"):
            self.logger.log_memory(
                content="test",
                node_id="test_node",
                trace_id="test_trace",
            )

    def test_search_with_corrupted_metadata(self):
        """Test search handling corrupted metadata."""
        self.logger.embedder = None

        mock_doc = Mock()
        mock_doc.id = "orka_memory:123"
        mock_search_results = Mock()
        mock_search_results.docs = [mock_doc]

        self.mock_redis_client.ft().search.return_value = mock_search_results
        mock_memory_data = {
            "content": "test content",
            "metadata": "invalid json{",
            "importance_score": "0.8",
        }
        self.mock_redis_client.hgetall.return_value = mock_memory_data
        self.logger._is_expired = Mock(return_value=False)

        results = self.logger._fallback_text_search("test", num_results=10)
        assert isinstance(results, list)

    def test_stats_with_invalid_data(self):
        """Test statistics generation with invalid data."""
        self.mock_redis_client.keys.return_value = ["orka_memory:123"]
        mock_memory_data = {
            "importance_score": "invalid",
            "timestamp": "invalid",
        }
        self.mock_redis_client.hgetall.return_value = mock_memory_data
        self.logger._is_expired = Mock(return_value=False)
        self.logger._get_thread_safe_client = Mock(return_value=self.mock_redis_client)

        with patch("time.time", return_value=1001.0):
            stats = self.logger.get_memory_stats()

            assert "total_entries" in stats

    def test_empty_content_extraction(self):
        """Test content extraction with empty payloads."""
        content = self.logger._extract_content_from_payload({}, "unknown")
        assert "Event: unknown" in content

    def test_index_creation_failure(self):
        """Test handling of index creation failure."""
        with patch("orka.utils.bootstrap_memory_index.ensure_enhanced_memory_index") as mock_ensure:
            mock_ensure.side_effect = Exception("Index creation failed")

            try:
                self.logger._ensure_index()
            except Exception:
                pytest.fail("_ensure_index should handle exceptions gracefully")

    def test_vector_search_with_no_results(self):
        """Test vector search returning no results."""
        mock_embedder = Mock()
        mock_vector = np.array([0.1, 0.2, 0.3] * 128, dtype=np.float32)
        self.logger.embedder = mock_embedder
        self.logger._get_embedding_sync = Mock(return_value=mock_vector)

        with patch("orka.utils.bootstrap_memory_index.hybrid_vector_search") as mock_search:
            mock_search.return_value = []

            results = self.logger.search_memories("test query")

            assert results == []

    def test_cleanup_with_redis_errors(self):
        """Test cleanup handling Redis operation errors."""
        self.mock_redis_client.keys.return_value = ["orka_memory:123"]
        self.mock_redis_client.hgetall.side_effect = Exception("Redis error")

        result = self.logger.cleanup_expired_memories()

        assert "errors" in result
        assert len(result["errors"]) > 0

    def test_memory_type_edge_cases(self):
        """Test memory type determination edge cases."""
        memory_type = self.logger._determine_memory_type("critical", 1.5)
        assert memory_type in ["short_term", "long_term"]

        memory_type = self.logger._determine_memory_type("test", -0.1)
        assert memory_type in ["short_term", "long_term"]

    def test_embedding_with_no_embedder(self):
        """Test embedding generation when no embedder is set."""
        self.logger.embedder = None

        with patch("time.time", return_value=1000.0):
            with patch("uuid.uuid4") as mock_uuid:
                # Mock the UUID to return a value that when str() and replace("-", "") gives "test123"
                mock_uuid.return_value.__str__ = Mock(return_value="test-1-2-3")

                memory_key = self.logger.log_memory(
                    content="Test content",
                    node_id="test_node",
                    trace_id="test_trace",
                )

        assert memory_key == "orka_memory:test123"

    def test_classify_memory_category(self):
        """Test memory category classification."""
        # Add mock for _classify_memory_category if it exists
        if hasattr(self.logger, "_classify_memory_category"):
            result = self.logger._classify_memory_category("test_event", "test_agent", {}, "log")
            assert isinstance(result, str)
        else:
            # Skip if method doesn't exist
            pass
