"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'When creating a VPC': {
        'with the default CIDR range': {
            'vpc.vpcId returns a token to the VPC ID'(test) {
                const stack = getTestStack();
                const vpc = new lib_1.Vpc(stack, 'TheVPC');
                test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
                test.done();
            },
            'it uses the correct network range'(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                    CidrBlock: lib_1.Vpc.DEFAULT_CIDR_RANGE,
                    EnableDnsHostnames: true,
                    EnableDnsSupport: true,
                    InstanceTenancy: lib_1.DefaultInstanceTenancy.DEFAULT,
                }));
                test.done();
            },
            'the Name tag is defaulted to path'(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([{ Key: 'Name', Value: 'TestStack/TheVPC' }])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::InternetGateway', hasTags([{ Key: 'Name', Value: 'TestStack/TheVPC' }])));
                test.done();
            },
        },
        'with all of the properties set, it successfully sets the correct VPC properties'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '192.168.0.0/16',
                enableDnsHostnames: false,
                enableDnsSupport: false,
                defaultInstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '192.168.0.0/16',
                EnableDnsHostnames: false,
                EnableDnsSupport: false,
                InstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            }));
            test.done();
        },
        'dns getters correspond to CFN properties': (() => {
            const tests = {};
            const inputs = [
                { dnsSupport: false, dnsHostnames: false },
                // {dnsSupport: false, dnsHostnames: true} - this configuration is illegal so its not part of the permutations.
                { dnsSupport: true, dnsHostnames: false },
                { dnsSupport: true, dnsHostnames: true },
            ];
            for (const input of inputs) {
                tests[`[dnsSupport=${input.dnsSupport},dnsHostnames=${input.dnsHostnames}]`] = (test) => {
                    const stack = getTestStack();
                    const vpc = new lib_1.Vpc(stack, 'TheVPC', {
                        cidr: '192.168.0.0/16',
                        enableDnsHostnames: input.dnsHostnames,
                        enableDnsSupport: input.dnsSupport,
                        defaultInstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
                    });
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                        CidrBlock: '192.168.0.0/16',
                        EnableDnsHostnames: input.dnsHostnames,
                        EnableDnsSupport: input.dnsSupport,
                        InstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
                    }));
                    test.equal(input.dnsSupport, vpc.dnsSupportEnabled);
                    test.equal(input.dnsHostnames, vpc.dnsHostnamesEnabled);
                    test.done();
                };
            }
            return tests;
        })(),
        'contains the correct number of subnets'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const zones = stack.availabilityZones.length;
            test.equal(vpc.publicSubnets.length, zones);
            test.equal(vpc.privateSubnets.length, zones);
            test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
            test.done();
        },
        'can refer to the internet gateway'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            test.deepEqual(stack.resolve(vpc.internetGatewayId), { Ref: 'TheVPCIGWFA25CC08' });
            test.done();
        },
        'with only isolated subnets, the VPC should not contain an IGW or NAT Gateways'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    },
                ],
            });
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::InternetGateway'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                MapPublicIpOnLaunch: false,
            }));
            test.done();
        },
        'with no private subnets, the VPC should have an IGW but no NAT Gateways'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            test.done();
        },
        'with private subnets and custom networkAcl.'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.PRIVATE,
                        name: 'private',
                    },
                ],
            });
            const nacl1 = new lib_1.NetworkAcl(stack, 'myNACL1', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PRIVATE },
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSEgress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.EGRESS,
                cidr: lib_1.AclCidr.ipv4('10.0.0.0/16'),
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSIngress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.INGRESS,
                cidr: lib_1.AclCidr.anyIpv4(),
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAcl', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAclEntry', 2));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SubnetNetworkAclAssociation', 3));
            test.done();
        },
        'with no subnets defined, the VPC should have an IGW, and a NAT Gateway per AZ'(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {});
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', zones));
            test.done();
        },
        'with isolated and public subnet, should be able to use the internet gateway to define routes'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'isolated',
                    },
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'public',
                    },
                ],
            });
            vpc.isolatedSubnets[0].addRoute('TheRoute', {
                routerId: vpc.internetGatewayId,
                routerType: lib_1.RouterType.GATEWAY,
                destinationCidrBlock: '8.8.8.8/32',
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::InternetGateway'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '8.8.8.8/32',
                GatewayId: {},
            }));
            test.done();
        },
        'with only isolated subnets the internet gateway should be undefined'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'isolated',
                    },
                ],
            });
            test.equal(vpc.internetGatewayId, undefined);
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::InternetGateway'));
            test.done();
        },
        'with subnets and reserved subnets defined, VPC subnet count should not contain reserved subnets '(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
                maxAzs: 3,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            test.done();
        },
        'with reserved subnets, any other subnets should not have cidrBlock from within reserved space'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 24,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                maxAzs: 3,
            });
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            for (let i = 3; i < 6; i++) {
                assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            for (let i = 6; i < 9; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            test.done();
        },
        'with custom subnets, the VPC should have the right number of subnets, an IGW, and a NAT Gateway per AZ'(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
                maxAzs: 3,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', zones));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.6.${i * 16}/28`,
                }));
            }
            test.done();
        },
        'with custom subents and natGateways = 2 there should be only two NATGW'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                natGateways: 2,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
                maxAzs: 3,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 2));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.6.${i * 16}/28`,
                }));
            }
            test.done();
        },
        'with enableDnsHostnames enabled but enableDnsSupport disabled, should throw an Error'(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'TheVPC', {
                enableDnsHostnames: true,
                enableDnsSupport: false,
            }));
            test.done();
        },
        'with public subnets MapPublicIpOnLaunch is true'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 1,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 1));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                MapPublicIpOnLaunch: true,
            }));
            test.done();
        },
        'maxAZs defaults to 3 if unset'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC');
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 6));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i * 32}.0/19`,
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with maxAZs set to 2'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', { maxAzs: 2 });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 4));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 4));
            for (let i = 0; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i * 64}.0/18`,
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with natGateway set to 1'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                natGateways: 1,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 1));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with natGateway subnets defined'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'egress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'egress',
                },
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 3));
            for (let i = 1; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                        Key: 'Name',
                        Value: `TestStack/VPC/egressSubnet${i}`,
                    }, {
                        Key: 'aws-cdk:subnet-name',
                        Value: 'egress',
                    }])));
            }
            test.done();
        },
        'natGateways = 0 throws if no PRIVATE subnets configured'(test) {
            const stack = getTestStack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'VPC', {
                    natGateways: 0,
                    subnetConfiguration: [
                        {
                            name: 'public',
                            subnetType: lib_1.SubnetType.PUBLIC,
                        },
                        {
                            name: 'private',
                            subnetType: lib_1.SubnetType.PRIVATE,
                        },
                    ],
                });
            }, /make sure you don't configure any PRIVATE subnets/);
            test.done();
        },
        'natGateway = 0 defaults with ISOLATED subnet'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                natGateways: 0,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                    Key: 'aws-cdk:subnet-type',
                    Value: 'Isolated',
                }])));
            test.done();
        },
        'unspecified natGateways constructs with PRIVATE subnet'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                    Key: 'aws-cdk:subnet-type',
                    Value: 'Private',
                }])));
            test.done();
        },
        'natGateways = 0 allows RESERVED PRIVATE subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                ],
                natGateways: 0,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                    Key: 'aws-cdk:subnet-name',
                    Value: 'ingress',
                }])));
            test.done();
        },
        'with mis-matched nat and subnet configs it throws'(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'notthere',
                },
            }));
            test.done();
        },
        'with a vpn gateway'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                vpnGateway: true,
                vpnGatewayAsn: 65000,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGateway', {
                AmazonSideAsn: 65000,
                Type: 'ipsec.1',
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCGatewayAttachment', {
                VpcId: {
                    Ref: 'VPCB9E5F0B4',
                },
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027',
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on private and isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'Private' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027',
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'route propagation defaults to isolated subnets when there are no private subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'route propagation defaults to public subnets when there are no private/isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                ],
                vpnGateway: true,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPublicSubnet1RouteTableFEE4B781',
                    },
                    {
                        Ref: 'VPCPublicSubnet2RouteTable6F1A15F1',
                    },
                    {
                        Ref: 'VPCPublicSubnet3RouteTable98AE0E14',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'fails when specifying vpnConnections with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnConnections: {
                    VpnConnection: {
                        asn: 65000,
                        ip: '192.0.2.1',
                    },
                },
            }), /`vpnConnections`.+`vpnGateway`.+false/);
            test.done();
        },
        'fails when specifying vpnGatewayAsn with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnGatewayAsn: 65000,
            }), /`vpnGatewayAsn`.+`vpnGateway`.+false/);
            test.done();
        },
        'Subnets have a defaultChild'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            test.ok(vpc.publicSubnets[0].node.defaultChild instanceof lib_1.CfnSubnet);
            test.done();
        },
        'CIDR cannot be a Token'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'Vpc', {
                    cidr: core_1.Lazy.stringValue({ produce: () => 'abc' }),
                });
            }, /property must be a concrete CIDR string/);
            test.done();
        },
        'Default NAT gateway provider'(test) {
            const stack = new core_1.Stack();
            const natGatewayProvider = lib_1.NatProvider.gateway();
            new lib_1.Vpc(stack, 'VpcNetwork', { natGatewayProvider });
            test.ok(natGatewayProvider.configuredGateways.length > 0);
            test.done();
        },
        'Can add an IPv6 route'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            vpc.publicSubnets[0].addRoute('SomeRoute', {
                destinationIpv6CidrBlock: '2001:4860:4860::8888/32',
                routerId: 'router-1',
                routerType: lib_1.RouterType.NETWORK_INTERFACE,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationIpv6CidrBlock: '2001:4860:4860::8888/32',
                NetworkInterfaceId: 'router-1',
            }));
            test.done();
        },
        'Can add an IPv4 route'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            vpc.publicSubnets[0].addRoute('SomeRoute', {
                destinationCidrBlock: '0.0.0.0/0',
                routerId: 'router-1',
                routerType: lib_1.RouterType.NETWORK_INTERFACE,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NetworkInterfaceId: 'router-1',
            }));
            test.done();
        },
    },
    'NAT instances': {
        'Can configure NAT instances instead of NAT gateways'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const natGatewayProvider = lib_1.NatProvider.instance({
                instanceType: new lib_1.InstanceType('q86.mega'),
                machineImage: new lib_1.GenericLinuxImage({
                    'us-east-1': 'ami-1',
                }),
            });
            new lib_1.Vpc(stack, 'TheVPC', { natGatewayProvider });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 3));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
                ImageId: 'ami-1',
                InstanceType: 'q86.mega',
                SourceDestCheck: false,
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Route', {
                RouteTableId: { Ref: 'TheVPCPrivateSubnet1RouteTableF6513BC2' },
                DestinationCidrBlock: '0.0.0.0/0',
                InstanceId: { Ref: 'TheVPCPublicSubnet1NatInstanceCC514192' },
            }));
            test.done();
        },
        'natGateways controls amount of NAT instances'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            new lib_1.Vpc(stack, 'TheVPC', {
                natGatewayProvider: lib_1.NatProvider.instance({
                    instanceType: new lib_1.InstanceType('q86.mega'),
                    machineImage: new lib_1.GenericLinuxImage({
                        'us-east-1': 'ami-1',
                    }),
                }),
                natGateways: 1,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 1));
            test.done();
        },
        'can configure Security Groups of NAT instances'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const provider = lib_1.NatProvider.instance({
                instanceType: new lib_1.InstanceType('q86.mega'),
                machineImage: new lib_1.GenericLinuxImage({
                    'us-east-1': 'ami-1',
                }),
                allowAllTraffic: false,
            });
            new lib_1.Vpc(stack, 'TheVPC', {
                natGatewayProvider: provider,
            });
            provider.connections.allowFrom(lib_1.Peer.ipv4('1.2.3.4/32'), lib_1.Port.tcp(86));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                SecurityGroupIngress: [
                    {
                        CidrIp: '1.2.3.4/32',
                        Description: 'from 1.2.3.4/32:86',
                        FromPort: 86,
                        IpProtocol: 'tcp',
                        ToPort: 86,
                    },
                ],
            }));
            test.done();
        },
    },
    'Network ACL association': {
        'by default uses default ACL reference'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId,
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { 'Fn::GetAtt': ['TheVPCPublicSubnet1Subnet770D4FF2', 'NetworkAclAssociationId'] },
                    },
                },
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
        'if ACL is replaced new ACL reference is returned'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            // WHEN
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId,
            });
            new lib_1.NetworkAcl(stack, 'ACL', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PUBLIC },
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { Ref: 'ACLDBD1BB49' },
                    },
                },
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
    },
    'When creating a VPC with a custom CIDR range': {
        'vpc.vpcCidrBlock is the correct network range'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '192.168.0.0/16',
            }));
            test.done();
        },
    },
    'When tagging': {
        'VPC propagated tags will be on subnet, IGW, routetables, NATGW'(test) {
            const stack = getTestStack();
            const tags = {
                VpcType: 'Good',
            };
            const noPropTags = {
                BusinessUnit: 'Marketing',
            };
            const allTags = { ...tags, ...noPropTags };
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            // overwrite to set propagate
            core_1.Tags.of(vpc).add('BusinessUnit', 'Marketing', { includeResourceTypes: [lib_1.CfnVPC.CFN_RESOURCE_TYPE_NAME] });
            core_1.Tags.of(vpc).add('VpcType', 'Good');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags(toCfnTags(allTags))));
            const taggables = ['Subnet', 'InternetGateway', 'NatGateway', 'RouteTable'];
            const propTags = toCfnTags(tags);
            const noProp = toCfnTags(noPropTags);
            for (const resource of taggables) {
                assert_1.expect(stack).to(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(propTags)));
                assert_1.expect(stack).notTo(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(noProp)));
            }
            test.done();
        },
        'Subnet Name will propagate to route tables and NATGW'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            for (const subnet of vpc.publicSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::NatGateway', hasTags([tag])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            for (const subnet of vpc.privateSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            test.done();
        },
        'Tags can be added after the Vpc is created with `vpc.tags.setTag(...)`'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const tag = { Key: 'Late', Value: 'Adder' };
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            core_1.Tags.of(vpc).add(tag.Key, tag.Value);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            test.done();
        },
    },
    'subnet selection': {
        'selecting default subnets returns the private ones'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select public subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PUBLIC });
            // THEN
            test.deepEqual(subnetIds, vpc.publicSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select isolated subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.ISOLATED });
            // THEN
            test.deepEqual(subnetIds, vpc.isolatedSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select subnets by name'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ],
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetGroupName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'subnetName is an alias for subnetGroupName (backwards compat)'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ],
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'selecting default subnets in a VPC with only isolated subnets returns the isolateds'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                isolatedSubnetIds: ['iso-1', 'iso-2', 'iso-3'],
                isolatedSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['iso-1', 'iso-2', 'iso-3']);
            test.done();
        },
        'selecting default subnets in a VPC with only public subnets returns the publics'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['pub-1', 'pub-2', 'pub-3']);
            test.done();
        },
        'selecting subnets by name fails if the name is unknown'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            test.throws(() => {
                vpc.selectSubnets({ subnetGroupName: 'Toot' });
            }, /There are no subnet groups with name 'Toot' in this VPC. Available names: Public,Private/);
            test.done();
        },
        'select subnets with az restriction'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
                maxAzs: 1,
                subnetConfiguration: [
                    { name: 'lb', subnetType: lib_1.SubnetType.PUBLIC },
                    { name: 'app', subnetType: lib_1.SubnetType.PRIVATE },
                    { name: 'db', subnetType: lib_1.SubnetType.PRIVATE },
                ],
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ onePerAz: true });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], vpc.privateSubnets[0].subnetId);
            test.done();
        },
        'select explicitly defined subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            const subnet = new lib_1.PrivateSubnet(stack, 'Subnet', {
                availabilityZone: vpc.availabilityZones[0],
                cidrBlock: '10.0.0.0/28',
                vpcId: vpc.vpcId,
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnets: [subnet] });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], subnet.subnetId);
            test.done();
        },
        'subnet created from subnetId'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetId(stack, 'subnet1', 'pub-1');
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            test.done();
        },
        'Referencing AZ throws error when subnet created from subnetId'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetId(stack, 'subnet1', 'pub-1');
            // THEN
            // eslint-disable-next-line max-len
            test.throws(() => subnet.availabilityZone, "You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
            test.done();
        },
        'Referencing AZ throws error when subnet created from attributes without az'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetAttributes(stack, 'subnet1', { subnetId: 'pub-1', availabilityZone: '' });
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            // eslint-disable-next-line max-len
            test.throws(() => subnet.availabilityZone, "You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
            test.done();
        },
        'AZ have value when subnet created from attributes with az'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetAttributes(stack, 'subnet1', { subnetId: 'pub-1', availabilityZone: 'az-1234' });
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            test.deepEqual(subnet.availabilityZone, 'az-1234');
            test.done();
        },
        'Can select subnets by type and AZ'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 3,
            });
            // WHEN
            new lib_1.InterfaceVpcEndpoint(stack, 'VPC Endpoint', {
                vpc,
                privateDnsEnabled: false,
                service: new lib_1.InterfaceVpcEndpointService('com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc', 443),
                subnets: {
                    subnetType: lib_1.SubnetType.PRIVATE,
                    availabilityZones: ['dummy1a', 'dummy1c'],
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                SubnetIds: [
                    {
                        Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3Subnet3EDCD457',
                    },
                ],
            }));
            test.done();
        },
        'SubnetSelection filtered on az uses default subnetType when no subnet type specified'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 3,
            });
            // WHEN
            new lib_1.InterfaceVpcEndpoint(stack, 'VPC Endpoint', {
                vpc,
                service: new lib_1.InterfaceVpcEndpointService('com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc', 443),
                subnets: {
                    availabilityZones: ['dummy1a', 'dummy1c'],
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                SubnetIds: [
                    {
                        Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3Subnet3EDCD457',
                    },
                ],
            }));
            test.done();
        },
        'SubnetSelection doesnt throw error when selecting imported subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // THEN
            test.doesNotThrow(() => vpc.selectSubnets({
                subnets: [
                    lib_1.Subnet.fromSubnetId(stack, 'Subnet', 'sub-1'),
                ],
            }));
            test.done();
        },
    },
});
function getTestStack() {
    return new core_1.Stack(undefined, 'TestStack', { env: { account: '123456789012', region: 'us-east-1' } });
}
function toCfnTags(tags) {
    return Object.keys(tags).map(key => {
        return { Key: key, Value: tags[key] };
    });
}
function hasTags(expectedTags) {
    return (props) => {
        try {
            const tags = props.Tags;
            const actualTags = tags.filter((tag) => {
                for (const expectedTag of expectedTags) {
                    if (assert_1.isSuperObject(expectedTag, tag)) {
                        return true;
                    }
                    else {
                        continue;
                    }
                }
                // no values in array so expecting empty
                return false;
            });
            return actualTags.length === expectedTags.length;
        }
        catch (e) {
            /* eslint-disable no-console */
            console.error('Tags are incorrect');
            console.error('found tags ', props.Tags);
            console.error('expected tags ', expectedTags);
            /* eslint-enable no-console */
            throw e;
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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