"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.synthesizeBlockDeviceMappings = exports.BlockDeviceVolume = void 0;
const crypto = require("crypto");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName,
            ebs,
            virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * Highest-performance SSD volume for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(construct.node.uniqueId));
        return md5.digest('hex');
    }
}
/**
 * Creates a new EBS Volume in AWS EC2.
 */
class Volume extends VolumeBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_b = props.enableMultiAttach) !== null && _b !== void 0 ? _b : false,
            size: (_c = props.size) === null || _c === void 0 ? void 0 : _c.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_e = grant.principalStatement) === null || _e === void 0 ? void 0 : _e.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    validateProps(props) {
        var _a;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.iops) {
            if (props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`/`IO1`');
            }
            if (props.iops < 100 || props.iops > 64000) {
                throw new Error('`iops` must be in the range 100 to 64,000, inclusive.');
            }
            if (props.size && (props.iops > 50 * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error('`iops` has a maximum ratio of 50 IOPS/GiB.');
            }
        }
        if (props.enableMultiAttach && props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
            throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` volumes.');
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce maximum volume size:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html#ebs-volume-characteristics
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1000 };
            const volumeType = (_a = props.volumeType) !== null && _a !== void 0 ? _a : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
//# sourceMappingURL=data:application/json;base64,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