"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ChangeInstanceStateSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implemenation for aws:changeInstanceState
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-changestate.html
 */
class ChangeInstanceStateSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:changeInstanceState';
        this.changeInstanceStateStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        const checkStateOnly = this.changeInstanceStateStep.checkStateOnly?.resolveToBoolean(inputs) ?? false;
        const desiredState = this.changeInstanceStateStep.desiredState.resolveToString(inputs);
        if (!checkStateOnly) {
            console.log(`ChangeInstanceState: Setting instances state to ${desiredState}`);
            this.changeInstanceState(desiredState, inputs);
        }
        console.log(`ChangeInstanceState: Waiting for instances to be ${desiredState}`);
        this.waitForInstanceState(desiredState, inputs);
        console.log(`ChangeInstanceState: Instances are now ${desiredState}`);
        return {};
    }
    changeInstanceState(desiredState, inputs) {
        switch (desiredState) {
            case 'running':
                return this.changeInstanceStateToRunning(inputs);
            case 'stopped':
                return this.changeInstanceStateToStopped(inputs);
            case 'terminated':
                return this.changeInstanceStateToTerminated(inputs);
            default:
                throw new Error(`Desired state: ${desiredState} is not an end-state to change to.`);
        }
    }
    changeInstanceStateToRunning(inputs) {
        this.changeInstanceStateCall('StartInstances', {
            AdditionalInfo: this.changeInstanceStateStep.additionalInfo,
        }, inputs);
    }
    changeInstanceStateToStopped(inputs) {
        this.changeInstanceStateCall('StopInstances', {
            Force: this.changeInstanceStateStep.force,
        }, inputs);
    }
    changeInstanceStateToTerminated(inputs) {
        this.changeInstanceStateCall('TerminateInstances', {}, inputs);
    }
    changeInstanceStateCall(endpoint, moreParams, inputs) {
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord({
            InstanceIds: this.changeInstanceStateStep.instanceIds,
            ...moreParams,
        }, x => x.resolve(inputs));
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'changeInstanceState', {
            service: 'EC2',
            pascalCaseApi: endpoint,
            apiParams,
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Change instance state failed for ${apiParams.InstanceIds}: ${result.stackTrace}`);
        }
    }
    waitForInstanceState(desiredState, inputs) {
        const instanceIds = this.changeInstanceStateStep.instanceIds.resolveToStringList(inputs);
        switch (desiredState) {
            case 'running':
                return this.waitForStateRunningAndStatusOk(instanceIds);
            case 'stopped':
                return this.waitForStateStopped(instanceIds);
            case 'terminated':
                return this.waitForStateTerminated(instanceIds);
            default:
                throw new Error(`Desired state: ${desiredState} is not an end-state to wait for.`);
        }
    }
    waitForStateRunningAndStatusOk(instanceIds) {
        wait_for_and_assert_1.waitForAndAssertInstanceState({
            waitForState: ['shutting-down', 'terminated', 'running'],
            assertState: 'running',
            instanceIds: instanceIds,
        }, this.props);
        // For running, we also check if the state is 'ok'
        wait_for_and_assert_1.waitForAndAssertInstanceStatus({
            waitForStatus: ['ok', 'impaired', 'insufficient-data', 'not-applicable'],
            assertStatus: 'ok',
            instanceIds: instanceIds,
        }, this.props);
    }
    waitForStateStopped(instanceIds) {
        wait_for_and_assert_1.waitForAndAssertInstanceState({
            waitForState: ['shutting-down', 'terminated', 'stopped'],
            assertState: 'stopped',
            instanceIds: instanceIds,
        }, this.props);
    }
    waitForStateTerminated(instanceIds) {
        wait_for_and_assert_1.waitForAndAssertInstanceState({
            waitForState: ['terminated'],
            assertState: 'terminated',
            instanceIds: instanceIds,
        }, this.props);
    }
}
exports.ChangeInstanceStateSimulation = ChangeInstanceStateSimulation;
_a = JSII_RTTI_SYMBOL_1;
ChangeInstanceStateSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.ChangeInstanceStateSimulation", version: "0.0.28" };
//# sourceMappingURL=data:application/json;base64,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