/**
 * IEnvironment is an abstraction for EC2 instances.
 * The implementations of this interface provides customers with
 * alternatives to test their commands rather than a real instance.
 */
export interface IEnvironment {
    run(command: string): string;
}
/**
 * Provides a Docker client against which customers can execute their commands.
 * This utility will not download docker images, rather will create containers from images provided.
 * You can use this rather than running your run commands against a real EC2 machine.
 */
export declare class DockerEnvironment implements IEnvironment {
    /**
     * Use an existing container against which to run commands using the run function.
     */
    static fromContainer(containerId: string): DockerEnvironment;
    /**
     * Create a container from the provided image.
     * The container created will be used by this instance to run commands using the run function.
     */
    static fromImage(image: string): DockerEnvironment;
    private static execute;
    private static removeTrailingNewLine;
    readonly containerId: string;
    private constructor();
    /**
     * Runs commands against the docker specified during construction.
     * This function runs synchronously.
     * @returns the response from the docker as a string.
     */
    run(command: string): string;
    /**
     * Force removes the container associated with this instance.
     */
    removeContainer(): void;
}
/**
 * Environment that simply saves commands into a previousCommands variable.
 * This is useful if you want to unit test the commands that would be sent to a real environment.
 */
export declare class MockEnvironment implements IEnvironment {
    previousCommands: string[];
    run(command: string): string;
}
/**
 * Environment that simply logs the commands that it receives and displays them on the console.
 */
export declare class LoggingEnvironment implements IEnvironment {
    run(command: string): string;
}
