"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DualAlbFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class DualAlbFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this.hasExternalLoadBalancer = false;
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            if (!t.internalOnly) {
                this.hasExternalLoadBalancer = true;
            }
        });
        if (this.hasExternalLoadBalancer) {
            this.externalAlb = new elbv2.ApplicationLoadBalancer(this, 'ExternalAlb', {
                vpc: this.vpc,
                internetFacing: true,
            });
        }
        this.internalAlb = new elbv2.ApplicationLoadBalancer(this, 'InternalAlb', {
            vpc: this.vpc,
            internetFacing: false,
        });
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        props.tasks.forEach(t => {
            var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
            });
            this.service.push(svc);
            // default scaling policy
            const scaling = svc.autoScaleTaskCount({ maxCapacity: (_f = (_e = t.scalingPolicy) === null || _e === void 0 ? void 0 : _e.maxCapacity) !== null && _f !== void 0 ? _f : 10 });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_h = (_g = t.scalingPolicy) === null || _g === void 0 ? void 0 : _g.targetCpuUtilization) !== null && _h !== void 0 ? _h : 50,
            });
            // not internalOnly
            if (!t.internalOnly) {
                const exttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}ExtTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                });
                // listener for the external ALB
                new elbv2.ApplicationListener(this, `ExtAlbListener${t.listenerPort}`, {
                    loadBalancer: this.externalAlb,
                    open: true,
                    port: t.listenerPort,
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    defaultTargetGroups: [exttg],
                });
                scaling.scaleOnRequestCount('RequestScaling', {
                    requestsPerTarget: (_k = (_j = t.scalingPolicy) === null || _j === void 0 ? void 0 : _j.requestPerTarget) !== null && _k !== void 0 ? _k : 1000,
                    targetGroup: exttg,
                });
                exttg.addTarget(svc);
            }
            const inttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}IntTG`, {
                protocol: elbv2.ApplicationProtocol.HTTP,
                vpc: this.vpc,
            });
            // listener for the internal ALB
            new elbv2.ApplicationListener(this, `IntAlbListener${t.listenerPort}`, {
                loadBalancer: this.internalAlb,
                open: true,
                port: t.listenerPort,
                protocol: elbv2.ApplicationProtocol.HTTP,
                defaultTargetGroups: [inttg],
            });
            // extra scaling policy
            scaling.scaleOnRequestCount('RequestScaling2', {
                requestsPerTarget: (_m = (_l = t.scalingPolicy) === null || _l === void 0 ? void 0 : _l.requestPerTarget) !== null && _m !== void 0 ? _m : 1000,
                targetGroup: inttg,
            });
            inttg.addTarget(svc);
        });
        // Route53
        const zoneName = (_d = (_c = props.route53Ops) === null || _c === void 0 ? void 0 : _c.zoneName) !== null && _d !== void 0 ? _d : 'svc.local';
        const externalAlbRecordName = (_f = (_e = props.route53Ops) === null || _e === void 0 ? void 0 : _e.externalAlbRecordName) !== null && _f !== void 0 ? _f : 'external';
        const internalAlbRecordName = (_h = (_g = props.route53Ops) === null || _g === void 0 ? void 0 : _g.internalAlbRecordName) !== null && _h !== void 0 ? _h : 'internal';
        const zone = new route53.PrivateHostedZone(this, 'HostedZone', {
            zoneName,
            vpc: this.vpc,
        });
        new route53.ARecord(this, 'InternalAlbAlias', {
            zone,
            recordName: internalAlbRecordName,
            target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.internalAlb)),
        });
        if (this.externalAlb) {
            new route53.ARecord(this, 'ExternalAlbAlias', {
                zone,
                recordName: externalAlbRecordName,
                target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.externalAlb)),
            });
        }
        if (this.externalAlb) {
            new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
            new cdk.CfnOutput(this, 'ExternalEndpointPrivate', { value: `http://${externalAlbRecordName}.${zoneName}` });
        }
        new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
        new cdk.CfnOutput(this, 'InternalEndpointPrivate', { value: `http://${internalAlbRecordName}.${zoneName}` });
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.DualAlbFargateService = DualAlbFargateService;
_a = JSII_RTTI_SYMBOL_1;
DualAlbFargateService[_a] = { fqn: "cdk-fargate-patterns.DualAlbFargateService", version: "0.0.17" };
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
//# sourceMappingURL=data:application/json;base64,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