"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack2', { env });
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
const nginxTask = new ecs.FargateTaskDefinition(stack, 'nginxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nginxTask.addContainer('nginx', {
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    portMappings: [
        { containerPort: 80 },
    ],
});
const phpTask = new ecs.FargateTaskDefinition(stack, 'phpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
phpTask.addContainer('php', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nginx-php')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// NuxtJS service
const nuxtTask = new ecs.FargateTaskDefinition(stack, 'nuxtTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nuxtTask.addContainer('nuxt', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nuxt')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// Node.js service
const nodeTask = new ecs.FargateTaskDefinition(stack, 'nodeTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nodeTask.addContainer('node', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/node')),
    portMappings: [
        { containerPort: 80 },
    ],
});
const svc = new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            listenerPort: 80,
            task: orderTask,
            desiredCount: 2,
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            internalOnly: true,
            listenerPort: 8080,
            task: customerTask,
            desiredCount: 1,
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        { listenerPort: 9090, task: productTask, desiredCount: 1, internalOnly: true },
        // The nginx service(external/internal)
        { listenerPort: 9091, task: nginxTask, desiredCount: 1 },
        // The NuxtJS service(external/internal)
        { listenerPort: 9092, task: phpTask, desiredCount: 1 },
        { listenerPort: 9093, task: nuxtTask, desiredCount: 1 },
        { listenerPort: 9094, task: nodeTask, desiredCount: 1 },
    ],
    route53Ops: {
        zoneName,
        externalAlbRecordName,
        internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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