# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/03_template.template.ipynb (unless otherwise specified).

__all__ = ['Template', 'BlankTemplate', 'ValidMoleculeTemplate', 'RuleOf5Template', 'GhoseTemplate', 'VeberTemplate',
           'REOSTemplate', 'RuleOf3Template']

# Cell
from ..imports import *
from ..core import *
from ..chem import *
from .filters import *

# Cell

class Template():
    '''
    Template - class for managing hard and soft filters

    Inputs:

    - `hard_filters list[Filter]`: list of `Filter` objects used for pass/fail screening

    - `soft_filters list[Filter]`: list of `Filter` objects used for soft scoring

    - `log bool`: if True, template will log screened compounds

    - `use_lookup bool`: if True, filter results are stored in a lookup table. If a compound
    is re-screened, the lookup value is returned

    - `fail_score float`: placeholder score for compounds that fail to pass hard filters

    - `cpus Optional[int]`: number of CPUs to use. If None, defaults to `os.environ['ncpus']`

    - `mode str['smile', 'protein', 'dna', 'rna']`: determines
    how inputs are converted to Mol objects
    '''
    def __init__(self, hard_filters, soft_filters=None, log=False, use_lookup=False, fail_score=0.,
                cpus=None, mode='smile'):

        if soft_filters is None:
            soft_filters = []

        self.hard_filters = hard_filters
        self.soft_filters = soft_filters
        self.log = log
        self.use_lookup = use_lookup
        self.fail_score = fail_score
        self.mode = mode

        for f in self.hard_filters:
            f.mode = self.mode

        for f in self.soft_filters:
            f.mode = self.mode

        self.hard_log = pd.DataFrame(columns=['smiles']+list(range(len(self.hard_filters)))+['final'])
        self.hard_col_names = ['smiles'] + [i.name for i in self.hard_filters] + ['final']
        self.hard_lookup = {}

        self.soft_log = pd.DataFrame(columns=['smiles']+list(range(len(self.soft_filters)))+['final'])
        self.soft_col_names = ['smiles'] + [i.name for i in self.soft_filters] + ['final']
        self.soft_lookup = {}
        self.cpus = cpus

    def __call__(self, mols, filter_type='hard'):

        if filter_type=='hard':
            outputs = maybe_parallel(self.hf, mols, cpus=self.cpus)
        else:
            outputs = maybe_parallel(self.sf, mols, cpus=self.cpus)

        if is_container(mols):
            return_outputs = [i[0] for i in outputs]
            log_outputs = [i[1] for i in outputs if i[1]]
        else:
            return_outputs = outputs[0]
            log_outputs = [outputs[1]] if outputs[1] else []

        self.log_data(log_outputs, filter_type=filter_type)
        self.clean_logs()

        return return_outputs

    def to_mol(self, input):
        if self.mode=='smile':
            mol = to_mol(input)
        elif self.mode=='protein':
            mol = to_protein(input)
        elif self.mode=='dna':
            mol = to_dna(input)
        elif self.mode=='rna':
            mol = to_rna(input)
        else:
            raise ValueError("`self.mode` must be one of `['smile', 'protein', 'dna', 'rna']`")

        return mol

    def to_string(self, input):
        if self.mode=='smile':
            string = to_smile(input)
        elif self.mode in set(['protein', 'dna', 'rna']):
            string = to_sequence(input)
        else:
            raise ValueError("`self.mode` must be one of `['smile', 'protein', 'dna', 'rna']`")

        return string

    def standardize(self, smiles, cpus=None):
        'Canonicalize/standardize smiles'
        if cpus is None:
            cpus = self.cpus

        if self.mode=='smile':
            # canonicalize smiles
            mols = maybe_parallel(self.to_mol, smiles, cpus=cpus)
            strings = maybe_parallel(self.to_string, mols, cpus=cpus)
        else:
            # other modaalities are sequences, don't bother
            strings = smiles
        return strings

    def validate(self, smiles, cpus=None):
        'Validate `smiles` (ie valid structure), should not be related to filters'
        if cpus is None:
            cpus = self.cpus
        mols = maybe_parallel(self.to_mol, smiles, cpus=cpus)

        return [i is not None for i in mols]

    def eval_mols(self, mols):
        hardpass = self.__call__(mols, filter_type='hard')

        remaining = []
        idxs = []
        scores = []
        for i, mol in enumerate(mols):
            if hardpass[i]:
                remaining.append(mol)
                idxs.append(i)
                scores.append(0)
            else:
                scores.append(self.fail_score)

        if remaining:
            soft_scores = self.__call__(remaining, filter_type='soft')
            for i, score in enumerate(soft_scores):
                idx = idxs[i]
                scores[idx] = score

        return scores

    def hf(self, mol, agg=True):
        'run hard filters'
        mol = self.to_mol(mol)
        smile = self.to_string(mol)

        if self.use_lookup and smile in self.hard_lookup.keys():
            output = self.hard_lookup[smile]
            log_data = []

        else:
            filter_results = []
            for filt in self.hard_filters:
                try:
                    filter_results.append(filt(mol, with_score=False))
                except:
                    filter_results.append(False)

            if agg:
                output = all(filter_results)
                log_data = [smile]+filter_results+[output]

            else:
                output = filter_results
                log_data = []

        return output, log_data

    def sf(self, mol):
        'run soft filters'
        mol = self.to_mol(mol)
        smile = self.to_string(mol)

        if self.use_lookup and smile in self.soft_lookup.keys():
            output = self.soft_lookup[smile]
            log_data = []

        else:
            filter_results = []
            for filt in self.soft_filters:
                filter_results.append(filt(mol, with_score=True))

            output = sum(filter_results)
            log_data = [smile]+filter_results+[output]

        return output, log_data

    def screen_mols(self, mols):
        'separate `mols` into passes and failures'
        hardpasses = self.__call__(mols, filter_type='hard')

        fails = []
        remaining = []
        remaining_idxs = []

        for i in range(len(hardpasses)):
            if hardpasses[i]:
                remaining.append(mols[i])
                remaining_idxs.append(i)
            else:
                fails.append(mols[i])

        passes = []
        if remaining:
            softpasses = self.__call__(remaining, filter_type='soft')
            passes = list(zip(remaining, softpasses, remaining_idxs))

        return [passes, fails]

    def log_data(self, new_data, filter_type='hard'):

        if self.log and new_data:
            if filter_type=='hard':
                new_df = pd.DataFrame(new_data, columns=self.hard_log.columns)
                self.hard_log = self.hard_log.append(new_df)

            if filter_type=='soft':
                new_df = pd.DataFrame(new_data, columns=self.soft_log.columns)
                self.soft_log = self.soft_log.append(new_df)

        if self.use_lookup and new_data:
            for item in new_data:
                smile = item[0]
                score = item[-1]

                if filter_type=='hard' and not smile in self.hard_lookup.keys():
                    self.hard_lookup[smile] = score

                if filter_type=='soft' and not smile in self.soft_lookup.keys():
                    self.soft_lookup[smile] = score

    def clean_logs(self):
        'de-duplicate logs'
        self.hard_log.drop_duplicates(subset='smiles', inplace=True)
        self.hard_log.reset_index(inplace=True, drop=True)
        self.soft_log.drop_duplicates(subset='smiles', inplace=True)
        self.soft_log.reset_index(inplace=True, drop=True)

    def clear_data(self):
        'delete logged data'
        self.hard_log = pd.DataFrame(columns=['smiles']+list(range(len(self.hard_filters)))+['final'])
        self.hard_lookup = {}

        self.soft_log = pd.DataFrame(columns=['smiles']+list(range(len(self.soft_filters)))+['final'])
        self.soft_lookup = {}

    def sample(self, n, log='hard', seed=None):
        'sample logged data'

        if seed is None:
            seed = np.random.seed() # required to prevent identical sampling in multiprocessing

        if log=='hard':
            to_sample = self.hard_log[self.hard_log.final==True]
            sample = to_sample.sample(n, replace=False, random_state=seed).reset_index(drop=True)
        else:
            sample = self.soft_log.sample(n, replace=False, random_state=seed).reset_index(drop=True)

        return sample[['smiles', 'final']]


    def sample_smiles(self, n, log='hard'):
        return list(self.sample(n, log=log).smiles.values)

    def save(self, filename, with_data=True):
        '''
        save - save `Template` object

        Inputs
            'filename' - str, save filename
            `with_data` - bool, if True Template is saved with logged data
        '''
        if not with_data:
            hard_log = self.hard_log
            hard_lookup = self.hard_lookup

            soft_log = self.soft_log
            soft_lookup = self.soft_lookup

            self.clear_data()

        with open(filename, 'wb') as f:
            pickle.dump(self, f)

        if not with_data:
            self.hard_log = hard_log
            self.hard_lookup = hard_lookup

            self.soft_log = soft_log
            self.soft_lookup = soft_lookup

    @classmethod
    def from_file(cls, filename):
        'load template from file'
        template = pickle.load(open(filename, 'rb'))
        return template

    def __add__(self, other, merge_data=True):
        'merge two templates. If `merge_data`, logged data from each template is rescreened'
        hard_filters = self.hard_filters + other.hard_filters
        hard_filters = sorted(hard_filters, key=lambda x: x.priority, reverse=True)

        soft_filters = self.soft_filters + other.soft_filters
        soft_filters = sorted(soft_filters, key=lambda x: x.priority, reverse=True)

        new_template = Template(hard_filters, soft_filters, use_lookup=self.use_lookup)

        if merge_data:
            soft_smiles = list(self.soft_log.smiles.values) + list(other.soft_log.smiles.values)
            soft_smiles = list(set(soft_smiles))

            hard_smiles = list(self.hard_log.smiles.values) + list(other.hard_log.smiles.values)
            hard_smiles = list(set(hard_smiles))

#             new_template = Template(hard_filters, soft_filters, use_lookup=self.use_lookup)
            _ = new_template(hard_smiles, filter_type='hard')
            _ = new_template(soft_smiles, filter_type='soft')

        return new_template

    def __repr__(self):
        hf = 'Hard Filter:\n\t\t' + '\n\t\t'.join([i.__repr__() for i in self.hard_filters])
        sf = 'Soft Filter:\n\t\t' + '\n\t\t'.join([i.__repr__() for i in self.soft_filters])
        rep_str = 'Template\n\t' + hf + '\n\t' + sf
        return rep_str

# Cell

class BlankTemplate(Template):
    "Empty template (no hard or soft filters)"
    def __init__(self, **template_kwargs):
        super().__init__([],[], **template_kwargs)

class ValidMoleculeTemplate(Template):
    'Template for checking if an input is a single valid chemical structure'
    def __init__(self, hard=True, soft=False, **template_kwargs):

        if hard:
            hard_filters = [
                ValidityFilter(),
                SingleCompoundFilter()
            ]
        else:
            hard_filters = []

        if soft:
            soft_fiters = [
                ValidityFilter(score=1),
                SingleCompoundFilter(score=1)
            ]
        else:
            soft_filters = []

        super().__init__(hard_filters, soft_filters, **template_kwargs)

class RuleOf5Template(Template):
    "Template for Lipinski's rule of 5 (en.wikipedia.org/wiki/Lipinski%27s_rule_of_five)"
    def __init__(self, hard=True, soft=False, **template_kwargs):

        if hard:
            hard_filters = [
                HBDFilter(None, 5),
                HBAFilter(None, 10),
                MolWtFilter(None, 500),
                LogPFilter(None, 5)
            ]
        else:
            hard_filters = []

        if soft:
            soft_filters = [
                HBDFilter(None, 5, score=1),
                HBAFilter(None, 10, score=1),
                MolWtFilter(None, 500, score=1),
                LogPFilter(None, 5, score=1)
            ]
        else:
            soft_filters = []

        super().__init__(hard_filters, soft_filters, **template_kwargs)

class GhoseTemplate(Template):
    "Template for Ghose filters (doi.org/10.1021/cc9800071)"
    def __init__(self, hard=True, soft=False, **template_kwargs):

        if hard:
            hard_filters = [
                MolWtFilter(160, 480),
                LogPFilter(-0.4, 5.6),
                HeavyAtomsFilter(20, 70),
                MRFilter(40, 130)
            ]
        else:
            hard_filters = []

        if soft:
            soft_filters = [
                MolWtFilter(160, 480, score=1),
                LogPFilter(-0.4, 5.6, score=1),
                HeavyAtomsFilter(20, 70, score=1),
                MRFilter(40, 130, score=1)
            ]
        else:
            soft_filters = []

        super().__init__(hard_filters, soft_filters, **template_kwargs)

class VeberTemplate(Template):
    "Template for Veber filters (doi.org/10.1021/jm020017n)"
    def __init__(self, hard=True, soft=False, **template_kwargs):

        if hard:
            hard_filters = [
                RotBondFilter(None, 10),
                TPSAFilter(none, 140)
            ]
        else:
            hard_filters = []

        if soft:
            soft_filters = [
                RotBondFilter(None, 10, score=1),
                TPSAFilter(none, 140, score=1)
            ]
        else:
            soft_filters = []

        super().__init__(hard_filters, soft_filters, **template_kwargs)

class REOSTemplate(Template):
    "Template for REOS filters (10.1016/s0169-409x(02)00003-0)"
    def __init__(self, hard=True, soft=False, **template_kwargs):

        if hard:
            hard_filters = [
                MolWtFilter(200, 500),
                LogPFilter(-5, 5),
                HBDFilter(0, 5),
                HBAFilter(0, 10),
                ChargeFilter(-2, 2),
                RotBondFilter(0, 8),
                HeavyAtomsFilter(15, 50)
            ]
        else:
            hard_filters = []

        if soft:
            soft_filters = [
                MolWtFilter(200, 500, score=1),
                LogPFilter(-5, 5, score=1),
                HBDFilter(0, 5, score=1),
                HBAFilter(0, 10, score=1),
                ChargeFilter(-2, 2, score=1),
                RotBondFilter(0, 8, score=1),
                HeavyAtomsFilter(15, 50, score=1)
            ]
        else:
            soft_filters = []

        super().__init__(hard_filters, soft_filters, **template_kwargs)

class RuleOf3Template(Template):
    "Template for rule of 5 filter (doi.org/10.1016/S1359-6446(03)02831-9)"
    def __init__(self, hard=True, soft=False, **template_kwargs):

        if hard:
            hard_filters = [
                MolWtFilter(None, 300),
                LogPFilter(None, 3),
                HBDFilter(None, 3),
                HBAFilter(None, 3),
                RotBondFilter(None, 3)
            ]
        else:
            hard_filters = []

        if soft:
            soft_filters = [
                MolWtFilter(None, 300, score=1),
                LogPFilter(None, 3, score=1),
                HBDFilter(None, 3, score=1),
                HBAFilter(None, 3, score=1),
                RotBondFilter(None, 3, score=1)
            ]
        else:
            soft_filters = []

        super().__init__(hard_filters, soft_filters, **template_kwargs)
