# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fau_colors']

package_data = \
{'': ['*']}

install_requires = \
['matplotlib>=3.4.3', 'seaborn>=0.11.2', 'typing-extensions>=3.7']

setup_kwargs = {
    'name': 'fau-colors',
    'version': '1.5.2',
    'description': 'The official colors of FAU as matplotlib/seaborn colormaps',
    'long_description': '# FAU - Colors\n\n[![PyPI](https://img.shields.io/pypi/v/fau-colors)](https://pypi.org/project/fau-colors/)\n![GitHub](https://img.shields.io/github/license/mad-lab-fau/fau_colors)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n![PyPI - Downloads](https://img.shields.io/pypi/dm/fau-colors)\n![GitHub commit activity](https://img.shields.io/github/commit-activity/m/mad-lab-fau/fau_colors)\n\n\nThe official colors of Friedrich-Alexander-Universität Erlangen-Nürnberg (FAU) as \n[matplotlib](https://matplotlib.org/) / [seaborn](https://seaborn.pydata.org/) colormaps and GIMP colopalette\n\nWe support the old colors based on the \n2019 CI-guidelines <!-- formerly available on https://www.intern.fau.de/files/2020/03/FAU-Design-Manual.pdf --> and the brand new\n[2021 Brand redesign](https://www.intern.fau.de/kommunikation-marketing-und-corporate-identity/corporate-identity/).\n\n## Download the colors\n\n### Gimp/Inkscape\nIf you just need the colors outside Python, you can download a GIMP colorpalette with all colors (see image below).\nRight click any of the links below and then select "Save Link as":\n\n<a href="https://raw.githubusercontent.com/mad-lab-fau/fau_colors/main/color_palettes/fau_colors_2019.gpl" download>2019 colors</a>  \n<a href="https://raw.githubusercontent.com/mad-lab-fau/fau_colors/main/color_palettes/fau_colors_2021.gpl" download>2021 colors</a>\n\nThese files can be imported into GIMP or Inkscape.\n\n### Latex\n\nWe also provide a LaTeX color file for the 2021 colors. You can download it here:\n\n<a href="https://raw.githubusercontent.com/mad-lab-fau/fau_colors/main/color_palettes/fau_colors_2021.tex" download>2021 colors</a>\n\nFor this to work you need to include the xcolor package in your preamble (`\\usepackage{xcolor}`) and then you can use \nthe colors by copying the file into your project and including it in your document (`\\input{fau_colors_2021.tex}`).\n\nIf you need the colors in other formats, please open an issue with details about the required file format.\n\n## Installation\n\n```bash\npip install fau-colors\n```\n\n## Quick Guide\n\n#### 2021 colormaps\n\n![2021 colors](_docs/cms_21.png)\n\n```python\nimport seaborn as sns\n\nfrom fau_colors import register_cmaps\nregister_cmaps()\n\nsns.set_palette("tech")\n```\n\n#### 2019 colormaps\n\n![2019 colors](_docs/cms_19.png)\n\n```python\nimport seaborn as sns\n\nfrom fau_colors.v2019 import register_cmaps\nregister_cmaps()\n\nsns.set_palette("tech")\n```\n\n## General Usage\n\nThe 2019 and the 2021 colors are available in the separate submodules `fau_colors.v2019` and `fau_colors.v2021` that \ncontain equivalent functions.\n\n**Note:** For convenience, the `v2021` colors can also be accessed from the top-level. In the following examples we\nwill use this shorter notation.\n\nThe methods below show the usage with the new color scheme.\nFor the old colors simply replace the module name.\n\n### Registering color palettes\n\nThe easiest way to use the provided color palettes is to register them as global matplotlib colormaps.\nThis can be done by calling the `register_cmaps()` function from the respective submodule.\nAll available cmaps can be seen in the images above.\n\n#### 2021 colors\n```pycon\n>>> from fau_colors import register_cmaps  # v2021 colors\n>>> register_cmaps()\n```\n\n#### 2019 colors\n```pycon\n>>> from fau_colors.v2019 import register_cmaps\n>>> register_cmaps()\n```\n\n**WARNING: The 2019 and 2021 cmaps have overlapping names! This means you can not register both at the same time.\nYou need to call `unregister_cmaps` from the correct module first, before you can register the other colormaps.\nIf you need colormaps from both CI-guides, use them individually, as shown below.**\n\n\n### Getting the raw colors\n\nAll primary faculty colors are stored in a `namedtuple` called `colors`.\n\n\n#### 2021 colors\n```pycon\n>>> from fau_colors import colors  # v2021 colors\n>>> colors\nFacultyColors(fau=\'#002F6C\', tech=\'#779FB5\', phil=\'#FFB81C\', med=\'#00A3E0\', nat=\'#43B02A\', wiso=\'#C8102E\')\n>>> colors.fau\n\'#002F6C\'\n```\n\n#### 2019 colors\n```pycon\n>>> from fau_colors.v2019 import colors\n>>> colors\nFacultyColors(fau=\'#003865\', tech=\'#98a4ae\', phil=\'#c99313\', med=\'#00b1eb\', nat=\'#009b77\', wiso=\'#8d1429\')\n>>> colors.fau\n\'#003865\'\n```\n\nFor the 2021 color scheme also the variable `colors_dark`, `colors_light`, and `colors_all` are available. They \ncontain the dark and light variants of each color, as well as regular, dark, and light colors combined, respectively.\n\n### Manually getting the colormaps\n\nThe colormaps are stored in a `namedtuple` called `cmaps`.\nThere are colormaps for the primary colors and colormaps with varying lightness using each color as the base color.\nThe latter colormaps contain 5 colors each with 12.5, 25, 37.5, 62.5, and 100% value of the base color.\nIf you need more than 5 colors see below.\n\n#### 2021 colors\n```pycon\n>>> from fau_colors import cmaps  # v2021 colors\n>>> # Only get the names here\n>>> cmaps._fields\n(\'faculties\', \'faculties_dark\', \'faculties_light\', \'faculties_all\', \'fau\', \'fau_dark\', \'fau_light\', \'tech\', \'tech_dark\', \'tech_light\', \'phil\', \'phil_dark\', \'phil_light\', \'med\', \'med_dark\', \'med_light\', \'nat\', \'nat_dark\', \'nat_light\', \'wiso\', \'wiso_dark\', \'wiso_light\')\n>>> cmaps.fau_dark\n[(0.01568627450980392, 0.11764705882352941, 0.25882352941176473), (0.3823913879277201, 0.4463667820069205, 0.5349480968858131), (0.629434832756632, 0.6678200692041523, 0.7209688581314879), (0.7529565551710881, 0.7785467128027682, 0.8139792387543252), (0.876478277585544, 0.889273356401384, 0.9069896193771626)]\n>>> import seaborn as sns\n>>> sns.set_palette(cmaps.fau_dark)\n```\n\n\n#### 2019 colors\n```pycon\n>>> from fau_colors.v2019 import cmaps\n>>> # Only get the names here\n>>> cmaps._fields\n(\'faculties\', \'fau\', \'tech\', \'phil\', \'med\', \'nat\', \'wiso\')\n>>> cmaps.fau\n[(0.0, 0.2196078431372549, 0.396078431372549), (0.37254901960784315, 0.5103421760861206, 0.6210688196847366), (0.6235294117647059, 0.7062053056516724, 0.772641291810842), (0.7490196078431373, 0.8041368704344483, 0.8484275278738946), (0.8745098039215686, 0.9020684352172241, 0.9242137639369473)]\n>>> import seaborn as sns\n>>> sns.set_palette(cmaps.fau)\n```\n\n### Modifying the colormaps\n\nSometimes five colors are not enough for a colormap.\nThe easiest way to generate more colors is to use one of the FAU colors as base and then create custom sequential\npalettes from it.\nThis can be done using `sns.light_palette` or `sns.dark_palette`, as explained \n[here](https://seaborn.pydata.org/tutorial/color_palettes.html#custom-sequential-palettes).\n\n#### 2021 colors\n```pycon\n>>> from fau_colors import colors  # v2021 colors\n>>> import seaborn as sns\n>>> sns.light_palette(colors.med, n_colors=8)\n[(0.9370639121761148, 0.9445189791516921, 0.9520035391049294), (0.8047725363394869, 0.9014173378043252, 0.9416168802970363), (0.6688064000629526, 0.8571184286417537, 0.9309417031889239), (0.5365150242263246, 0.8140167872943868, 0.9205550443810308), (0.40054888794979027, 0.7697178781318151, 0.9098798672729183), (0.2682575121131623, 0.7266162367844482, 0.8994932084650251), (0.13229137583662798, 0.6823173276218767, 0.8888180313569127), (0.0, 0.6392156862745098, 0.8784313725490196)]\n```\n\n#### 2019 colors\n```pycon\n>>> from fau_colors.v2019 import colors\n>>> import seaborn as sns\n>>> sns.light_palette(colors.med, n_colors=8)\n[(0.9363137612705862, 0.94473936725293, 0.9520047198366567), (0.8041282890912094, 0.9093574773431737, 0.9477078597351495), (0.6682709982401831, 0.8729927571581465, 0.9432916424086003), (0.5360855260608062, 0.8376108672483904, 0.9389947823070931), (0.40022823520978, 0.8012461470633632, 0.9345785649805439), (0.2680427630304031, 0.765864257153607, 0.9302817048790367), (0.13218547217937693, 0.7294995369685797, 0.9258654875524875), (0.0, 0.6941176470588235, 0.9215686274509803)]c\n```\n',
    'author': 'Robert Richer',
    'author_email': 'robert.richer@fau.de',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/mad-lab-fau/fau_colors',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
