# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['rofi_menu', 'rofi_menu.contrib']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'rofi-menu',
    'version': '0.4.0',
    'description': 'Create rofi menus via python',
    'long_description': '<p align="center">\n    <a href="https://pypi.org/project/rofi-menu/">\n        <img src="https://badge.fury.io/py/rofi-menu.svg" alt="Package version">\n    </a>\n</p>\n\n# rofi-menu\n\nRofi allows defining custom modes ([see the spec](https://github.com/davatorium/rofi/wiki/mode-Specs)).\n\nThis lib is a reference implementation with some extra "sugar".\n\nFeatures:\n\n- simple menu definition via python\n- extendable\n- async in first place\n- allows keeping state during rofi session\n\nSimple demo:\n\n![custom menu](https://github.com/miphreal/python-rofi-menu/raw/master/docs/demo.gif)\n\n## Requirements\n\n- rofi >= 1.5.4\n- python >= 3.6\n\n\n## Installation\n\nUsing pip\n\n```sh\n$ pip install rofi-menu\n```\n\n## Example\n\nCreate a python script which will be used for rofi mode\ne.g. `example.py` (don\'t forget to mark it as executable -- `chmod +x ./example.py`)\n\nAssuming you installed `rofi-menu` into a virtual environment (let\'s say it\'s `~/.pyenv/versions/rofi/bin/python`).\nMake sure shebang points to the right python executable, e.g. `#!/home/user/pyenv/versions/rofi/bin/python`.\n\n```python\n#!/home/user/pyenv/versions/rofi/bin/python\nimport rofi_menu\n\n\nclass ProjectsMenu(rofi_menu.Menu):\n    prompt = "Projects"\n    items = [\n        rofi_menu.BackItem(),\n        rofi_menu.ShellItem("Project 1", "code-insiders ~/Develop/project1"),\n        rofi_menu.ShellItem("Project 2", "code-insiders ~/Develop/project2"),\n        rofi_menu.ShellItem("Project X", "code-insiders ~/Develop/projectx"),\n    ]\n\n\nclass LogoutMenu(rofi_menu.Menu):\n    prompt = "Logout"\n    items = [\n        rofi_menu.ShellItem("Yes", "i3-msg exit", flags={rofi_menu.FLAG_STYLE_URGENT}),\n        rofi_menu.ExitItem("No", flags={rofi_menu.FLAG_STYLE_ACTIVE}),\n    ]\n\n\nclass MainMenu(rofi_menu.Menu):\n    prompt = "menu"\n    items = [\n        rofi_menu.TouchpadItem(),\n        rofi_menu.NestedMenu("Projects >", ProjectsMenu()),\n        rofi_menu.ShellItem(\n            "Downloads (show size)", "du -csh ~/Downloads", show_output=True\n        ),\n        rofi_menu.NestedMenu("Second monitor", rofi_menu.SecondMonitorMenu()),\n        rofi_menu.ShellItem("Lock screen", "i3lock -i ~/.config/i3/bg.png"),\n        rofi_menu.ShellItem("Sleep", "systemctl suspend"),\n        rofi_menu.NestedMenu("Logout", LogoutMenu()),\n    ]\n\n\nif __name__ == "__main__":\n    rofi_menu.run(MainMenu())\n```\n\nRun it as:\n\n```sh\n$ rofi -modi mymenu:/path/to/example.py -show mymenu\n```\n\nIt\'ll result in\n\n![rofi menu](https://github.com/miphreal/python-rofi-menu/raw/master/docs/menu-example.png)\n\n\n### Advanced example\n\n\n```\n#!/home/user/pyenv/versions/rofi/bin/python\nimport asyncio\nfrom datetime import datetime\nimport os\n\nimport rofi_menu\n\n\nclass OutputSomeTextItem(rofi_menu.Item):\n    """Output arbitrary text on selection"""\n    async def on_select(self, meta):\n        # any python code\n        await asyncio.sleep(0.1)\n        return rofi_menu.Operation(rofi_menu.OP_OUTPUT, (\n            "💢 simple\\n"\n            "💥 multi-\\n"\n            "💫 <b>line</b>\\n"\n            "💣 <i>text</i>\\n"\n        ))\n\n\nclass DoAndExitItem(rofi_menu.Item):\n    """Do something and exit"""\n    async def on_select(self, meta):\n        os.system(\'notify-send msg\')\n        return rofi_menu.Operation(rofi_menu.OP_EXIT)\n\n\nclass CurrentDatetimeItem(rofi_menu.Item):\n    """Show current datetime inside menu item"""\n    async def load(self, meta):\n        self.state = datetime.now().strftime(\'%A %d. %B %Y (%H:%M:%S)\')\n\n    async def render(self, meta):\n        return f"🕑 {self.state}"\n\n\nclass CounterItem(rofi_menu.Item):\n    """Increment counter on selection"""\n    async def load(self, meta):\n        await super().load(meta)\n        self.state = self.state or 0\n        meta.session.setdefault("counter_total", 0)\n\n    async def on_select(self, meta):\n        self.state += 1\n        meta.session["counter_total"] += 1\n        return await super().on_select(meta)\n\n    async def render(self, meta):\n        per_menu_item = self.state\n        total = meta.session["counter_total"]\n        return f"🏃 Selected #{per_menu_item} time(s) (across menu items #{total})"\n\n\nclass HandleUserInputMenu(rofi_menu.Menu):\n    class CustomItem(rofi_menu.Item):\n        async def render(self, meta):\n            entered_text = meta.session.get("text", "[ no text ]")\n            return f"You entered: {entered_text}"\n\n    items = [CustomItem()]\n\n    async def on_user_input(self, meta):\n        meta.session[\'text\'] = meta.user_input\n        return rofi_menu.Operation(rofi_menu.OP_REFRESH_MENU)\n\n\n\nmain_menu = rofi_menu.Menu(\n    prompt="menu",\n    items=[\n        OutputSomeTextItem("Output anything"),\n        DoAndExitItem("Do something and exit"),\n        CurrentDatetimeItem(),\n        CounterItem(),\n        CounterItem(),\n        rofi_menu.NestedMenu("User input", HandleUserInputMenu()),\n    ],\n)\n\n\nif __name__ == "__main__":\n    rofi_menu.run(main_menu)\n```\n\n![advanced example](https://github.com/miphreal/python-rofi-menu/raw/master/docs/menu-example-advanced.png)\n\n## TODO\n\n- [ ] documentation of API\n- [ ] examples\n- [ ] tests\n- [ ] `nonselectable` and other new\n- [ ] check what ROFI_* envs can do\n',
    'author': 'miphreal',
    'author_email': 'miphreal@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/miphreal/python-rofi-menu',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
