/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
import { App, AppProps, Stack, StageSynthesisOptions } from "aws-cdk-lib";
import { CloudAssembly } from "aws-cdk-lib/cx-api";
import { NagPackSuppression } from "cdk-nag";
/**
 * Message instance.
 */
export interface Message {
    /**
     * Message description.
     */
    readonly messageDescription: string;
    /**
     * Message type as returned from cdk-nag.
     */
    readonly messageType: string;
}
/**
 * Nag result.
 */
export interface NagResult {
    /**
     * Resource which triggered the message.
     */
    readonly resource: string;
    /**
     * List of messages.
     */
    readonly messages: Message[];
}
/**
 * @inheritDoc
 */
export interface PDKNagAppProps extends AppProps {
    /**
     * Determines whether any errors encountered should trigger a test failure.
     *
     * @default false
     */
    readonly failOnError?: boolean;
    /**
     * Determines whether any warnings encountered should trigger a test failure.
     *
     * @default false
     */
    readonly failOnWarning?: boolean;
}
/**
 * @inheritDoc
 */
export declare class PDKNagApp extends App {
    private readonly _nagResults;
    private readonly failOnError;
    private readonly failOnWarning;
    constructor(props?: PDKNagAppProps);
    synth(options?: StageSynthesisOptions): CloudAssembly;
    addNagResult(result: NagResult): void;
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults(): NagResult[];
}
/**
 * Helper for create a Nag Enabled App.
 */
export declare class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props?: PDKNagAppProps): PDKNagApp;
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack: Stack, path: string, suppressions: NagPackSuppression[], applyToChildren?: boolean): void;
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack: Stack): string;
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack: Stack): string;
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack: Stack): string;
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack: Stack): string;
}
