"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack2', { env });
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
// order service
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        WAREHOUSE_SVC_URL: `${internalALBEndpoint}:9092/api.php`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
// customer service
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
// product service
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
// nginx service
const nginxTask = new ecs.FargateTaskDefinition(stack, 'nginxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nginxTask.addContainer('nginx', {
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    portMappings: [
        { containerPort: 80 },
    ],
});
// php service
const phpTask = new ecs.FargateTaskDefinition(stack, 'phpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
phpTask.addContainer('php', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nginx-php')),
    portMappings: [
        { containerPort: 80 },
    ],
    environment: {
        serviceName: 'warehouse',
        versionNum: '0.1',
    },
});
// laravel-nginx-php-fpm service
const laravelNginxPhpFpmTask = new ecs.FargateTaskDefinition(stack, 'laravelNginxPhpFpmTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelNginxPhpFpmTask.addContainer('laravel-nginx-php-fpm', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-nginx-php-fpm')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// laravel-bitnami service
const laravelBitnamiTask = new ecs.FargateTaskDefinition(stack, 'laravelBitnamiTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelBitnamiTask.addContainer('laravel-bitnami', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-bitnami')),
    portMappings: [
        { containerPort: 3000 },
    ],
});
// NuxtJS service
const nuxtTask = new ecs.FargateTaskDefinition(stack, 'nuxtTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nuxtTask.addContainer('nuxt', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nuxt')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// Node.js service
const nodeTask = new ecs.FargateTaskDefinition(stack, 'nodeTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nodeTask.addContainer('node', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/node')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// java spring boot service
const javaTask = new ecs.FargateTaskDefinition(stack, 'javaTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
javaTask.addContainer('java', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/java-spring-boot')),
    portMappings: [
        { containerPort: 8080 },
    ],
});
// apache-php service
const apachePHP = new ecs.FargateTaskDefinition(stack, 'apachephpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
apachePHP.addContainer('apachephp', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/apache-php')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// python-flask service
const flaskTask = new ecs.FargateTaskDefinition(stack, 'flaskTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
flaskTask.addContainer('flask', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/python-flask')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// gorilla-mux
const muxTask = new ecs.FargateTaskDefinition(stack, 'muxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
muxTask.addContainer('mux', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/gorilla-mux')),
    portMappings: [
        { containerPort: 8080 },
    ],
});
// go-grpc
const gogrpc = new ecs.FargateTaskDefinition(stack, 'gogrpcTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
gogrpc.addContainer('gogrpc', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/go-grpc')),
    portMappings: [
        { containerPort: 50051 },
    ],
});
const certArn = stack.node.tryGetContext('ACM_CERT_ARN');
const cert = certArn ? acm.Certificate.fromCertificateArn(stack, 'Cert', certArn) : undefined;
const svc = new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            task: orderTask,
            desiredCount: 2,
            internal: { port: 80 },
            external: cert ? { port: 443, certificate: [cert] } : { port: 80 },
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            task: customerTask,
            desiredCount: 1,
            internal: { port: 8080 },
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        {
            task: productTask,
            desiredCount: 1,
            internal: { port: 9090 },
        },
        // The nginx service(external only)
        {
            task: nginxTask,
            desiredCount: 1,
            external: { port: 9091 },
        },
        // The nginx-php-fpm service(external/internal)
        {
            task: phpTask,
            desiredCount: 1,
            internal: { port: 9092 },
            external: { port: 9092 },
        },
        // The NuxtJS service(external/internal)
        {
            task: nuxtTask,
            desiredCount: 1,
            internal: { port: 9093 },
            external: { port: 9093 },
        },
        // The node service(external/internal)
        {
            task: nodeTask,
            desiredCount: 1,
            internal: { port: 9094 },
            external: { port: 9094 },
        },
        // The laravel-nginx-php-fpm service(external/internal)
        {
            task: laravelNginxPhpFpmTask,
            desiredCount: 1,
            internal: { port: 9095 },
            external: { port: 9095 },
        },
        // The laravel-bitnami service(external/internal)
        {
            task: laravelBitnamiTask,
            desiredCount: 1,
            internal: { port: 9096 },
            external: { port: 9096 },
        },
        // java spring boot service(external/internal)
        {
            task: javaTask,
            desiredCount: 1,
            internal: { port: 9097 },
            external: { port: 9097 },
            healthCheck: { path: '/hello-world' },
        },
        // apache-php(external/internal)
        {
            task: apachePHP,
            desiredCount: 1,
            internal: { port: 9098 },
            external: { port: 9098 },
        },
        // python-flask(external/internal)
        {
            task: flaskTask,
            desiredCount: 1,
            internal: { port: 9099 },
            external: { port: 9099 },
        },
        // gorilla-mux(external/internal)
        {
            task: muxTask,
            desiredCount: 1,
            internal: { port: 9100 },
            external: { port: 9100 },
        },
        // go-grpc(external/internal)
        {
            task: gogrpc,
            desiredCount: 1,
            internal: cert ? { port: 50051, certificate: [cert] } : { port: 80 },
            external: cert ? { port: 50051, certificate: [cert] } : { port: 80 },
            protocolVersion: elbv2.ApplicationProtocolVersion.GRPC,
            healthCheck: {
                healthyGrpcCodes: '12',
            },
        },
    ],
    route53Ops: {
        zoneName,
        externalElbRecordName: externalAlbRecordName,
        internalElbRecordName: internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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