# JSON Schema for FireWorks
This package provides a [JSON schema](https://json-schema.org/) for the [FireWorks](https://github.com/materialsproject/fireworks) package.

## Why should I use JSON schema?

The input for FireWorks is often provided in JSON or YAML formats and generated by
third-party software that is unaware of the valid data types in FireWorks. Latent
mismatches of data types may produce run-time errors, such as missing keywords
or wrong data types, that are more difficult to handle than a validation of the
initial input.

The *fireworks_schema* package provides a formal human- and machine-readable description of
the data types used in classes in FireWorks. Additionally, a function is provided
that checks the validity of JSON and YAML inputs immediately before deserialization.


## Installing *fireworks_schema*

The recommended way to install this package into your virtual environment is using ``pip`` (in any folder):

```
python -m pip install fireworks-schema
```

Alternatively you can download a release from the GitHub
[repository](https://github.com/ikondov/fireworks_schema), unpack the archive, change into the folder containing ``setup.py`` and run:

```
python setup.py install
```

After the installation you can run the tests, for example using the *nose* package:

```
nosetests fireworks_schema.tests
```


## Using *fireworks_schema* to validate input for FireWorks

There are two ways to perform JSON schema validation:

* Call the schema validator explicitly
* Activate automatic schema validation


### Call the schema validator explicitly

This is the case when you use Python but read JSON/YAML serialized objects
provided externally. In the following example, a serialized workflow object is
loaded from a YAML file and validated against the Workflow schema:

```python
  import yaml
  import fireworks_schema
  from fireworks import Workflow

  with open('empty_fws.yaml', 'rt') as yf:
      dct = yaml.safe_load(yf)
  fireworks_schema.validate(dct, 'Workflow')
  wf = Workflow.from_dict(dct)
```

### Activate automatic schema validation

To activate automatic schema validation you must specify:

```yaml
  JSON_SCHEMA_VALIDATE: true
```

in your FWConfig file. For more details about managing your FWConfig file see the
[FW Config tutorial](https://materialsproject.github.io/fireworks/config_tutorial.html).

The default value of ``JSON_SCHEMA_VALIDATE`` is ``false``.

If automatic validation is turned on, i.e. ``JSON_SCHEMA_VALIDATE`` is ``true``,
then validation is performed only for the classes specified in the list
``JSON_SCHEMA_VALIDATE_LIST``, whenever an object of these
classes is loaded from file. There is no default for ``JSON_SCHEMA_VALIDATE_LIST``
and therefore you must set ``JSON_SCHEMA_VALIDATE_LIST`` in your FWConfig file.
For example, to turn on automatic validation for serialized ``Firework`` and
``Workflow`` objects these two lines must be added to the FWConfig file:

```yaml
  JSON_SCHEMA_VALIDATE: true
  JSON_SCHEMA_VALIDATE_LIST: [Firework, Workflow]
```
