import terrascript.core as core


@core.resource(type="aws_cloudfront_public_key", namespace="cloudfront")
class PublicKey(core.Resource):
    """
    Internal value used by CloudFront to allow future updates to the public key configuration.
    """

    caller_reference: str | core.StringOut = core.attr(str, computed=True)

    """
    (Optional) An optional comment about the public key.
    """
    comment: str | core.StringOut | None = core.attr(str, default=None)

    """
    (Required) The encoded public key that you want to add to CloudFront to use with features like field
    level encryption.
    """
    encoded_key: str | core.StringOut = core.attr(str)

    """
    The current version of the public key. For example: `E2QWRUHAPOMQZL`.
    """
    etag: str | core.StringOut = core.attr(str, computed=True)

    """
    The identifier for the public key. For example: `K3D5EWEUDCCXON`.
    """
    id: str | core.StringOut = core.attr(str, computed=True)

    """
    (Optional) The name for the public key. By default generated by Terraform.
    """
    name: str | core.StringOut | None = core.attr(str, default=None, computed=True)

    """
    (Optional) The name for the public key. Conflicts with `name`.
    """
    name_prefix: str | core.StringOut | None = core.attr(str, default=None, computed=True)

    def __init__(
        self,
        resource_name: str,
        *,
        encoded_key: str | core.StringOut,
        comment: str | core.StringOut | None = None,
        name: str | core.StringOut | None = None,
        name_prefix: str | core.StringOut | None = None,
        depends_on: list[str] | core.ArrayOut[core.StringOut] | None = None,
        provider: str | core.StringOut | None = None,
        lifecycle: core.Lifecycle | None = None,
    ):
        super().__init__(
            name=resource_name,
            args=PublicKey.Args(
                encoded_key=encoded_key,
                comment=comment,
                name=name,
                name_prefix=name_prefix,
                depends_on=depends_on,
                provider=provider,
                lifecycle=lifecycle,
            ),
        )

    @core.schema_args
    class Args(core.Resource.Args):
        comment: str | core.StringOut | None = core.arg(default=None)

        encoded_key: str | core.StringOut = core.arg()

        name: str | core.StringOut | None = core.arg(default=None)

        name_prefix: str | core.StringOut | None = core.arg(default=None)
