# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['numlab',
 'numlab.automata',
 'numlab.compiler',
 'numlab.compiler.parsers',
 'numlab.ia',
 'numlab.lang',
 'numlab.nl_builtins',
 'numlab.visitors']

package_data = \
{'': ['*']}

install_requires = \
['typer>=0.4.0,<0.5.0']

entry_points = \
{'console_scripts': ['numlab = numlab.main:app']}

setup_kwargs = {
    'name': 'numlab',
    'version': '0.1.1',
    'description': "Programming language for analyzing mathematical numerical problem's solutions",
    'long_description': '# Numlab\n\n## Contenidos\n\n- [Objetivos](https://github.com/jmorgadov/NumLab#objetivos)\n- [¿Cómo instalar numlab?](https://github.com/jmorgadov/NumLab#c%C3%B3mo-instalar-numlab)\n- [Lenguaje](https://github.com/jmorgadov/NumLab#lenguaje)\n  - [Características básicas](https://github.com/jmorgadov/NumLab#caracter%C3%ADsticas-b%C3%A1sicas)\n  - [Estadísticas en tiempo de ejecución](https://github.com/jmorgadov/NumLab#estad%C3%ADsticas-en-tiempo-de-ejecuci%C3%B3n)\n  - [Simulación de código](https://github.com/jmorgadov/NumLab#simulaci%C3%B3n-de-c%C3%B3digo)\n    - [Configuración de la simulación](https://github.com/jmorgadov/NumLab#configuraci%C3%B3n-de-la-simulaci%C3%B3n)\n    - [Regiones de código simuladas](https://github.com/jmorgadov/NumLab#regiones-de-c%C3%B3digo-simuladas)\n- [Implementación](https://github.com/jmorgadov/NumLab#implementaci%C3%B3n)\n  - [Motor de expresiones regulares](https://github.com/jmorgadov/NumLab#motor-de-expresiones-regulares)\n  - [Tokenizador](https://github.com/jmorgadov/NumLab#tokenizador)\n  - [Gramáticas](https://github.com/jmorgadov/NumLab#gram%C3%A1ticas)\n  - [Árbol de Sintaxis Abstracta (AST)](https://github.com/jmorgadov/NumLab#%C3%A1rbol-de-sintaxis-abstracta-ast)\n  - [Parser](https://github.com/jmorgadov/NumLab#parser)\n  - [Visitors](https://github.com/jmorgadov/NumLab#visitors)\n  - [Ejecución](https://github.com/jmorgadov/NumLab#ejecuci%C3%B3n)\n    - [Tipos y funciones predefinidas](https://github.com/jmorgadov/NumLab#tipos-y-funciones-predefinidas)\n    - [Contextos](https://github.com/jmorgadov/NumLab#contextos)\n    - [Evaluación](https://github.com/jmorgadov/NumLab#evaluaci%C3%B3n)\n- [Optimización de código](https://github.com/jmorgadov/NumLab#optimizaci%C3%B3n-de-c%C3%B3digo)\n- [Extras](https://github.com/jmorgadov/NumLab#extras)\n  - [Aplicación CLI](https://github.com/jmorgadov/NumLab#aplicaci%C3%B3n-cli)\n  - [Extensión de VSCode](https://github.com/jmorgadov/NumLab#extensi%C3%B3n-de-vscode)\n\n## Objetivos\n\nEl objetivo de este proyecto es crear una herramienta de aprendizaje\ndesarrollada para ayudar a los estudiantes de Ciencias de la Computación que\nestán cursando la asignatura Matemática Numérica. Se quiere crear un lenguaje\nbasado en Python en el cual los estudiantes implementen soluciones de problemas\nnuméricos como: hallar cero de funciones, resolver sistemas de ecuaciones,\nentre otros; y a la vez, realizar un análisis de la ejecución de estas\nsoluciones.\n\nUna de las características que tendrá este lenguaje es la capacidad de simular\nregiones de código bajo ciertas condiciones o limitantes, como por ejemplo:\nlimitar el tiempo de ejecución, limitar la cantidad de variables que se pueden\ncrear, lmitar el tiempo que toma realizar una operación determinada, entre\notras. Esto fomenta en los estudiantes la búsqueda de soluciones más\neficientes. Además, el lenguaje también constará con la posibilidad, mediante\nun algoritmo genético, de optimizar códigos ya escritos.\n\n## ¿Cómo instalar numlab?\n\n```bash\npip install numlab\n```\n\nPara más información sobre el uso del programa:\n\n```bash\nnumlab --help\n```\n\n> También se encuentra una breve descripción en la sección\n> [Aplicación CLI](https://github.com/jmorgadov/NumLab#aplicaci%C3%B3n-cli)\n\n\n## Lenguaje\n\n**Numlab** es el lenguaje que se ha implementado para dar solución a los\nobjetivos mencionados anteriormente. En las siquientes secciones, se muestran\nlas principales características del mismo.\n\n### Características básicas\n\n**Numlab** está basado en Python, aunque no implementa todas las\nfuncionalidades del mismo, la sintaxis básica sí es la misma. Ejemplos:\n\n```python\n# Declaración de variables\na = 5\npi = 3.14\nname = "John"\nval = True \n\n# Funciones built-in\nprint("my name is", name)\n\n# Operaciones aritméticas\nprint(a + pi)\nprint(a - pi)\nprint(a ** 2)\n\n# Control de flujo\nif val:\n    print("val is true")\nelse:\n    print("val is false")\n\nfor i in range(10):\n    if i % 2 == 0:\n         print(i)\n\n\n# Declaración de funciones\ndef foo(a, b):\n    return a + b\n\nbar = lambda a, b: a + b\n\n# Declaración de tipos\nclass Foo:\n    def __init__(self, a, b):\n        self.a = a\n        self.b = b\n\n    def __str__(self):\n        return "Foo(%s, %s)" % (self.a, self.b)\n```\n\n### Estadísticas en tiempo de ejecución\n\nEn cada ejecución existe una variable llamada `stats`, la cual contiene un\ndiccionario con las estadísticas de la ejecución. Entre las estadísticas que se\npueden obtener se encuentran:\n\n - `stats["time"]`: tiempo de ejecución en segundos.\n - `stats["assign_count"]`: cantidad de asignaciones realizadas.\n - `stats["var_count"]`: cantidad de variables creadas.\n - `stats["call_count"]`: cantidad de llamados a funciones realizados.\n - `stats["add_count"]`: cantidad de veces que se realizó la operación suma.\n - `stats["sub_count"]`: cantidad de veces que se realizó la operación resta.\n - `stats["mul_count"]`: cantidad de veces que se realizó la operación multiplicación.\n - `stats["truediv_count"]`: cantidad de veces que se realizó la operación división.\n - `stats["pow_count"]`: cantidad de veces que se realizó la operación potencia.\n - `stats["mod_count"]`: cantidad de veces que se realizó la operación módulo.\n - `stats["floordiv_count"]`: cantidad de veces que se realizó la operación división entera.\n - `stats["lshift_count"]`: cantidad de veces que se realizó la operación **left shift**.\n - `stats["rshift_count"]`: cantidad de veces que se realizó la operación **right shift**.\n - `stats["matmul_count"]`: cantidad de veces que se realizó la operación multiplicación de matrices.\n - `stats["bit_xor_count"]`: cantidad de veces que se realizó la operación **bitwise xor**.\n - `stats["bit_and_count"]`: cantidad de veces que se realizó la operación **bitwise and**.\n - `stats["bit_or_count"]`: cantidad de veces que se realizó la operación **bitwise or**.\n - `stats["contains_count"]`: cantidad de veces que se comprobó si un elemento está contenido en otro.\n - `stats["eq_count"]`: cantidad de veces que se comprobó si dos elementos son iguales.\n - `stats["ne_count"]`: cantidad de veces que se comprobó si dos elementos son distintos.\n - `stats["lt_count"]`: cantidad de veces que se comprobó si un elemento es menor que otro.\n - `stats["gt_count"]`: cantidad de veces que se comprobó si un elemento es mayor que otro.\n - `stats["le_count"]`: cantidad de veces que se comprobó si un elemento es menor o igual que otro.\n - `stats["ge_count"]`: cantidad de veces que se comprobó si un elemento es mayor o igual que otro.\n - `stats["and_count"]`: cantidad de veces que se realizó la operación lógica **and**.\n - `stats["or_count"]`: cantidad de veces que se realizó la operación lógica **or**.\n\nEjemplo:\n\n```python\nfor i in range(10):\n    a = i * 2\n\nprint(stats["mul_count"])  # Output: 10\n```\n\nEn cualquier momento de la ejecución, se puede restablecer todas las estadísticas\na cero usando la palabra clave `resetstats`.\n\nEjemplo:\n```python\nfor i in range(10):\n    a = i * 2\n    if i == 4:\n        resetstats\n\nprint(stats["mul_count"])  # Output: 5\n```\n\n### Simulación de código\n\nEn **Numlab** es posble simular regiones de código bajo diferentes\nrestricciones, como lo son: limitar la cantidad de variables que se pueden\ncrear (`max_var_count`) y limitar el tiempo de ejecución (`max_time`), limitar\nla cantidad de veces que se realiza una operación determinada y establecer el\ntiempo que toma en realizarse una operación.\n\nEl nombre de las restrcciones relacionadas con operadores se nombran:\n`max_<operator>_count` y `<operator>_time` para establecer la catidad máxima de\nveces que se puede realizar una operación y el tiempo que toma en realizarse\nrespectivamente. Por ejemplo, si se desea limitar la cantidad de veces que se\nrealiza la operación suma: `max_add_count`.\n\n#### Configuración de la simulación\n\nPara establecer estas restricciones en **Numlab** las mismas se deben escribir\nen un bloque de configuración:\n\n```python\nconf general_conf:\n    max_time 0.5\n    max_var_count 10\n    max_add_count 100\n    max_sub_count 100\n    sub_time 0.1\n```\n\nEs posible también crear jerarquías de restricciones, por ejemplo:\n\n```python\nconf general_conf:\n    max_time 0.5\n    max_var_count 10\n    max_add_count 100\n    max_sub_count 100\n    sub_time 0.1\n\nconf add_config1(general_conf):\n    add_time 0.3\n\nconf add_config2(general_conf):\n    add_time 0.5\n```\n\nEstas configuraciones heredan las restricciones de la configuración base. De\nesta forma se pueden declarar configuraciones diferentes pero que tengan\nrestricciones en común sin necesidad de repetir las mismas. Si se establece una\nrestricción que ya estaba en la configuración base, se sobreescribe.\n\nLos valores de una configuración pueden ser resultados de una operación, por\nejemplo:\n\n```python\na = 2\n\nconf c1:\n    max_time a ** 3 + 8\n```\n\nSi la restricción es sobre el tiempo de ejecución de un operador, es posible\ntambién especificar en vez de un valor, una función que devuelva el tiempo de\nejecución de la operación. Esta función se ejecuta cada vez que se realiza la\noperación determinada y debe devolver el tiempo que debe durar la operación.\nPor ejemplo:\n\n```python\nconf c1:\n    add_time lambda : rand() * 2 \n```\n\n> `rand` es una función built-in que devuelve un número aleatorio entre 0 y 1\n> con una distribución uniforme.\n\nEn el ejemplo anterior, cada vez que se simule la operación suma, la misma\ndemorará un tiempo aleatorio entre 0 y 2.\n\n#### Regiones de código simuladas\n\nPara establecer una región de código donde se simulen las restricciones\nestablescidas en una configuración, se utilizan las palabras claves `begsim\n<config>` y `endsim`, donde `<config>` es el nombre de la configuración que se\ndesea usar. Estas palabras indican donde comienza y termina la simulación\nrespectivamente. Por ejemplo:\n\n```python\nconf c1:\n    max_time 0.5\n\nbegsim c1\na, b = 1, 1\nprint(a)\nprint(b)\nfor _ in range(98):\n    a, b = b, a + b\n    print(b)\nendsim\n```\n\nEs posible utilizar `begsim` y `endsim` más de una vez en una misma ejecución\n(en tal caso se recomienta usar `resetstats` antes de cada empezar de\nsimulación). Se pueden relizar simulaciónes dentro de simulaciones, en tal caso\nse pasa a utilizar la configuración de la última simulación. Cada `endsim`\ntermina un nivel de simulación (la última que se haya hecho). Estas palabras\nclaves pueden estar en cualquier parte del código, la única restricción es que\nno se pueden ejecutar más `endsim` que `begsim`.\n\n## Implementación\n\n**Numlab** es un lenguaje evaluado escrito en Python. A continuación se\nexponen las características principales de la implementación de cada estapa.\n\n### Autómatas\n\nPara la creación de las algunas de las proximas funcionalidades, se realizaó\nuna implementación de un tipo `Automata` que permite simular una\nmáquina de estados de forma genérica. A los mismos se le pueden agregar\nestados así como transiciones entre los mismos. Cada autómata tiene un estado\ninicial y uno o varios estados finales.\n\nLa ejecución de una máquina de estados realizada con un autómata es bastante\nsimple. Dado una entrada iterable, se comienza en el estado inicial y se va\nejecutando cada transición hasta llegar a un estado final. En caso de llegar a\nun estado en el que ninguna transición es válida, se termina la ejecución y la\nentrada no es válida. En caso de terminar de recorrer la entrada se clasifica\nla entrada como válida o inválida en dependencia de si se llegó a un estado\nfinal o no respectivamente.\n\nLos autómatas pueden tener transiciones **épsilon** entre estados, en este\ncaso, la ejecución se bifurca y la maquina de estados se mueve por todos los\nestaos posibles al mismo timepo. Esto da la posibliadad de ejecutar autómatas\nno deterministas.\n\nSe implementó además, utilizando el algoritmo visto en clase (calculando los\n**goto** y **epsilon clausuras**) la opción de convertir un autómata no\ndeterminista (NFA) a un autómata determinista (DFA).\n\n### Motor de expresiones regulares\n\nLas principales funcionalidades implementadas son:\n\n- Operador `*`: Matchea cero o más veces la expresión anterior.\n- Operador `|`: Mathcea la expresión anterior o la siguiente.\n- Operador `^`: Matchea cualquier expresion excepto la expresión que le prosigue.\n- Caracter `.`: Matchea cualquier caracter (ASCII).\n- Caracter `\\`: Inicio de un caracter especial.\n- Caracter `\\d`: Matchea un dígito.\n- Caracter `\\a`: Matchea una letra minúscula.\n- Caracter `\\A`: Matchea una letra mayúscula.\n- Parentesis `(` y `)`: Agrupan una expresión regular.\n\n> Cualquier operador o caracter especal puede ser escapado con `\\`.\n\nPara la realización del motor de expresiones regulares se utilizó la clase\n`Automata`. Para cada expresión regular se construye un autómata finito no\ndeterminista (NFA) usando el algoritmo de Thompson y luego el mismo se\nconvierte a un DFA utlizando el método `to_dfa` de la clase `Automata`.\n\nSe ofrecen además dos funciones para el matcheo de cadenas segun una expresión\nregular: `match` (la cual tiene un comportamiento similar a `re.match`) y\n`compile_patt` (la cual tiene un comportamiento similar a `re.compile`). La\nventaja principal de usar `compile_patt` es que se no es necesario crear un\nautómata para cada vez que se desea matchear una cadena (ya que el autómata es\nconstruido una sola vez).\n\n### Tokenizador\n\nPara la implementación del tokenizador se creó una clase `Tokenizer`. Esta\nclase se encarga de tomar un texto y dividirlo en diferentes tipos de tokens.\nCada patrón que se agrega está definido por un nombre (tipo del token) y una\nexpresión regular (se hace uso del motor de expresiones regulares\nimplementado).\n\n```python\ntknz = Tokenizer()\ntknz.add_pattern("NUMBER", r"\\d\\d*|\\d\\d*\\.\\d\\d*")\n```\n\nAl tokenizar un texto, se revisan los patrones comenzando por el primero (en el\nmismo orden en el que fueron agregados) y el primero que matchee con un prefijo\nde la cadena se establece como un token nuevo (se toma como lexema la subcadena\nque matcheó con la expresión regular). Luego se vuelve a realizar esta\noperación con el resto de la cadena, así sucesivamente hasta terminar la misma.\nSi en algún punto no se encuentra un token que matchee con el inicio de la\ncadena, se considera que la cadena no se puede tokenizar (con los tipos de\ntokens establecidos).\n\nCada vez que se agrega un patrón al tokenizador se puede establecer una\nfunción que se aplicará al lexema antes de guardar su valor en el token.\n\nPor ejemplo, para quitar las comillas al tokenizar un **string**:\n\n```python\ntknz.add_pattern("STRING", r"\'((^\')|(\\\\\'))*(^\\\\)\'", lambda t: t[1:-1])\n```\n\nEsta función tambien puede ser utilizada para indicar que se quiere ignorar\nlos tokens de un tipo determinado. En tal caso basta con que la función devuelva\n`None`:\n\n```python\ntknz.add_pattern("SPACE", r"( | \\t)( |\\t)*", lambda t: None)\n```\n\nSe ofrece también la opción de agregar `keywords` (palabras claves) para una\nmayor comodidad. Esto se hace mediante el método `add_keywords()` el cual recibe\nuna lista de palabras. En el proceso de tokenización, si el prefijo matcheado\nconicide con alguna de las palabras clave, entonces el tipo del token se\nestablece como `KEYWORD`.\n\nEn caso de que se quiera aplicar una función para procesar todos los tokens\nobtenidos, se puede usar el decorador `process_tokens` de la clase `Tokenizer`.\nEste debe ser usado en una función que reciba un solo argumento (la lista de\ntokens) y devuelva una lista de tokens procesados.\n\n```python\n@tknz.process_tokens\ndef process_tokens(tokens):\n    # ...\n    return tokens\n```\n\nFinalmente, para obtener los tokens de un texto basta con usar la función\n`tokenize`:\n\n```python\ntokens = tknz.tokenize("some text")\n```\n\n### Gramáticas\n\nSe implementaron las clases `Grammar`, `NonTerminal`, `Terminal` y `Production`\nlas cuales son usadas para la representación de una gramática general. Se\nimplementó además un parser de gramáticas con el cual es posible crear\ngramáticas dado un formato, esto permite definir la gramática del lenguaje en\nun archivo y poder cambiarla fácilmente. Dado la sencillez del formato (el\nlenguaje de las gramáticas), se implementó un sencillo parser recursivo\ndescendente para la creación de las mismas.\n\nEl formato especificado es el siguiente:\n\n```\nexpression: production_1 | production_2 | ... | production_n\n```\n\nDe forma equivalente, para mayor legibilidad:\n\n```\nexpression:\n    | production_1 \n    | production_2\n    | ...\n    | production_n\n```\n\nEjemplo:\n\n```\nExprAB:\n    | \'a\' ExprAB \'b\'\n    | EPS\n```\n\n> EPS es un elemento especial en las gramáticas para representar *epsilon*\n\nLas gramáticas luego pueden ser cargadas como se muestra a continuación:\n\n```python\nfrom grammar im port Grammar\ngm = Grammar.open("expr_ab.gm")\n```\n\nLas gramáticas están compuestas por una lista de expresiones (no terminales).\nCada no terminal de la gramática, contiene una lista de producciones. Cada\nproducción contiene una lista de elementos (terminales o no terminales).\n\n### Árbol de Sintaxis Abstracta (AST)\n\nPara la creación de un AST se creó la clase abstracta `AST`. De esta clase\nheredan todos las clases que representan los nodos del árbol de sintaxis \nabstracta del lenguaje. En la clase se implementa también un método `dump`\nque permite mostrar el árbol de forma legible. Este método usa el\natributo `__slots__` mediante el cual se definen los atributos que se\nquieren mostrar.\n\nEjemplo del árbol generado a partir del código:\n\n```python\nconf c1:\n    max_time 5\n    max_var_count 5\n\nbegsim c1\ndef foo(a, b):\n    print("hola", a, b)\n\na, b = 1, 2\nfoo(a, b)\nendsim\n```\n\nÁrbol generado:\n```text\nProgram:\n   stmts: [\n      ConfDefStmt:\n         name: c1\n         configs: [\n            ConfOption:\n               name: max_time\n               value: ConstantExpr(0.5)\n         ]\n      Begsim:\n         config: NameExpr(\'c1\', ctx=ExprCtx.LOAD)\n      ForStmt:\n         target: NameExpr(\'i\', ctx=ExprCtx.LOAD)\n         iter_expr: (TupleExpr)\n            elts: [\n               CallExpr:\n                  func: NameExpr(\'range\', ctx=ExprCtx.LOAD)\n                  args: [\n                     ConstantExpr(100)\n                  ]\n            ]\n            ctx: ExprCtx.LOAD\n         body: [\n            TupleExpr:\n               elts: [\n                  CallExpr:\n                     func: NameExpr(\'print\', ctx=ExprCtx.LOAD)\n                     args: [\n                        BinOpExpr:\n                           left: NameExpr(\'i\', ctx=ExprCtx.LOAD)\n                           op: Operator.POW\n                           right: ConstantExpr(2)\n                     ]\n               ]\n               ctx: ExprCtx.LOAD\n         ]\n      Endsim:\n   ]\n```\n\nPara definir cómo se construye cada nodo del AST se pueden asignar los\nconstructores a cada producción de la gramática usando la función\n`assign_builders`. Esta función recibe un diccionario donde las llaves son la\nrepresentación textual de la producción y los valores son funciones que reciben\ncomo argumentos los elementos de la producción. En caso de que el símbolo sea\nun terminal la función recibirá dicho terminal, en caso de ser un no terminal,\nla función recibirá el resultado de la ejecución algunas de las funciones\nconstructoras de las producciones que tengan como cabeza a dicho no terminal.\n\nPor ejemplo, a continuación se muestran algunos de los constructores para\nla gramática de **Numlab**:\n\n```python\nbuilders = {\n    # -------------------------------------------------------------------------\n    "program -> stmt program": lambda s, p: ast.Program([s] + p.stmts),\n    "program -> NEWLINE program": lambda n, p: p,\n    "program -> EPS": lambda: ast.Program([]),\n    # -------------------------------------------------------------------------\n    "stmt -> simple_stmt": lambda s: s,\n    "stmt -> compound_stmt": lambda c: c,\n    # -------------------------------------------------------------------------\n    "stmt_list -> stmt": lambda s: [s],\n    "stmt_list -> stmt stmt_list": lambda s, sl: [s] + sl,\n    # -------------------------------------------------------------------------\n    # ...\n    # ...\n```\n\n### Parser\n\nPara la implementación del parser principal del lenguaje se creó la clase\nabstacta `Parser`. Usando esta clase como base se creó una clase `LR1Parser`,\nla cual implementa un parser LR(1).\n\nPara la realización del parser LR(1) fue necesario implementar las clases\n`LR1Item` y `LR1Table`. La primera de estas clases representa un item del\nparser, el cual contiene: la producción que lo genera, la posición del punto\n(dot) en la producción y el terminal que le debe proseguir (lookahead).\n\nLa segunda clase (`LR1Table`) representa la tabla de transición del parser.\nCada posición de la tabla puede contener tres tipos de elementos: un **string**\n`"OK"`, que indica que el estado de aceptación; un valór numérico entero, que\nindica cual es el siguiente estado; o un no terminal de la gramática, el cual\nrepresenta que hay que realizar una reducción. Para no tener que recalcular la\ntabla cada vez que se va a parsear un texto, la misma puede ser serializada y\nluego cargada.\n\nLa construcción de la tabla se realizó siguiendo el algoritmo visto en las\nconferencias de la asignatura (calculando los **goto** y las **clausuras** de\nlos estados).\n\nEn el proceso de parsing, al realizar una acción de reducción, es donde se\nutilizan las funciones constructoras vistas en la sección anterior. En\ndependencia de la producción que se está reduciendo, se llama a la función\nconstructora correspondiente.\n\nPara una mayor comodidad se implementó también la clase `ParserManager`. Esta\nclase ofrece, dado una gramática, un tokenizador (opcional) y un parser\n(opcional, por defecto LR(1)), métodos como: `parse_file` (para parsear un\narchivo), `parse` (para parsear un texto) y `parse_tokens` (para parsear una\nlista de tokens directamete). Estas funciones devuelven el AST resultante del\nproceso de parsing.\n\n### Visitors\n\nUna vez obtenido el AST de un programa es necesario realizar recorridos sobre\nél. Para ello se implmentó una clase `Visitor` la cual contiene dos decoradores\n`@visitor` y `@callback`. Por cada **visitor** que se quiera implementar para\nel AST, se debe implementar una nueva clase que tenga como atributo de clase\nuna instancia de la clase `Visitor`. Luego, cada método de la clase que tenga\nel decorador `@visitor`, se establecerá como una sobrecarga. Es por ello que\ntodos estos métodos deben tener sus argumentos tipados (esta es la forma en la\nque el **visitor** sabe cual de los métodos de la clase debe llamar).\n\nPor ejemplo:\n\n```python\nfrom numlab.lang.visitor import Visitor\n\nclass EvalVisitor:\n    visitor_dec = Visitor().visitor\n\n    @visitor_dec\n    def eval(self, node: ast.Program):\n        for stmt in node.stmts:\n            stmt.eval(self)\n\n    @visitor_dec\n    def eval(self, node: ast.Stmt): ...\n\n    # ...\n```\n\nEl decorador `@callback` se utiliza para definir funciones que se van a llamar\ncada vez que se llame a una función marcada como **visitor**. En el proyecto\nuno de los usos que se le da a este decorador es para comprobar que el tiempo de\nejecución de una simulación es menor que el límite establecido en cada momento.\n\n### Ejecución\n\nPara la ejecución de un programa representado en un AST se implementó un\nvisitor `EvalVisitor` (muy similar al ejemplo de la sección anterior). A\ncontinuación se muestran algunas de las características más importantes\nimplementadas en el proceso de evaluación.\n\n\n#### Tipos y funciones predefinidas\n\nPara representar los tipos predefinidos de **Numlab** se creó la clase `Type`.\nCada instancia de esta clase representa un tipo de **Numlab**, entre ellos:\n`int`, `str`, `list`, etc (los tipos básicos existentes en python).\n\nA cada tipo de le pude agregar atributos (las funciones también se agregan como\natributo). Además, cada tipo puede derivar de otro (permitiendo la herencia), por\nconsiguiente, en la resolución de un atributo si el mismo no está definido en el\ntipo actual, se busca en el tipo padre (y así susesivamente).\n\nSe implementaron tambíen varias de las funciones built-in de python en **Numlab**.\nEn la ejecución del código se puede acceder a estas funciones directamente.\n\n#### Contextos\n\nPara definir el contexto donde se encuentra cada variable, tipo o función que se\ncrea, se implementó la clase `Context`. Un **context** tiene un diccionario\ncon el nombre de cada objeto creado en el mismo y su respectivo valor. Cada contexto\ntiene además una referencia al contexto padre (en caso de que exista). Esto\npermite que al realizar la resolución de una variable, si la misma no se ecuentra\nen el contexto actual, se busque en el contexto padre.\n\n> En el EvalVisitor al realizar la resolución de una variable, se busca primero\n> en el context actual, y si no se encuentra, se busca entre las funciones y\n> tipos predefinidos.\n\nEl contexto también es utilizado también en las secciones de código simuladas\npara comprobar la cantidad de variables creadas (en caso de existir alguna\nrestricción sobre este valor).\n\n#### Evaluación\n\nComo se había mencionado anteriormente, la evaluación de un programa se realiza\nmediante un **visitor**. En este **visitor** se implementaron las funciones\nque definen cómo se evalúa cada uno de los nodos del AST.\n\nEn ocasiones, existen nodos que afectan la evaluación de otros, como por\nejemplo las palabras claves de control de flujo (`break`, `continue`, `return`,\netc). Para ello, el `EvalVisitor` cuenta con un diccionario `flags` (atributo\nde instancia) que contiene diversa información que se puede utlizar en común\nentre la evaluación de los nodos.\n\nEn estas evaluaciones es también donde se van guardando las estadísiticas de la\nejecución del programa (en un diccionario `stats` el cual tiene accesibilidad\nincluso desde el código que se está ejecutando). Por ejemplo, al realizar un\nllamado a una función, se incrementa el contador de llamados:\n\n```python\nclass EvalVisitor:\n    \n    # ... Other eval methods\n\n    @visitor\n    def eval(self, node: ast.CallExpr):\n        self.set_stat("call_count", self.stats["call_count"] + 1)\n        # Call eval implementation ...\n\n    # ... Other eval methods\n```\n\n## Optimización de código\n\nSe implementó también un optimizador de código, el cual, mediante un algorítmo\ngenético, cambia la estructura de un AST para reducir el tiempo de ejecución.\nPara ello se creó otro **visitor**, el cual se puede clasificar como un sistema\nexperto. Este **visitor**, busca en un programa determinado en qué nodos se\npueden realizar, bajo ciertas reglas, cambios que puedan reducir el tiempo de\nejecución.\n\nEntre estas reglas se pueden mencionar por ejemplo, cambiar el orden de las\ncomprobaciónes en una condicional (o ciclo while) para que se evalúen primero\nlas condiciones que son más sencillas, o incluso, sustituir operaciones de\nmultiplicación por operaciones **shift** si es posible, etc.\n\nUna vez se recorre el AST en busca de cambios, se crea un vector el cual\ncontiene en cada posición la información de dónde se puede realizar un cambio\n(el nodo) y dos funciones: una que realiza el cambio y otra que devuelve el\nnodo a su estado original.\n\nEsta información se lleva a la ejecución de un algorítmo genético. La población\nde este algoritmo consiste en vectores booleanos que indican si se debe realizar\nun cambio en un nodo o no. Al evaluar un AST se realizan los cambios necesarios\ny se obtiene el tiempo de ejecución (en caso de existir un error el tiempo de\nejecución será infinito).\n\nLa población inicial se genera aleatoriamente. El entrecruzamiento entre dos\nvectores se realiza dividiéndolos en dos partes, he intercambiando las mismas.\nEsta forma de entrecruzamiento es posible ya que cada cambio es independiente.\nLa mutación consiste en cambiar aleatoriamente algún valor del vector.\n\nAl ejecutar la optimización se obtiene el AST resultante de realizar los cambios\ndefinidos por el mejor vector de la población (luego de algunas generaciones).\n\nA continuación se muestra un ejemplo de código y su optimización:\n\n```python\nitems = [1, 1, 1, 1, 1]\n\ndef foo():\n    a = [i for i in range(100)]\n    return items\n\nfor i in range(50):\n    if i in [j for j in range(48, 500)] or i < 40: \n        a = i + 3\n    if foo() and items[0] == 1:\n        items.remove(1)\n        \nprint(stats["time"])\n```\n\nEn este código se puede ver que el primer `if` realiza dos comprobaciones. La\nprimera de ellas genera una lista de al rededor de 450 elementos cada vez que\nse ejecuta el `if`, mientras que la segunda solamente compara dos valores.\n\nSe puede observar además que existe otra comprobación, donde también se\nrealizan dos operaciones y la primera de ellas es más costosa temporalmente que\nla segunda. Sin embargo en este caso la segunda operación es dependiente de la\nprimera, por lo que no se puede realizar antes. Esto no se comprueba en ningún\nmomento al extraer los cambios posibles debido a la complegidad que puede\npresentar comprobar este tipo relaciones entre los nodos. Más adelante el\nalgoritmo genético es quien decidirá si es correcto realizar el cambio o no.\n\nAl ejecutar este código se obtuvo un tiempo de ejecución promedio de 1.8\nsegundos. Al optimizar el AST, se detectaron dos cambios potenciales (el orden\nde comprobación en las expresiones condicionales mencionadas anteriormente). El\nalgoritmo genético obtuvo como mejor resultado realizar el primer cambio y no\nel segundo. Finalmente, el código optimizado tuvo un tiempo de ejecución de\naproximadamente 0.6 segundos.\n\n## Extras\n\n### Aplicación CLI\n\nAl instalar **Numlab** es posible usar una interfaz de comando mediente la\nterminal. La misma cuenta con dos comandos principales: `run` y `optimize`, \nlas cuales ejecutan u optimizan un programa respectivamente.\n\n```text\n$ numlab --help\nUsage: numlab [OPTIONS] COMMAND [ARGS]...\n\nOptions:\n  --help  Show this message and exit.\n\nCommands:\n  optimize  Optimize a program given in the input file\n  run       Run the program given in the input file\n  version   Print the version number\n```\n\nEjemplo:\n\n```bash\nnumlab "my_script.nl" --verbose\n```\n\nPara más información sobre los comandos:\n\n```bash\nnumlab run --help\nnumlab optimize --help\n```\n\n### Extensión de VSCode\n\nUna [extensión](https://marketplace.visualstudio.com/items?itemName=JessicaNunez.numlab)\nde VSCode fue desarrollada también para el uso del lenguaje.\nLa misma aporta ***highlighting*** de las palabras claves del lenguaje\npara los archivos `.nl`.\n',
    'author': 'Jorge Morgado Vega',
    'author_email': 'jorge.morgadov@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
