"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
describe('fromBucket', () => {
    test('as bucket, renders all required properties, including S3Origin config', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = lib_1.Origin.fromBucket(bucket);
        origin._bind(stack, { originIndex: 0 });
        expect(origin._renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketRegionalDomainName,
            s3OriginConfig: {
                originAccessIdentity: 'origin-access-identity/cloudfront/${Token[TOKEN.69]}',
            },
        });
    });
    test('as bucket, creates an OriginAccessIdentity and grants read permissions on the bucket', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = lib_1.Origin.fromBucket(bucket);
        new lib_1.Distribution(stack, 'Dist', { defaultBehavior: { origin } });
        expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
            CloudFrontOriginAccessIdentityConfig: {
                Comment: 'Allows CloudFront to reach the bucket',
            },
        });
        expect(stack).toHaveResourceLike('AWS::S3::BucketPolicy', {
            PolicyDocument: {
                Statement: [{
                        Principal: {
                            CanonicalUser: { 'Fn::GetAtt': ['DistS3Origin1C4519663', 'S3CanonicalUserId'] },
                        },
                    }],
            },
        });
    });
    test('as website bucket, renders all required properties, including custom origin config', () => {
        const bucket = new s3.Bucket(stack, 'Bucket', {
            websiteIndexDocument: 'index.html',
        });
        const origin = lib_1.Origin.fromBucket(bucket);
        origin._bind(stack, { originIndex: 0 });
        expect(origin._renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketWebsiteDomainName,
            customOriginConfig: {
                originProtocolPolicy: 'http-only',
            },
        });
    });
});
describe('HttpOrigin', () => {
    test('renders a minimal example with required props', () => {
        const origin = new lib_1.HttpOrigin({ domainName: 'www.example.com' });
        origin._bind(stack, { originIndex: 0 });
        expect(origin._renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: 'www.example.com',
            customOriginConfig: {
                originProtocolPolicy: 'https-only',
            },
        });
    });
    test('renders an example with all available props', () => {
        const origin = new lib_1.HttpOrigin({
            domainName: 'www.example.com',
            originPath: '/app',
            connectionTimeout: core_1.Duration.seconds(5),
            connectionAttempts: 2,
            customHeaders: { AUTH: 'NONE' },
            protocolPolicy: lib_1.OriginProtocolPolicy.MATCH_VIEWER,
            httpPort: 8080,
            httpsPort: 8443,
            readTimeout: core_1.Duration.seconds(45),
            keepaliveTimeout: core_1.Duration.seconds(3),
        });
        origin._bind(stack, { originIndex: 0 });
        expect(origin._renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: 'www.example.com',
            originPath: '/app',
            connectionTimeout: 5,
            connectionAttempts: 2,
            originCustomHeaders: [{
                    headerName: 'AUTH',
                    headerValue: 'NONE',
                }],
            customOriginConfig: {
                originProtocolPolicy: 'match-viewer',
                httpPort: 8080,
                httpsPort: 8443,
                originReadTimeout: 45,
                originKeepaliveTimeout: 3,
            },
        });
    });
    test.each([
        core_1.Duration.seconds(0),
        core_1.Duration.seconds(0.5),
        core_1.Duration.seconds(60.5),
        core_1.Duration.seconds(61),
        core_1.Duration.minutes(5),
    ])('validates readTimeout is an integer between 1 and 60 seconds', (readTimeout) => {
        expect(() => {
            new lib_1.HttpOrigin({
                domainName: 'www.example.com',
                readTimeout,
            });
        }).toThrow(`readTimeout: Must be an int between 1 and 60 seconds (inclusive); received ${readTimeout.toSeconds()}.`);
    });
    test.each([
        core_1.Duration.seconds(0),
        core_1.Duration.seconds(0.5),
        core_1.Duration.seconds(60.5),
        core_1.Duration.seconds(61),
        core_1.Duration.minutes(5),
    ])('validates keepaliveTimeout is an integer between 1 and 60 seconds', (keepaliveTimeout) => {
        expect(() => {
            new lib_1.HttpOrigin({
                domainName: 'www.example.com',
                keepaliveTimeout,
            });
        }).toThrow(`keepaliveTimeout: Must be an int between 1 and 60 seconds (inclusive); received ${keepaliveTimeout.toSeconds()}.`);
    });
});
;
describe('Origin', () => {
    test.each([
        core_1.Duration.seconds(0),
        core_1.Duration.seconds(0.5),
        core_1.Duration.seconds(10.5),
        core_1.Duration.seconds(11),
        core_1.Duration.minutes(5),
    ])('validates connectionTimeout is an int between 1 and 10 seconds', (connectionTimeout) => {
        expect(() => {
            new TestOrigin({
                domainName: 'www.example.com',
                connectionTimeout,
            });
        }).toThrow(`connectionTimeout: Must be an int between 1 and 10 seconds (inclusive); received ${connectionTimeout.toSeconds()}.`);
    });
    test.each([-0.5, 0.5, 1.5, 4])('validates connectionAttempts is an int between 1 and 3', (connectionAttempts) => {
        expect(() => {
            new TestOrigin({
                domainName: 'www.example.com',
                connectionAttempts,
            });
        }).toThrow(`connectionAttempts: Must be an int between 1 and 3 (inclusive); received ${connectionAttempts}.`);
    });
    test.each(['api', '/api', '/api/', 'api/'])('enforces that originPath starts but does not end, with a /', (originPath) => {
        const origin = new TestOrigin({
            domainName: 'www.example.com',
            originPath,
        });
        origin._bind(stack, { originIndex: 0 });
        expect(origin._renderOrigin().originPath).toEqual('/api');
    });
});
/** Used for testing common Origin functionality */
class TestOrigin extends lib_1.Origin {
    constructor(props) { super(props); }
    renderS3OriginConfig() {
        return { originAccessIdentity: 'origin-access-identity/cloudfront/MyOAIName' };
    }
}
//# sourceMappingURL=data:application/json;base64,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