"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.GeoRestriction = exports.ViewerCertificate = exports.LambdaEdgeEventType = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginSslPolicy = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.FailoverStatusCode = exports.HttpVersion = void 0;
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const distribution_1 = require("./distribution");
var HttpVersion;
(function (HttpVersion) {
    HttpVersion["HTTP1_1"] = "http1.1";
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * HTTP status code to failover to second origin
 */
var FailoverStatusCode;
(function (FailoverStatusCode) {
    /**
     * Forbidden (403)
     */
    FailoverStatusCode[FailoverStatusCode["FORBIDDEN"] = 403] = "FORBIDDEN";
    /**
     * Not found (404)
     */
    FailoverStatusCode[FailoverStatusCode["NOT_FOUND"] = 404] = "NOT_FOUND";
    /**
     * Internal Server Error (500)
     */
    FailoverStatusCode[FailoverStatusCode["INTERNAL_SERVER_ERROR"] = 500] = "INTERNAL_SERVER_ERROR";
    /**
     * Bad Gateway (502)
     */
    FailoverStatusCode[FailoverStatusCode["BAD_GATEWAY"] = 502] = "BAD_GATEWAY";
    /**
     * Service Unavailable (503)
     */
    FailoverStatusCode[FailoverStatusCode["SERVICE_UNAVAILABLE"] = 503] = "SERVICE_UNAVAILABLE";
    /**
     * Gateway Timeout (504)
     */
    FailoverStatusCode[FailoverStatusCode["GATEWAY_TIMEOUT"] = 504] = "GATEWAY_TIMEOUT";
})(FailoverStatusCode = exports.FailoverStatusCode || (exports.FailoverStatusCode = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Controls the countries in which your content is distributed.
 */
class GeoRestriction {
    /**
     * Creates an instance of GeoRestriction for internal use
     *
     * @param restrictionType Specifies the restriction type to impose (whitelist or blacklist)
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist/blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    constructor(restrictionType, locations) {
        this.restrictionType = restrictionType;
        this.locations = locations;
    }
    /**
     * Whitelist specific countries which you want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static whitelist(...locations) {
        return new GeoRestriction('whitelist', GeoRestriction.validateLocations(locations));
    }
    /**
     * Blacklist specific countries which you don't want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static blacklist(...locations) {
        return new GeoRestriction('blacklist', GeoRestriction.validateLocations(locations));
    }
    static validateLocations(locations) {
        if (locations.length === 0) {
            throw new Error('Should provide at least 1 location');
        }
        locations.forEach(location => {
            if (!GeoRestriction.LOCATION_REGEX.test(location)) {
                // eslint-disable-next-line max-len
                throw new Error(`Invalid location format for location: ${location}, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code`);
            }
        });
        return locations;
    }
}
exports.GeoRestriction = GeoRestriction;
GeoRestriction.LOCATION_REGEX = /^[A-Z]{2}$/;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 * @resource AWS::CloudFront::Distribution
 */
class CloudFrontWebDistribution extends cdk.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [SSLMethod.SNI]: [
                SecurityPolicyProtocol.TLS_V1, SecurityPolicyProtocol.TLS_V1_1_2016,
                SecurityPolicyProtocol.TLS_V1_2016, SecurityPolicyProtocol.TLS_V1_2_2018,
            ],
            [SSLMethod.VIP]: [SecurityPolicyProtocol.SSL_V3, SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || HttpVersion.HTTP2,
            priceClass: props.priceClass || distribution_1.PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // eslint-disable-next-line max-len
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        const originGroups = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            let originId = `origin${originIndex}`;
            const originProperty = this.toOriginProperty(originConfig, originId);
            if (originConfig.failoverCustomOriginSource || originConfig.failoverS3OriginSource) {
                const originSecondaryId = `originSecondary${originIndex}`;
                const originSecondaryProperty = this.toOriginProperty({
                    s3OriginSource: originConfig.failoverS3OriginSource,
                    customOriginSource: originConfig.failoverCustomOriginSource,
                    originPath: originConfig.originPath,
                    originHeaders: originConfig.originHeaders,
                }, originSecondaryId);
                const originGroupsId = `OriginGroup${originIndex}`;
                const failoverCodes = (_a = originConfig.failoverCriteriaStatusCodes) !== null && _a !== void 0 ? _a : [500, 502, 503, 504];
                originGroups.push({
                    id: originGroupsId,
                    members: {
                        items: [{ originId }, { originId: originSecondaryId }],
                        quantity: 2,
                    },
                    failoverCriteria: {
                        statusCodes: {
                            items: failoverCodes,
                            quantity: failoverCodes.length,
                        },
                    },
                });
                originId = originGroupsId;
                origins.push(originSecondaryProperty);
            }
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        const originGroupsDistConfig = originGroups.length > 0
            ? {
                items: originGroups,
                quantity: originGroups.length,
            }
            : undefined;
        distributionConfig = {
            ...distributionConfig,
            origins,
            originGroups: originGroupsDistConfig,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // eslint-disable-next-line max-len
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = { ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || distribution_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.functionArn,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
    toOriginProperty(originConfig, originId) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (!originConfig.s3OriginSource &&
            !originConfig.customOriginSource) {
            throw new Error('There must be at least one origin source - either an s3OriginSource, a customOriginSource');
        }
        if (originConfig.customOriginSource && originConfig.s3OriginSource) {
            throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
        }
        if ([
            originConfig.originHeaders,
            (_a = originConfig.s3OriginSource) === null || _a === void 0 ? void 0 : _a.originHeaders,
            (_b = originConfig.customOriginSource) === null || _b === void 0 ? void 0 : _b.originHeaders,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originHeaders field allowed across origin and failover origins');
        }
        if ([
            originConfig.originPath,
            (_c = originConfig.s3OriginSource) === null || _c === void 0 ? void 0 : _c.originPath,
            (_d = originConfig.customOriginSource) === null || _d === void 0 ? void 0 : _d.originPath,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originPath field allowed across origin and failover origins');
        }
        const headers = (_g = (_e = originConfig.originHeaders) !== null && _e !== void 0 ? _e : (_f = originConfig.s3OriginSource) === null || _f === void 0 ? void 0 : _f.originHeaders) !== null && _g !== void 0 ? _g : (_h = originConfig.customOriginSource) === null || _h === void 0 ? void 0 : _h.originHeaders;
        const originHeaders = [];
        if (headers) {
            Object.keys(headers).forEach((key) => {
                const oHeader = {
                    headerName: key,
                    headerValue: headers[key],
                };
                originHeaders.push(oHeader);
            });
        }
        let s3OriginConfig;
        if (originConfig.s3OriginSource) {
            // first case for backwards compatibility
            if (originConfig.s3OriginSource.originAccessIdentity) {
                // grant CloudFront OriginAccessIdentity read access to S3 bucket
                originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                s3OriginConfig = {
                    originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                };
            }
            else {
                s3OriginConfig = {};
            }
        }
        const connectionAttempts = (_j = originConfig.connectionAttempts) !== null && _j !== void 0 ? _j : 3;
        if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
            throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
        }
        const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
        if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
            throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
        }
        const originProperty = {
            id: originId,
            domainName: originConfig.s3OriginSource
                ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                : originConfig.customOriginSource.domainName,
            originPath: (_m = (_k = originConfig.originPath) !== null && _k !== void 0 ? _k : (_l = originConfig.customOriginSource) === null || _l === void 0 ? void 0 : _l.originPath) !== null && _m !== void 0 ? _m : (_o = originConfig.s3OriginSource) === null || _o === void 0 ? void 0 : _o.originPath,
            originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
            s3OriginConfig,
            customOriginConfig: originConfig.customOriginSource
                ? {
                    httpPort: originConfig.customOriginSource.httpPort || 80,
                    httpsPort: originConfig.customOriginSource.httpsPort || 443,
                    originKeepaliveTimeout: (originConfig.customOriginSource.originKeepaliveTimeout &&
                        originConfig.customOriginSource.originKeepaliveTimeout.toSeconds()) ||
                        5,
                    originReadTimeout: (originConfig.customOriginSource.originReadTimeout &&
                        originConfig.customOriginSource.originReadTimeout.toSeconds()) ||
                        30,
                    originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy ||
                        distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
                    originSslProtocols: originConfig.customOriginSource
                        .allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                }
                : undefined,
            connectionAttempts,
            connectionTimeout,
        };
        return originProperty;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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