"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpOrigin = exports.S3Origin = exports.Origin = void 0;
const core_1 = require("@aws-cdk/core");
const distribution_1 = require("./distribution");
const origin_access_identity_1 = require("./origin_access_identity");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class Origin {
    constructor(props) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = props.domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * Creates a pre-configured origin for a S3 bucket.
     * If this bucket has been configured for static website hosting, then `fromWebsiteBucket` should be used instead.
     *
     * An Origin Access Identity will be created and granted read access to the bucket.
     *
     * @param bucket the bucket to act as an origin.
     */
    static fromBucket(bucket) {
        if (bucket.isWebsite) {
            return new HttpOrigin({
                domainName: bucket.bucketWebsiteDomainName,
                protocolPolicy: distribution_1.OriginProtocolPolicy.HTTP_ONLY,
            });
        }
        else {
            return new S3Origin({ domainName: bucket.bucketRegionalDomainName, bucket });
        }
    }
    /**
     * Creates an origin from an HTTP server.
     */
    static fromHttpServer(props) {
        return new HttpOrigin(props);
    }
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id() {
        if (!this.originId) {
            throw new Error('Cannot access originId until `bind` is called.');
        }
        return this.originId;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     *
     * @internal
     */
    _bind(scope, options) {
        this.originId = new core_1.Construct(scope, `Origin${options.originIndex}`).node.uniqueId;
    }
    /**
     * Creates and returns the CloudFormation representation of this origin.
     *
     * @internal
     */
    _renderOrigin() {
        var _a;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            domainName: this.domainName,
            id: this.id,
            originPath: this.originPath,
            connectionAttempts: this.connectionAttempts,
            connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            originCustomHeaders: this.renderCustomHeaders(),
            s3OriginConfig,
            customOriginConfig,
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.Origin = Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 *
 * @experimental
 */
class S3Origin extends Origin {
    constructor(props) {
        super(props);
        this.bucket = props.bucket;
    }
    /** @internal */
    _bind(scope, options) {
        super._bind(scope, options);
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new origin_access_identity_1.OriginAccessIdentity(scope, `S3Origin${options.originIndex}`);
            this.bucket.grantRead(this.originAccessIdentity);
        }
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin for an HTTP server or S3 bucket configured for website hosting.
 *
 * @experimental
 */
class HttpOrigin extends Origin {
    constructor(props) {
        var _a, _b;
        super(props);
        this.props = props;
        validateIntInRangeOrUndefined('readTimeout', 1, 60, (_a = props.readTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('keepaliveTimeout', 1, 60, (_b = props.keepaliveTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds());
    }
    renderCustomOriginConfig() {
        var _a, _b, _c;
        return {
            originProtocolPolicy: (_a = this.props.protocolPolicy) !== null && _a !== void 0 ? _a : distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: this.props.httpPort,
            httpsPort: this.props.httpsPort,
            originReadTimeout: (_b = this.props.readTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            originKeepaliveTimeout: (_c = this.props.keepaliveTimeout) === null || _c === void 0 ? void 0 : _c.toSeconds(),
        };
    }
}
exports.HttpOrigin = HttpOrigin;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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