import { IBucket } from '@aws-cdk/aws-s3';
import { Duration } from '@aws-cdk/core';
import { CfnDistribution } from './cloudfront.generated';
import { OriginProtocolPolicy } from './distribution';
/**
 * Properties to be used to create an Origin. Prefer to use one of the Origin.from* factory methods rather than
 * instantiating an Origin directly from these properties.
 *
 * @experimental
 */
export interface OriginProps {
    /**
     * The domain name of the Amazon S3 bucket or HTTP server origin.
     */
    readonly domainName: string;
    /**
     * An optional path that CloudFront appends to the origin domain name when CloudFront requests content from the origin.
     * Must begin, but not end, with '/' (e.g., '/production/images').
     *
     * @default '/'
     */
    readonly originPath?: string;
    /**
     * The number of seconds that CloudFront waits when trying to establish a connection to the origin.
     * Valid values are 1-10 seconds, inclusive.
     *
     * @default Duration.seconds(10)
     */
    readonly connectionTimeout?: Duration;
    /**
     * The number of times that CloudFront attempts to connect to the origin; valid values are 1, 2, or 3 attempts.
     *
     * @default 3
     */
    readonly connectionAttempts?: number;
    /**
     * A list of HTTP header names and values that CloudFront adds to requests it sends to the origin.
     *
     * @default {}
     */
    readonly customHeaders?: Record<string, string>;
}
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
export declare abstract class Origin {
    /**
     * Creates a pre-configured origin for a S3 bucket.
     * If this bucket has been configured for static website hosting, then `fromWebsiteBucket` should be used instead.
     *
     * An Origin Access Identity will be created and granted read access to the bucket.
     *
     * @param bucket the bucket to act as an origin.
     */
    static fromBucket(bucket: IBucket): Origin;
    /**
     * Creates an origin from an HTTP server.
     */
    static fromHttpServer(props: HttpOriginProps): Origin;
    /**
     * The domain name of the origin.
     */
    readonly domainName: string;
    private readonly originPath?;
    private readonly connectionTimeout?;
    private readonly connectionAttempts?;
    private readonly customHeaders?;
    private originId?;
    protected constructor(props: OriginProps);
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id(): string;
    protected renderS3OriginConfig(): CfnDistribution.S3OriginConfigProperty | undefined;
    protected renderCustomOriginConfig(): CfnDistribution.CustomOriginConfigProperty | undefined;
    private renderCustomHeaders;
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    private validateOriginPath;
}
/**
 * Properties for an Origin backed by an S3 bucket
 *
 * @experimental
 */
export interface S3OriginProps extends OriginProps {
    /**
     * The bucket to use as an origin.
     */
    readonly bucket: IBucket;
}
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 *
 * @experimental
 */
export declare class S3Origin extends Origin {
    private readonly bucket;
    private originAccessIdentity;
    constructor(props: S3OriginProps);
    protected renderS3OriginConfig(): CfnDistribution.S3OriginConfigProperty | undefined;
}
/**
 * Properties for an Origin backed by an S3 website-configured bucket, load balancer, or custom HTTP server.
 *
 * @experimental
 */
export interface HttpOriginProps extends OriginProps {
    /**
     * Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin.
     *
     * @default OriginProtocolPolicy.HTTPS_ONLY
     */
    readonly protocolPolicy?: OriginProtocolPolicy;
    /**
     * The HTTP port that CloudFront uses to connect to the origin.
     *
     * @default 80
     */
    readonly httpPort?: number;
    /**
     * The HTTPS port that CloudFront uses to connect to the origin.
     *
     * @default 443
     */
    readonly httpsPort?: number;
    /**
     * Specifies how long, in seconds, CloudFront waits for a response from the origin, also known as the origin response timeout.
     * The valid range is from 1 to 60 seconds, inclusive.
     *
     * @default Duration.seconds(30)
     */
    readonly readTimeout?: Duration;
    /**
     * Specifies how long, in seconds, CloudFront persists its connection to the origin.
     * The valid range is from 1 to 60 seconds, inclusive.
     *
     * @default Duration.seconds(5)
     */
    readonly keepaliveTimeout?: Duration;
}
/**
 * An Origin for an HTTP server or S3 bucket configured for website hosting.
 *
 * @experimental
 */
export declare class HttpOrigin extends Origin {
    private readonly props;
    constructor(props: HttpOriginProps);
    protected renderCustomOriginConfig(): CfnDistribution.CustomOriginConfigProperty | undefined;
}
