"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AllowedMethods = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.Distribution = void 0;
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.additionalBehaviors = [];
        this.origins = new Set();
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error('Distribution certificates must be in the us-east-1 region and the certificate you provided is in $Region.');
            }
        }
        this.defaultBehavior = new cache_behavior_1.CacheBehavior({ pathPattern: '*', ...props.defaultBehavior });
        this.addOrigin(this.defaultBehavior.origin);
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_a = props.errorResponses) !== null && _a !== void 0 ? _a : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig: {
                enabled: true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                viewerCertificate: this.certificate ? { acmCertificateArn: this.certificate.certificateArn } : undefined,
                customErrorResponses: this.renderErrorResponses(),
                priceClass: (_b = props.priceClass) !== null && _b !== void 0 ? _b : undefined,
            } });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior({ pathPattern, origin, ...behaviorOptions }));
        this.addOrigin(origin);
    }
    addOrigin(origin) {
        if (!this.origins.has(origin)) {
            this.origins.add(origin);
            origin._bind(this, { originIndex: this.origins.size });
        }
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.origins.forEach(origin => renderedOrigins.push(origin._renderOrigin()));
        return renderedOrigins;
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
}
exports.Distribution = Distribution;
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGlzdHJpYnV0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZGlzdHJpYnV0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHdDQUE2RjtBQUM3RixpRUFBeUQ7QUFFekQsNkRBQXlEO0FBNkZ6RDs7OztHQUlHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsZUFBUTtJQStCeEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF3Qjs7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQVBGLHdCQUFtQixHQUFvQixFQUFFLENBQUM7UUFDMUMsWUFBTyxHQUFnQixJQUFJLEdBQUcsRUFBVSxDQUFDO1FBUXhELElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRTtZQUNyQixNQUFNLGlCQUFpQixHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsTUFBTSxDQUFDO1lBQzNGLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLElBQUksaUJBQWlCLEtBQUssV0FBVyxFQUFFO2dCQUMvRSxNQUFNLElBQUksS0FBSyxDQUFDLDJHQUEyRyxDQUFDLENBQUM7YUFDOUg7U0FDRjtRQUVELElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSw4QkFBYSxDQUFDLEVBQUUsV0FBVyxFQUFFLEdBQUcsRUFBRSxHQUFHLEtBQUssQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO1FBQ3pGLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM1QyxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtZQUM3QixNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLGVBQWUsQ0FBQyxFQUFFLEVBQUU7Z0JBQ25GLElBQUksQ0FBQyxXQUFXLENBQUMsV0FBVyxFQUFFLGVBQWUsQ0FBQyxNQUFNLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDekUsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsY0FBYyxTQUFHLEtBQUssQ0FBQyxjQUFjLG1DQUFJLEVBQUUsQ0FBQztRQUVqRCxNQUFNLFlBQVksR0FBRyxJQUFJLHNDQUFlLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQ3JGLE9BQU8sRUFBRSxJQUFJO2dCQUNiLE9BQU8sRUFBRSxXQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsRUFBRSxDQUFDO2dCQUMvRCxvQkFBb0IsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRTtnQkFDNUQsY0FBYyxFQUFFLFdBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLG9CQUFvQixFQUFFLEVBQUUsQ0FBQztnQkFDN0UsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsRUFBRSxpQkFBaUIsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUN4RyxvQkFBb0IsRUFBRSxJQUFJLENBQUMsb0JBQW9CLEVBQUU7Z0JBQ2pELFVBQVUsUUFBRSxLQUFLLENBQUMsVUFBVSxtQ0FBSSxTQUFTO2FBQzFDLEVBQUUsQ0FBQyxDQUFDO1FBRUwsSUFBSSxDQUFDLFVBQVUsR0FBRyxZQUFZLENBQUMsY0FBYyxDQUFDO1FBQzlDLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxZQUFZLENBQUMsY0FBYyxDQUFDO1FBQzFELElBQUksQ0FBQyxjQUFjLEdBQUcsWUFBWSxDQUFDLEdBQUcsQ0FBQztJQUN6QyxDQUFDO0lBL0REOztPQUVHO0lBQ0ksTUFBTSxDQUFDLDBCQUEwQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTZCO1FBQ2xHLE9BQU8sSUFBSSxLQUFNLFNBQVEsZUFBUTtZQUsvQjtnQkFDRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO2dCQUNqQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7Z0JBQ25DLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO2dCQUMvQyxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDN0MsQ0FBQztTQUNGLEVBQUUsQ0FBQztJQUNOLENBQUM7SUFpREQ7Ozs7O09BS0c7SUFDSSxXQUFXLENBQUMsV0FBbUIsRUFBRSxNQUFjLEVBQUUsa0JBQXNDLEVBQUU7UUFDOUYsSUFBSSxXQUFXLEtBQUssR0FBRyxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztTQUMvRTtRQUNELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsSUFBSSw4QkFBYSxDQUFDLEVBQUUsV0FBVyxFQUFFLE1BQU0sRUFBRSxHQUFHLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUM5RixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ3pCLENBQUM7SUFFTyxTQUFTLENBQUMsTUFBYztRQUM5QixJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDN0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDekIsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1NBQ3hEO0lBQ0gsQ0FBQztJQUVPLGFBQWE7UUFDbkIsTUFBTSxlQUFlLEdBQXFDLEVBQUUsQ0FBQztRQUM3RCxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUM3RSxPQUFPLGVBQWUsQ0FBQztJQUN6QixDQUFDO0lBRU8sb0JBQW9CO1FBQzFCLElBQUksSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFBRSxPQUFPLFNBQVMsQ0FBQztTQUFFO1FBQ2hFLE9BQU8sSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFFTyxvQkFBb0I7UUFDMUIsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFBRSxPQUFPLFNBQVMsQ0FBQztTQUFFO1FBQzNELFNBQVMsMkJBQTJCLENBQUMsYUFBNEI7WUFDL0QsSUFBSSxhQUFhLENBQUMsZ0JBQWdCLElBQUksQ0FBQyxhQUFhLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ3ZFLE1BQU0sSUFBSSxLQUFLLENBQUMsc0VBQXNFLENBQUMsQ0FBQzthQUN6RjtZQUNELElBQUksQ0FBQyxhQUFhLENBQUMsa0JBQWtCLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxFQUFFO2dCQUMzRCxNQUFNLElBQUksS0FBSyxDQUFDLG1HQUFtRyxDQUFDLENBQUM7YUFDdEg7UUFDSCxDQUFDO1FBQ0QsSUFBSSxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRWpFLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUU7O1lBQzNDLE9BQU87Z0JBQ0wsa0JBQWtCLFFBQUUsV0FBVyxDQUFDLEdBQUcsMENBQUUsU0FBUyxFQUFFO2dCQUNoRCxTQUFTLEVBQUUsV0FBVyxDQUFDLFVBQVU7Z0JBQ2pDLFlBQVksRUFBRSxXQUFXLENBQUMsa0JBQWtCO2dCQUM1QyxnQkFBZ0IsRUFBRSxXQUFXLENBQUMsZ0JBQWdCO2FBQy9DLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FFRjtBQXpIRCxvQ0F5SEM7QUFFRDs7O0dBR0c7QUFDSCxJQUFZLFVBT1g7QUFQRCxXQUFZLFVBQVU7SUFDcEIsb0NBQW9DO0lBQ3BDLGdEQUFrQyxDQUFBO0lBQ2xDLDBIQUEwSDtJQUMxSCxnREFBa0MsQ0FBQTtJQUNsQyxvQkFBb0I7SUFDcEIsZ0RBQWtDLENBQUE7QUFDcEMsQ0FBQyxFQVBXLFVBQVUsR0FBVixrQkFBVSxLQUFWLGtCQUFVLFFBT3JCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLG9CQU9YO0FBUEQsV0FBWSxvQkFBb0I7SUFDOUIsaUJBQWlCO0lBQ2pCLGlEQUF5QixDQUFBO0lBQ3pCLDJDQUEyQztJQUMzQywrREFBdUMsQ0FBQTtJQUN2QyxvQ0FBb0M7SUFDcEMsK0NBQXVCLENBQUE7QUFDekIsQ0FBQyxFQVBXLG9CQUFvQixHQUFwQiw0QkFBb0IsS0FBcEIsNEJBQW9CLFFBTy9CO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLG9CQU9YO0FBUEQsV0FBWSxvQkFBb0I7SUFDOUIsMkJBQTJCO0lBQzNCLCtDQUF1QixDQUFBO0lBQ3ZCLG1EQUFtRDtJQUNuRCxxREFBNkIsQ0FBQTtJQUM3Qiw0QkFBNEI7SUFDNUIsaURBQXlCLENBQUE7QUFDM0IsQ0FBQyxFQVBXLG9CQUFvQixHQUFwQiw0QkFBb0IsS0FBcEIsNEJBQW9CLFFBTy9CO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUFXekIsWUFBb0IsT0FBaUIsSUFBSSxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQyxDQUFDLENBQUM7O0FBWHBFLHdDQVlDO0FBWEMsbUJBQW1CO0FBQ0ksNkJBQWMsR0FBRyxJQUFJLGNBQWMsQ0FBQyxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO0FBQzVFLDZCQUE2QjtBQUNOLHFDQUFzQixHQUFHLElBQUksY0FBYyxDQUFDLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO0FBQy9GLGlDQUFpQztBQUNWLHdCQUFTLEdBQUcsSUFBSSxjQUFjLENBQUMsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgYWNtIGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIExhenksIFJlc291cmNlLCBTdGFjaywgVG9rZW4sIER1cmF0aW9uIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDZm5EaXN0cmlidXRpb24gfSBmcm9tICcuL2Nsb3VkZnJvbnQuZ2VuZXJhdGVkJztcbmltcG9ydCB7IE9yaWdpbiB9IGZyb20gJy4vb3JpZ2luJztcbmltcG9ydCB7IENhY2hlQmVoYXZpb3IgfSBmcm9tICcuL3ByaXZhdGUvY2FjaGUtYmVoYXZpb3InO1xuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgQ2xvdWRGcm9udCBkaXN0cmlidXRpb25zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSURpc3RyaWJ1dGlvbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgZG9tYWluIG5hbWUgb2YgdGhlIERpc3RyaWJ1dGlvbiwgc3VjaCBhcyBkMTExMTExYWJjZGVmOC5jbG91ZGZyb250Lm5ldC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKiBAZGVwcmVjYXRlZCAtIFVzZSBgZGlzdHJpYnV0aW9uRG9tYWluTmFtZWAgaW5zdGVhZC5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRvbWFpbiBuYW1lIG9mIHRoZSBEaXN0cmlidXRpb24sIHN1Y2ggYXMgZDExMTExMWFiY2RlZjguY2xvdWRmcm9udC5uZXQuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRpc3RyaWJ1dGlvbkRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRpc3RyaWJ1dGlvbiBJRCBmb3IgdGhpcyBkaXN0cmlidXRpb24uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRpc3RyaWJ1dGlvbklkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQXR0cmlidXRlcyB1c2VkIHRvIGltcG9ydCBhIERpc3RyaWJ1dGlvbi5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGlzdHJpYnV0aW9uQXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBUaGUgZ2VuZXJhdGVkIGRvbWFpbiBuYW1lIG9mIHRoZSBEaXN0cmlidXRpb24sIHN1Y2ggYXMgZDExMTExMWFiY2RlZjguY2xvdWRmcm9udC5uZXQuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRpc3RyaWJ1dGlvbiBJRCBmb3IgdGhpcyBkaXN0cmlidXRpb24uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRpc3RyaWJ1dGlvbklkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBEaXN0cmlidXRpb25cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRGlzdHJpYnV0aW9uUHJvcHMge1xuICAvKipcbiAgICogVGhlIGRlZmF1bHQgYmVoYXZpb3IgZm9yIHRoZSBkaXN0cmlidXRpb24uXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0QmVoYXZpb3I6IEJlaGF2aW9yT3B0aW9ucztcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBiZWhhdmlvcnMgZm9yIHRoZSBkaXN0cmlidXRpb24sIG1hcHBlZCBieSB0aGUgcGF0aFBhdHRlcm4gdGhhdCBzcGVjaWZpZXMgd2hpY2ggcmVxdWVzdHMgdG8gYXBwbHkgdGhlIGJlaGF2aW9yIHRvLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGFkZGl0aW9uYWwgYmVoYXZpb3JzIGFyZSBhZGRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGFkZGl0aW9uYWxCZWhhdmlvcnM/OiBSZWNvcmQ8c3RyaW5nLCBCZWhhdmlvck9wdGlvbnM+O1xuXG4gIC8qKlxuICAgKiBBIGNlcnRpZmljYXRlIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBkaXN0cmlidXRpb24uIFRoZSBjZXJ0aWZpY2F0ZSBtdXN0IGJlIGxvY2F0ZWQgaW4gTi4gVmlyZ2luaWEgKHVzLWVhc3QtMSkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIENsb3VkRnJvbnQgd2lsZGNhcmQgY2VydGlmaWNhdGUgKCouY2xvdWRmcm9udC5uZXQpIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlPzogYWNtLklDZXJ0aWZpY2F0ZTtcblxuICAvKipcbiAgICogVGhlIHByaWNlIGNsYXNzIHRoYXQgY29ycmVzcG9uZHMgd2l0aCB0aGUgbWF4aW11bSBwcmljZSB0aGF0IHlvdSB3YW50IHRvIHBheSBmb3IgQ2xvdWRGcm9udCBzZXJ2aWNlLlxuICAgKiBJZiB5b3Ugc3BlY2lmeSBQcmljZUNsYXNzX0FsbCwgQ2xvdWRGcm9udCByZXNwb25kcyB0byByZXF1ZXN0cyBmb3IgeW91ciBvYmplY3RzIGZyb20gYWxsIENsb3VkRnJvbnQgZWRnZSBsb2NhdGlvbnMuXG4gICAqIElmIHlvdSBzcGVjaWZ5IGEgcHJpY2UgY2xhc3Mgb3RoZXIgdGhhbiBQcmljZUNsYXNzX0FsbCwgQ2xvdWRGcm9udCBzZXJ2ZXMgeW91ciBvYmplY3RzIGZyb20gdGhlIENsb3VkRnJvbnQgZWRnZSBsb2NhdGlvblxuICAgKiB0aGF0IGhhcyB0aGUgbG93ZXN0IGxhdGVuY3kgYW1vbmcgdGhlIGVkZ2UgbG9jYXRpb25zIGluIHlvdXIgcHJpY2UgY2xhc3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IFByaWNlQ2xhc3MuUFJJQ0VfQ0xBU1NfQUxMXG4gICAqL1xuICByZWFkb25seSBwcmljZUNsYXNzPzogUHJpY2VDbGFzcztcblxuICAvKipcbiAgICogSG93IENsb3VkRnJvbnQgc2hvdWxkIGhhbmRsZSByZXF1ZXN0cyB0aGF0IGFyZSBub3Qgc3VjY2Vzc2Z1bCAoZS5nLiwgUGFnZU5vdEZvdW5kKS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBjdXN0b20gZXJyb3IgcmVzcG9uc2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgZXJyb3JSZXNwb25zZXM/OiBFcnJvclJlc3BvbnNlW107XG59XG5cbi8qKlxuICogQSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiB3aXRoIGFzc29jaWF0ZWQgb3JpZ2luKHMpIGFuZCBjYWNoaW5nIGJlaGF2aW9yKHMpLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIERpc3RyaWJ1dGlvbiBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSURpc3RyaWJ1dGlvbiB7XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBEaXN0cmlidXRpb24gY29uc3RydWN0IHRoYXQgcmVwcmVzZW50cyBhbiBleHRlcm5hbCAoaW1wb3J0ZWQpIGRpc3RyaWJ1dGlvbi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbURpc3RyaWJ1dGlvbkF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IERpc3RyaWJ1dGlvbkF0dHJpYnV0ZXMpOiBJRGlzdHJpYnV0aW9uIHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJRGlzdHJpYnV0aW9uIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZGlzdHJpYnV0aW9uRG9tYWluTmFtZTogc3RyaW5nO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGRpc3RyaWJ1dGlvbklkOiBzdHJpbmc7XG5cbiAgICAgIGNvbnN0cnVjdG9yKCkge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICB0aGlzLmRvbWFpbk5hbWUgPSBhdHRycy5kb21haW5OYW1lO1xuICAgICAgICB0aGlzLmRpc3RyaWJ1dGlvbkRvbWFpbk5hbWUgPSBhdHRycy5kb21haW5OYW1lO1xuICAgICAgICB0aGlzLmRpc3RyaWJ1dGlvbklkID0gYXR0cnMuZGlzdHJpYnV0aW9uSWQ7XG4gICAgICB9XG4gICAgfSgpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRpc3RyaWJ1dGlvbkRvbWFpbk5hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRpc3RyaWJ1dGlvbklkOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBkZWZhdWx0QmVoYXZpb3I6IENhY2hlQmVoYXZpb3I7XG4gIHByaXZhdGUgcmVhZG9ubHkgYWRkaXRpb25hbEJlaGF2aW9yczogQ2FjaGVCZWhhdmlvcltdID0gW107XG4gIHByaXZhdGUgcmVhZG9ubHkgb3JpZ2luczogU2V0PE9yaWdpbj4gPSBuZXcgU2V0PE9yaWdpbj4oKTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGVycm9yUmVzcG9uc2VzOiBFcnJvclJlc3BvbnNlW107XG4gIHByaXZhdGUgcmVhZG9ubHkgY2VydGlmaWNhdGU/OiBhY20uSUNlcnRpZmljYXRlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEaXN0cmlidXRpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMuY2VydGlmaWNhdGUpIHtcbiAgICAgIGNvbnN0IGNlcnRpZmljYXRlUmVnaW9uID0gU3RhY2sub2YodGhpcykucGFyc2VBcm4ocHJvcHMuY2VydGlmaWNhdGUuY2VydGlmaWNhdGVBcm4pLnJlZ2lvbjtcbiAgICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKGNlcnRpZmljYXRlUmVnaW9uKSAmJiBjZXJ0aWZpY2F0ZVJlZ2lvbiAhPT0gJ3VzLWVhc3QtMScpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdEaXN0cmlidXRpb24gY2VydGlmaWNhdGVzIG11c3QgYmUgaW4gdGhlIHVzLWVhc3QtMSByZWdpb24gYW5kIHRoZSBjZXJ0aWZpY2F0ZSB5b3UgcHJvdmlkZWQgaXMgaW4gJFJlZ2lvbi4nKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICB0aGlzLmRlZmF1bHRCZWhhdmlvciA9IG5ldyBDYWNoZUJlaGF2aW9yKHsgcGF0aFBhdHRlcm46ICcqJywgLi4ucHJvcHMuZGVmYXVsdEJlaGF2aW9yIH0pO1xuICAgIHRoaXMuYWRkT3JpZ2luKHRoaXMuZGVmYXVsdEJlaGF2aW9yLm9yaWdpbik7XG4gICAgaWYgKHByb3BzLmFkZGl0aW9uYWxCZWhhdmlvcnMpIHtcbiAgICAgIE9iamVjdC5lbnRyaWVzKHByb3BzLmFkZGl0aW9uYWxCZWhhdmlvcnMpLmZvckVhY2goKFtwYXRoUGF0dGVybiwgYmVoYXZpb3JPcHRpb25zXSkgPT4ge1xuICAgICAgICB0aGlzLmFkZEJlaGF2aW9yKHBhdGhQYXR0ZXJuLCBiZWhhdmlvck9wdGlvbnMub3JpZ2luLCBiZWhhdmlvck9wdGlvbnMpO1xuICAgICAgfSk7XG4gICAgfVxuXG4gICAgdGhpcy5jZXJ0aWZpY2F0ZSA9IHByb3BzLmNlcnRpZmljYXRlO1xuICAgIHRoaXMuZXJyb3JSZXNwb25zZXMgPSBwcm9wcy5lcnJvclJlc3BvbnNlcyA/PyBbXTtcblxuICAgIGNvbnN0IGRpc3RyaWJ1dGlvbiA9IG5ldyBDZm5EaXN0cmlidXRpb24odGhpcywgJ0NGRGlzdHJpYnV0aW9uJywgeyBkaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICBvcmlnaW5zOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5yZW5kZXJPcmlnaW5zKCkgfSksXG4gICAgICBkZWZhdWx0Q2FjaGVCZWhhdmlvcjogdGhpcy5kZWZhdWx0QmVoYXZpb3IuX3JlbmRlckJlaGF2aW9yKCksXG4gICAgICBjYWNoZUJlaGF2aW9yczogTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyQ2FjaGVCZWhhdmlvcnMoKSB9KSxcbiAgICAgIHZpZXdlckNlcnRpZmljYXRlOiB0aGlzLmNlcnRpZmljYXRlID8geyBhY21DZXJ0aWZpY2F0ZUFybjogdGhpcy5jZXJ0aWZpY2F0ZS5jZXJ0aWZpY2F0ZUFybiB9IDogdW5kZWZpbmVkLFxuICAgICAgY3VzdG9tRXJyb3JSZXNwb25zZXM6IHRoaXMucmVuZGVyRXJyb3JSZXNwb25zZXMoKSxcbiAgICAgIHByaWNlQ2xhc3M6IHByb3BzLnByaWNlQ2xhc3MgPz8gdW5kZWZpbmVkLFxuICAgIH0gfSk7XG5cbiAgICB0aGlzLmRvbWFpbk5hbWUgPSBkaXN0cmlidXRpb24uYXR0ckRvbWFpbk5hbWU7XG4gICAgdGhpcy5kaXN0cmlidXRpb25Eb21haW5OYW1lID0gZGlzdHJpYnV0aW9uLmF0dHJEb21haW5OYW1lO1xuICAgIHRoaXMuZGlzdHJpYnV0aW9uSWQgPSBkaXN0cmlidXRpb24ucmVmO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgYmVoYXZpb3IgdG8gdGhpcyBkaXN0cmlidXRpb24gZm9yIHRoZSBnaXZlbiBwYXRoUGF0dGVybi5cbiAgICpcbiAgICogQHBhcmFtIHBhdGhQYXR0ZXJuIHRoZSBwYXRoIHBhdHRlcm4gKGUuZy4sICdpbWFnZXMvKicpIHRoYXQgc3BlY2lmaWVzIHdoaWNoIHJlcXVlc3RzIHRvIGFwcGx5IHRoZSBiZWhhdmlvciB0by5cbiAgICogQHBhcmFtIGJlaGF2aW9yT3B0aW9ucyB0aGUgb3B0aW9ucyBmb3IgdGhlIGJlaGF2aW9yIGF0IHRoaXMgcGF0aC5cbiAgICovXG4gIHB1YmxpYyBhZGRCZWhhdmlvcihwYXRoUGF0dGVybjogc3RyaW5nLCBvcmlnaW46IE9yaWdpbiwgYmVoYXZpb3JPcHRpb25zOiBBZGRCZWhhdmlvck9wdGlvbnMgPSB7fSkge1xuICAgIGlmIChwYXRoUGF0dGVybiA9PT0gJyonKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgdGhlIGRlZmF1bHQgYmVoYXZpb3IgY2FuIGhhdmUgYSBwYXRoIHBhdHRlcm4gb2YgXFwnKlxcJycpO1xuICAgIH1cbiAgICB0aGlzLmFkZGl0aW9uYWxCZWhhdmlvcnMucHVzaChuZXcgQ2FjaGVCZWhhdmlvcih7IHBhdGhQYXR0ZXJuLCBvcmlnaW4sIC4uLmJlaGF2aW9yT3B0aW9ucyB9KSk7XG4gICAgdGhpcy5hZGRPcmlnaW4ob3JpZ2luKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkT3JpZ2luKG9yaWdpbjogT3JpZ2luKSB7XG4gICAgaWYgKCF0aGlzLm9yaWdpbnMuaGFzKG9yaWdpbikpIHtcbiAgICAgIHRoaXMub3JpZ2lucy5hZGQob3JpZ2luKTtcbiAgICAgIG9yaWdpbi5fYmluZCh0aGlzLCB7IG9yaWdpbkluZGV4OiB0aGlzLm9yaWdpbnMuc2l6ZSB9KTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHJlbmRlck9yaWdpbnMoKTogQ2ZuRGlzdHJpYnV0aW9uLk9yaWdpblByb3BlcnR5W10ge1xuICAgIGNvbnN0IHJlbmRlcmVkT3JpZ2luczogQ2ZuRGlzdHJpYnV0aW9uLk9yaWdpblByb3BlcnR5W10gPSBbXTtcbiAgICB0aGlzLm9yaWdpbnMuZm9yRWFjaChvcmlnaW4gPT4gcmVuZGVyZWRPcmlnaW5zLnB1c2gob3JpZ2luLl9yZW5kZXJPcmlnaW4oKSkpO1xuICAgIHJldHVybiByZW5kZXJlZE9yaWdpbnM7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlckNhY2hlQmVoYXZpb3JzKCk6IENmbkRpc3RyaWJ1dGlvbi5DYWNoZUJlaGF2aW9yUHJvcGVydHlbXSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKHRoaXMuYWRkaXRpb25hbEJlaGF2aW9ycy5sZW5ndGggPT09IDApIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuICAgIHJldHVybiB0aGlzLmFkZGl0aW9uYWxCZWhhdmlvcnMubWFwKGJlaGF2aW9yID0+IGJlaGF2aW9yLl9yZW5kZXJCZWhhdmlvcigpKTtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyRXJyb3JSZXNwb25zZXMoKTogQ2ZuRGlzdHJpYnV0aW9uLkN1c3RvbUVycm9yUmVzcG9uc2VQcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAodGhpcy5lcnJvclJlc3BvbnNlcy5sZW5ndGggPT09IDApIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuICAgIGZ1bmN0aW9uIHZhbGlkYXRlQ3VzdG9tRXJyb3JSZXNwb25zZShlcnJvclJlc3BvbnNlOiBFcnJvclJlc3BvbnNlKSB7XG4gICAgICBpZiAoZXJyb3JSZXNwb25zZS5yZXNwb25zZVBhZ2VQYXRoICYmICFlcnJvclJlc3BvbnNlLnJlc3BvbnNlSHR0cFN0YXR1cykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1xcJ3Jlc3BvbnNlQ29kZVxcJyBtdXN0IGJlIHByb3ZpZGVkIGlmIFxcJ3Jlc3BvbnNlUGFnZVBhdGhcXCcgaXMgZGVmaW5lZCcpO1xuICAgICAgfVxuICAgICAgaWYgKCFlcnJvclJlc3BvbnNlLnJlc3BvbnNlSHR0cFN0YXR1cyAmJiAhZXJyb3JSZXNwb25zZS50dGwpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdBIGN1c3RvbSBlcnJvciByZXNwb25zZSB3aXRob3V0IGVpdGhlciBhIFxcJ3Jlc3BvbnNlQ29kZVxcJyBvciBcXCdlcnJvckNhY2hpbmdNaW5UdGxcXCcgaXMgbm90IHZhbGlkLicpO1xuICAgICAgfVxuICAgIH1cbiAgICB0aGlzLmVycm9yUmVzcG9uc2VzLmZvckVhY2goZSA9PiB2YWxpZGF0ZUN1c3RvbUVycm9yUmVzcG9uc2UoZSkpO1xuXG4gICAgcmV0dXJuIHRoaXMuZXJyb3JSZXNwb25zZXMubWFwKGVycm9yQ29uZmlnID0+IHtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIGVycm9yQ2FjaGluZ01pblR0bDogZXJyb3JDb25maWcudHRsPy50b1NlY29uZHMoKSxcbiAgICAgICAgZXJyb3JDb2RlOiBlcnJvckNvbmZpZy5odHRwU3RhdHVzLFxuICAgICAgICByZXNwb25zZUNvZGU6IGVycm9yQ29uZmlnLnJlc3BvbnNlSHR0cFN0YXR1cyxcbiAgICAgICAgcmVzcG9uc2VQYWdlUGF0aDogZXJyb3JDb25maWcucmVzcG9uc2VQYWdlUGF0aCxcbiAgICAgIH07XG4gICAgfSk7XG4gIH1cblxufVxuXG4vKipcbiAqIFRoZSBwcmljZSBjbGFzcyBkZXRlcm1pbmVzIGhvdyBtYW55IGVkZ2UgbG9jYXRpb25zIENsb3VkRnJvbnQgd2lsbCB1c2UgZm9yIHlvdXIgZGlzdHJpYnV0aW9uLlxuICogU2VlIGh0dHBzOi8vYXdzLmFtYXpvbi5jb20vY2xvdWRmcm9udC9wcmljaW5nLyBmb3IgZnVsbCBsaXN0IG9mIHN1cHBvcnRlZCByZWdpb25zLlxuICovXG5leHBvcnQgZW51bSBQcmljZUNsYXNzIHtcbiAgLyoqIFVTQSwgQ2FuYWRhLCBFdXJvcGUsICYgSXNyYWVsICovXG4gIFBSSUNFX0NMQVNTXzEwMCA9ICdQcmljZUNsYXNzXzEwMCcsXG4gIC8qKiBQUklDRV9DTEFTU18xMDAgKyBTb3V0aCBBZnJpY2EsIEtlbnlhLCBNaWRkbGUgRWFzdCwgSmFwYW4sIFNpbmdhcG9yZSwgU291dGggS29yZWEsIFRhaXdhbiwgSG9uZyBLb25nLCAmIFBoaWxpcHBpbmVzICovXG4gIFBSSUNFX0NMQVNTXzIwMCA9ICdQcmljZUNsYXNzXzIwMCcsXG4gIC8qKiBBbGwgbG9jYXRpb25zICovXG4gIFBSSUNFX0NMQVNTX0FMTCA9ICdQcmljZUNsYXNzX0FsbCdcbn1cblxuLyoqXG4gKiBIb3cgSFRUUHMgc2hvdWxkIGJlIGhhbmRsZWQgd2l0aCB5b3VyIGRpc3RyaWJ1dGlvbi5cbiAqL1xuZXhwb3J0IGVudW0gVmlld2VyUHJvdG9jb2xQb2xpY3kge1xuICAvKiogSFRUUFMgb25seSAqL1xuICBIVFRQU19PTkxZID0gJ2h0dHBzLW9ubHknLFxuICAvKiogV2lsbCByZWRpcmVjdCBIVFRQIHJlcXVlc3RzIHRvIEhUVFBTICovXG4gIFJFRElSRUNUX1RPX0hUVFBTID0gJ3JlZGlyZWN0LXRvLWh0dHBzJyxcbiAgLyoqIEJvdGggSFRUUCBhbmQgSFRUUFMgc3VwcG9ydGVkICovXG4gIEFMTE9XX0FMTCA9ICdhbGxvdy1hbGwnXG59XG5cbi8qKlxuICogRGVmaW5lcyB3aGF0IHByb3RvY29scyBDbG91ZEZyb250IHdpbGwgdXNlIHRvIGNvbm5lY3QgdG8gYW4gb3JpZ2luLlxuICovXG5leHBvcnQgZW51bSBPcmlnaW5Qcm90b2NvbFBvbGljeSB7XG4gIC8qKiBDb25uZWN0IG9uIEhUVFAgb25seSAqL1xuICBIVFRQX09OTFkgPSAnaHR0cC1vbmx5JyxcbiAgLyoqIENvbm5lY3Qgd2l0aCB0aGUgc2FtZSBwcm90b2NvbCBhcyB0aGUgdmlld2VyICovXG4gIE1BVENIX1ZJRVdFUiA9ICdtYXRjaC12aWV3ZXInLFxuICAvKiogQ29ubmVjdCBvbiBIVFRQUyBvbmx5ICovXG4gIEhUVFBTX09OTFkgPSAnaHR0cHMtb25seScsXG59XG5cbi8qKlxuICogVGhlIEhUVFAgbWV0aG9kcyB0aGF0IHRoZSBCZWhhdmlvciB3aWxsIGFjY2VwdCByZXF1ZXN0cyBvbi5cbiAqL1xuZXhwb3J0IGNsYXNzIEFsbG93ZWRNZXRob2RzIHtcbiAgLyoqIEhFQUQgYW5kIEdFVCAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEFMTE9XX0dFVF9IRUFEID0gbmV3IEFsbG93ZWRNZXRob2RzKFsnR0VUJywgJ0hFQUQnXSk7XG4gIC8qKiBIRUFELCBHRVQsIGFuZCBPUFRJT05TICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQUxMT1dfR0VUX0hFQURfT1BUSU9OUyA9IG5ldyBBbGxvd2VkTWV0aG9kcyhbJ0dFVCcsICdIRUFEJywgJ09QVElPTlMnXSk7XG4gIC8qKiBBbGwgc3VwcG9ydGVkIEhUVFAgbWV0aG9kcyAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEFMTE9XX0FMTCA9IG5ldyBBbGxvd2VkTWV0aG9kcyhbJ0dFVCcsICdIRUFEJywgJ09QVElPTlMnLCAnUFVUJywgJ1BBVENIJywgJ1BPU1QnLCAnREVMRVRFJ10pO1xuXG4gIC8qKiBIVFRQIG1ldGhvZHMgc3VwcG9ydGVkICovXG4gIHB1YmxpYyByZWFkb25seSBtZXRob2RzOiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKG1ldGhvZHM6IHN0cmluZ1tdKSB7IHRoaXMubWV0aG9kcyA9IG1ldGhvZHM7IH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBjb25maWd1cmluZyBjdXN0b20gZXJyb3IgcmVzcG9uc2VzLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFcnJvclJlc3BvbnNlIHtcbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIGFtb3VudCBvZiB0aW1lLCBpbiBzZWNvbmRzLCB0aGF0IHlvdSB3YW50IENsb3VkRnJvbnQgdG8gY2FjaGUgdGhlIEhUVFAgc3RhdHVzIGNvZGUgc3BlY2lmaWVkIGluIEVycm9yQ29kZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0aGUgZGVmYXVsdCBjYWNoaW5nIFRUTCBiZWhhdmlvciBhcHBsaWVzXG4gICAqL1xuICByZWFkb25seSB0dGw/OiBEdXJhdGlvbjtcbiAgLyoqXG4gICAqIFRoZSBIVFRQIHN0YXR1cyBjb2RlIGZvciB3aGljaCB5b3Ugd2FudCB0byBzcGVjaWZ5IGEgY3VzdG9tIGVycm9yIHBhZ2UgYW5kL29yIGEgY2FjaGluZyBkdXJhdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGh0dHBTdGF0dXM6IG51bWJlcjtcbiAgLyoqXG4gICAqIFRoZSBIVFRQIHN0YXR1cyBjb2RlIHRoYXQgeW91IHdhbnQgQ2xvdWRGcm9udCB0byByZXR1cm4gdG8gdGhlIHZpZXdlciBhbG9uZyB3aXRoIHRoZSBjdXN0b20gZXJyb3IgcGFnZS5cbiAgICpcbiAgICogSWYgeW91IHNwZWNpZnkgYSB2YWx1ZSBmb3IgYHJlc3BvbnNlSHR0cFN0YXR1c2AsIHlvdSBtdXN0IGFsc28gc3BlY2lmeSBhIHZhbHVlIGZvciBgcmVzcG9uc2VQYWdlUGF0aGAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm90IHNldCwgdGhlIGVycm9yIGNvZGUgd2lsbCBiZSByZXR1cm5lZCBhcyB0aGUgcmVzcG9uc2UgY29kZS5cbiAgICovXG4gIHJlYWRvbmx5IHJlc3BvbnNlSHR0cFN0YXR1cz86IG51bWJlcjtcbiAgLyoqXG4gICAqIFRoZSBwYXRoIHRvIHRoZSBjdXN0b20gZXJyb3IgcGFnZSB0aGF0IHlvdSB3YW50IENsb3VkRnJvbnQgdG8gcmV0dXJuIHRvIGEgdmlld2VyIHdoZW4geW91ciBvcmlnaW4gcmV0dXJucyB0aGVcbiAgICogYGh0dHBTdGF0dXNgLCBmb3IgZXhhbXBsZSwgLzR4eC1lcnJvcnMvNDAzLWZvcmJpZGRlbi5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRlZmF1bHQgQ2xvdWRGcm9udCByZXNwb25zZSBpcyBzaG93bi5cbiAgICovXG4gIHJlYWRvbmx5IHJlc3BvbnNlUGFnZVBhdGg/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYWRkaW5nIGEgbmV3IGJlaGF2aW9yIHRvIGEgRGlzdHJpYnV0aW9uLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBZGRCZWhhdmlvck9wdGlvbnMge1xuICAvKipcbiAgICogSFRUUCBtZXRob2RzIHRvIGFsbG93IGZvciB0aGlzIGJlaGF2aW9yLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEdFVCBhbmQgSEVBRFxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb3dlZE1ldGhvZHM/OiBBbGxvd2VkTWV0aG9kcztcblxuICAvKipcbiAgICogV2hldGhlciBDbG91ZEZyb250IHdpbGwgZm9yd2FyZCBxdWVyeSBzdHJpbmdzIHRvIHRoZSBvcmlnaW4uXG4gICAqIElmIHRoaXMgaXMgc2V0IHRvIHRydWUsIENsb3VkRnJvbnQgd2lsbCBmb3J3YXJkIGFsbCBxdWVyeSBwYXJhbWV0ZXJzIHRvIHRoZSBvcmlnaW4sIGFuZCBjYWNoZVxuICAgKiBiYXNlZCBvbiBhbGwgcGFyYW1ldGVycy4gU2VlIGBmb3J3YXJkUXVlcnlTdHJpbmdDYWNoZUtleXNgIGZvciBhIHdheSB0byBsaW1pdCB0aGUgcXVlcnkgcGFyYW1ldGVyc1xuICAgKiBDbG91ZEZyb250IGNhY2hlcyBvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGZvcndhcmRRdWVyeVN0cmluZz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgc2V0IG9mIHF1ZXJ5IHN0cmluZyBwYXJhbWV0ZXIgbmFtZXMgdG8gdXNlIGZvciBjYWNoaW5nIGlmIGBmb3J3YXJkUXVlcnlTdHJpbmdgIGlzIHNldCB0byB0cnVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBbXVxuICAgKi9cbiAgcmVhZG9ubHkgZm9yd2FyZFF1ZXJ5U3RyaW5nQ2FjaGVLZXlzPzogc3RyaW5nW107XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgY3JlYXRpbmcgYSBuZXcgYmVoYXZpb3IuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEJlaGF2aW9yT3B0aW9ucyBleHRlbmRzIEFkZEJlaGF2aW9yT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgb3JpZ2luIHRoYXQgeW91IHdhbnQgQ2xvdWRGcm9udCB0byByb3V0ZSByZXF1ZXN0cyB0byB3aGVuIHRoZXkgbWF0Y2ggdGhpcyBiZWhhdmlvci5cbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbjogT3JpZ2luO1xufVxuIl19