import React, { useMemo, useState, useEffect } from 'react';
import { createUseStyles } from 'react-jss';
import Select from 'react-select';
import { useStoreState } from 'pullstate';
import { UIStore, resetRucioCaches } from '../stores/UIStore';
import { Button } from '../components/Button';
import { withRequestAPI } from '../utils/Actions';
import { authTypeOptions } from '../const';
import { UserPassAuth } from '../components/@Settings/UserPassAuth';
import { X509Auth } from '../components/@Settings/X509Auth';
import { HorizontalHeading } from '../components/HorizontalHeading';
const useStyles = createUseStyles({
    content: {
        height: '100%',
        display: 'flex',
        overflow: 'auto',
        flexDirection: 'column'
    },
    scrollable: {
        flex: 1,
        paddingTop: '8px'
    },
    container: {
        padding: '8px 16px 8px 16px'
    },
    buttonContainer: {
        extend: 'container'
    },
    instanceName: {
        fontSize: '16pt'
    },
    formControl: {
        marginTop: '8px'
    },
    formItem: {
        marginBottom: '16px'
    },
    label: {
        margin: '4px 0 4px 0'
    },
    textFieldContainer: {
        margin: '8px 0 8px 0'
    },
    subtitle: {
        color: 'var(--jp-ui-font-color2)',
        fontSize: '9pt'
    },
    warning: {
        extend: 'subtitle',
        margin: '8px 8px 16px 8px'
    },
    icon: {
        fontSize: '10pt',
        verticalAlign: 'middle'
    },
    iconText: {
        verticalAlign: 'middle',
        paddingLeft: '4px'
    },
    hidden: {
        display: 'none'
    },
    action: {
        cursor: 'pointer',
        color: 'var(--jp-rucio-primary-blue-color)',
        fontSize: '9pt'
    },
    buttonSavedAcknowledgement: {
        background: '#689f38',
        color: '#ffffff',
        '&:hover': {
            background: '#689f38'
        }
    },
    buttonPurgedAcknowledgement: {
        extend: 'purgeButton',
        background: 'var(--jp-error-color1)',
        color: '#ffffff',
        '&:hover': {
            background: 'var(--jp-error-color1)'
        }
    },
    purgeButton: {
        marginTop: '16px'
    }
});
const _Settings = props => {
    const { actions } = props;
    const classes = useStyles();
    const activeInstance = useStoreState(UIStore, s => s.activeInstance);
    const activeAuthType = useStoreState(UIStore, s => s.activeAuthType);
    const instances = useStoreState(UIStore, s => s.instances) || [];
    const instanceDefaultValue = activeInstance ? { label: activeInstance.displayName, value: activeInstance.name } : null;
    const authTypeDefaultValue = activeAuthType ? authTypeOptions.find(o => o.value === activeAuthType) : null;
    const [selectedInstance, setSelectedInstance] = useState(instanceDefaultValue === null || instanceDefaultValue === void 0 ? void 0 : instanceDefaultValue.value);
    const [selectedAuthType, setSelectedAuthType] = useState(activeAuthType);
    const [rucioUserpassAuthCredentials, setRucioUserpassAuthCredentials] = useState();
    const [rucioX509AuthCredentials, setRucioX509AuthCredentials] = useState();
    const [credentialsLoading, setCredentialsLoading] = useState(true);
    const [loading, setLoading] = useState(false);
    const [showSaved, setShowSaved] = useState(false);
    const [showAdvancedSettings, setShowAdvancedSettings] = useState(false);
    const [purgingCache, setPurgingCache] = useState(false);
    const [showCachePurged, setShowCachePurged] = useState(false);
    const instanceOptions = useMemo(() => instances === null || instances === void 0 ? void 0 : instances.map(i => ({ label: i.displayName, value: i.name })), [instances]);
    const setActiveInstance = (instanceName, authType) => {
        UIStore.update(s => {
            var _a;
            if (((_a = s.activeInstance) === null || _a === void 0 ? void 0 : _a.name) !== instanceName) {
                resetRucioCaches();
                const instance = instances.find(i => i.name === instanceName);
                s.activeInstance = instance;
            }
            if (s.activeAuthType !== authType) {
                s.activeAuthType = authType;
            }
        });
        return actions.postActiveInstance(instanceName, authType).catch(e => console.log(e));
    };
    const saveSettings = () => {
        const promises = [];
        if (selectedInstance && selectedAuthType) {
            const setActiveInstancePromise = setActiveInstance(selectedInstance, selectedAuthType);
            promises.push(setActiveInstancePromise);
        }
        if (selectedAuthType) {
            const rucioAuthCredentials = selectedAuthType === 'userpass' ? rucioUserpassAuthCredentials : rucioX509AuthCredentials;
            if (rucioAuthCredentials) {
                const setPutAuthConfigPromise = actions.putAuthConfig(selectedInstance, selectedAuthType, rucioAuthCredentials);
                promises.push(setPutAuthConfigPromise);
            }
        }
        setLoading(true);
        Promise.all(promises)
            .then(() => {
            setShowSaved(true);
            setTimeout(() => setShowSaved(false), 3000);
        })
            .finally(() => setLoading(false));
    };
    const reloadAuthConfig = () => {
        if (!selectedInstance) {
            return;
        }
        setCredentialsLoading(true);
        if (selectedAuthType === 'userpass') {
            actions
                .fetchAuthConfig(selectedInstance, selectedAuthType)
                .then(c => setRucioUserpassAuthCredentials(c))
                .catch(() => setRucioUserpassAuthCredentials(undefined))
                .finally(() => setCredentialsLoading(false));
        }
        else if (selectedAuthType === 'x509') {
            actions
                .fetchAuthConfig(selectedInstance, selectedAuthType)
                .then(c => setRucioX509AuthCredentials(c))
                .catch(() => setRucioX509AuthCredentials(undefined))
                .finally(() => setCredentialsLoading(false));
        }
    };
    const purgeCache = () => {
        setShowCachePurged(false);
        setPurgingCache(true);
        actions
            .purgeCache()
            .then(() => {
            setShowCachePurged(true);
            setTimeout(() => setShowCachePurged(false), 3000);
        })
            .finally(() => setPurgingCache(false));
    };
    useEffect(reloadAuthConfig, [selectedInstance, selectedAuthType]);
    const settingsComplete = selectedInstance && selectedAuthType;
    const selectStyles = {
        control: (provided, state) => (Object.assign(Object.assign({}, provided), { borderRadius: 0, borderColor: 'var(--jp-border-color1)', background: 'var(--jp-layout-color1)' })),
        singleValue: (provided, state) => (Object.assign(Object.assign({}, provided), { color: 'var(--jp-ui-font-color1)' })),
        menu: (provided, state) => (Object.assign(Object.assign({}, provided), { background: 'var(--jp-layout-color1)', color: 'var(--jp-ui-font-color1)' })),
        option: (provided, { isFocused, isSelected }) => (Object.assign(Object.assign({}, provided), { background: isFocused ? (isSelected ? provided.background : 'var(--jp-layout-color2)') : provided.background, ':active': Object.assign(Object.assign({}, provided[':active']), { background: isSelected ? provided.background : 'var(--jp-layout-color2)' }) }))
    };
    return (React.createElement("div", { className: classes.content },
        React.createElement("div", { className: classes.scrollable },
            React.createElement("div", { className: classes.container },
                React.createElement("div", { className: classes.formItem },
                    React.createElement("div", { className: classes.label }, "Active Instance"),
                    React.createElement(Select, { className: classes.formControl, options: instanceOptions, styles: selectStyles, defaultValue: instanceDefaultValue, onChange: (value) => {
                            setSelectedInstance(value.value);
                        } })),
                React.createElement("div", { className: classes.formItem },
                    React.createElement("div", { className: classes.label }, "Rucio Authentication"),
                    React.createElement(Select, { className: classes.formControl, options: authTypeOptions, styles: selectStyles, defaultValue: authTypeDefaultValue, onChange: (value) => {
                            setSelectedAuthType(value.value);
                        } }))),
            React.createElement("div", null,
                React.createElement("div", { className: selectedInstance && selectedAuthType === 'userpass' ? '' : classes.hidden },
                    React.createElement(HorizontalHeading, { title: "Username & Password" }),
                    React.createElement(UserPassAuth, { loading: credentialsLoading, params: rucioUserpassAuthCredentials, onAuthParamsChange: v => setRucioUserpassAuthCredentials(v) })),
                React.createElement("div", { className: selectedInstance && selectedAuthType === 'x509' ? '' : classes.hidden },
                    React.createElement(HorizontalHeading, { title: "X.509 User Certificate" }),
                    React.createElement(X509Auth, { loading: credentialsLoading, params: rucioX509AuthCredentials, onAuthParamsChange: v => setRucioX509AuthCredentials(v) }))),
            React.createElement("div", { className: showAdvancedSettings ? undefined : classes.hidden },
                React.createElement(HorizontalHeading, { title: "Advanced Settings" }),
                React.createElement("div", { className: classes.container },
                    React.createElement("div", { className: classes.formItem },
                        React.createElement("div", { className: classes.textFieldContainer },
                            React.createElement("div", { className: classes.label }, "Purge cache"),
                            React.createElement("div", { className: classes.subtitle }, "Remove all caches, including search results and DID paths."),
                            React.createElement(Button, { block: true, onClick: purgeCache, disabled: purgingCache, outlineColor: "var(--jp-error-color1)", color: !purgingCache && showCachePurged ? '#FFFFFF' : 'var(--jp-error-color1)', className: !purgingCache && showCachePurged ? classes.buttonPurgedAcknowledgement : classes.purgeButton },
                                !purgingCache && !showCachePurged && React.createElement(React.Fragment, null, "Purge Cache"),
                                purgingCache && React.createElement(React.Fragment, null, "Purging..."),
                                !purgingCache && showCachePurged && React.createElement(React.Fragment, null, "Purged!")))))),
            React.createElement("div", { className: classes.container },
                !showAdvancedSettings && (React.createElement("div", { className: classes.action, onClick: () => setShowAdvancedSettings(true) }, "Show Advanced Settings")),
                showAdvancedSettings && (React.createElement("div", { className: classes.action, onClick: () => setShowAdvancedSettings(false) }, "Hide Advanced Settings")))),
        React.createElement("div", { className: classes.buttonContainer },
            React.createElement(Button, { block: true, onClick: saveSettings, disabled: !settingsComplete || loading, outlineColor: !loading && showSaved ? '#689f38' : undefined, color: !loading && showSaved ? '#FFFFFF' : undefined, className: !loading && showSaved ? classes.buttonSavedAcknowledgement : undefined },
                !loading && !showSaved && React.createElement(React.Fragment, null, "Save Settings"),
                loading && React.createElement(React.Fragment, null, "Saving..."),
                !loading && showSaved && React.createElement(React.Fragment, null, "Saved!")))));
};
export const Settings = withRequestAPI(_Settings);
