import React, { useState, useEffect, useCallback } from 'react';
import Popover from 'react-popover';
import CopyToClipboard from 'react-copy-to-clipboard';
import { createUseStyles } from 'react-jss';
import { FixedSizeList } from 'react-window';
import { useStoreState } from 'pullstate';
import { UIStore } from '../../stores/UIStore';
import { actions } from '../../utils/Actions';
import { Spinning } from '../Spinning';
import { toHumanReadableSize } from '../../utils/Helpers';
const useStyles = createUseStyles({
    main: {
        width: '300px',
        background: 'var(--jp-layout-color1)',
        color: 'var(--jp-ui-font-color1)'
    },
    heading: {
        background: 'var(--jp-layout-color2)',
        color: 'var(--jp-ui-font-color2)',
        padding: '8px',
        borderBottom: '1px solid var(--jp-border-color2)',
        fontSize: '9pt',
        display: 'flex',
        flexDirection: 'row'
    },
    headingText: {
        textOverflow: 'ellipsis',
        overflow: 'hidden',
        whiteSpace: 'nowrap',
        flex: 1
    },
    content: {
        fontSize: '10pt',
        overflow: 'auto',
        maxHeight: '250px',
        '&.loading': {
            opacity: 0.4,
            pointerEvents: 'none'
        }
    },
    loading: {
        padding: '8px',
        boxSizing: 'border-box'
    },
    icon: {
        fontSize: '16px',
        verticalAlign: 'middle'
    },
    iconText: {
        fontSize: '9pt',
        verticalAlign: 'middle',
        paddingLeft: '4px'
    },
    headingCloseButton: {
        extend: 'icon',
        cursor: 'pointer'
    },
    listItem: {
        display: 'flex',
        padding: '8px',
        borderBottom: '1px solid var(--jp-border-color2)',
        flexDirection: 'row',
        fontSize: '9pt',
        alignItems: 'center',
        boxSizing: 'border-box',
        overflow: 'hidden'
    },
    textContainer: {
        flex: 1,
        textOverflow: 'ellipsis',
        overflow: 'hidden',
        whiteSpace: 'nowrap',
        cursor: 'pointer',
        '&:hover': {
            color: 'var(--jp-rucio-primary-blue-color)'
        },
        '& .copy': {
            display: 'none',
            fontSize: '12px'
        },
        '&:hover .copy': {
            display: 'inline'
        }
    },
    iconContainer: {
        lineHeight: 0,
        marginRight: '8px'
    },
    sizeContainer: {
        color: 'var(--jp-ui-font-color2)'
    },
    fileIcon: {
        extend: 'icon',
        color: '#66B100'
    },
    loadingIcon: {
        fontSize: '9pt',
        verticalAlign: 'middle'
    }
});
export const ListAttachedFilesPopover = ({ children, did }) => {
    const classes = useStyles();
    const [open, setOpen] = useState(false);
    const [loading, setLoading] = useState(false);
    const [files, setFiles] = useState([]);
    const activeInstance = useStoreState(UIStore, s => s.activeInstance);
    const escFunction = useCallback(event => {
        if (event.keyCode === 27) {
            setOpen(false);
        }
    }, []);
    useEffect(() => {
        document.addEventListener('keydown', escFunction, false);
        return () => {
            document.removeEventListener('keydown', escFunction, false);
        };
    }, []);
    const Row = ({ index, style }) => {
        const file = files[index];
        return React.createElement(ListItem, { style: style, did: file.did, size: file.size });
    };
    const popoverBody = (React.createElement("div", { className: classes.main },
        React.createElement("div", { className: classes.heading },
            React.createElement("div", { className: classes.headingText },
                "Files of ",
                React.createElement("b", null, did)),
            React.createElement("i", { className: `${classes.headingCloseButton} material-icons`, onClick: () => setOpen(false) }, "close")),
        React.createElement("div", { className: classes.content },
            loading && (React.createElement("div", { className: classes.loading },
                React.createElement(Spinning, { className: `${classes.loadingIcon} material-icons` }, "hourglass_top"),
                React.createElement("span", { className: classes.iconText }, "Loading..."))),
            React.createElement(FixedSizeList, { height: Math.min(250, 32 * files.length), itemCount: files.length, itemSize: 32, width: "100%" }, Row))));
    const loadAttachedFiles = () => {
        setLoading(true);
        setFiles([]);
        actions
            .fetchAttachedFileDIDs(activeInstance.name, did)
            .then(result => setFiles(result))
            .catch(e => console.log(e)) // TODO handle error
            .finally(() => setLoading(false));
    };
    const openPopover = () => {
        setOpen(true);
        loadAttachedFiles();
    };
    return (React.createElement(Popover, { enterExitTransitionDurationMs: 0, isOpen: open, preferPlace: "below", body: popoverBody, onOuterAction: () => setOpen(false) },
        React.createElement("div", { onClick: openPopover }, children)));
};
const ListItem = ({ did, size, style }) => {
    const classes = useStyles();
    return (React.createElement("div", { className: classes.listItem, style: style },
        React.createElement("div", { className: classes.iconContainer },
            React.createElement("i", { className: `${classes.fileIcon} material-icons` }, "attachment")),
        React.createElement(CopyToClipboard, { text: did },
            React.createElement("div", { className: classes.textContainer },
                did,
                " ",
                React.createElement("i", { className: "material-icons copy" }, "file_copy"))),
        !!size && React.createElement("div", { className: classes.sizeContainer }, toHumanReadableSize(size))));
};
