import React, { useMemo } from 'react';
import { VDomRenderer } from '@jupyterlab/apputils';
import { createUseStyles } from 'react-jss';
import { useNotebookResolveStatusStore } from './utils/NotebookListener';
import { Spinning } from './components/Spinning';
import { rucioIcon } from './icons/RucioIcon';
const useStyles = createUseStyles({
    main: {
        display: 'flex',
        flexDirection: 'row',
        alignItems: 'center',
        padding: '0 8px 0 8px',
        borderRadius: '4px',
        cursor: 'pointer',
        '&:hover': {
            background: 'var(--jp-layout-color2)'
        }
    },
    ready: {
        color: 'var(--jp-success-color0)'
    },
    failed: {
        color: 'var(--jp-error-color1)'
    },
    resolving: {
        color: 'var(--jp-rucio-yellow-color)'
    },
    injecting: {
        color: '#c0ca33'
    },
    rucioIcon: {
        marginRight: '4px',
        marginTop: '2px'
    },
    statusIcon: {
        fontSize: '16px',
        marginRight: '4px'
    },
    readyIcon: {
        extend: 'statusIcon',
        color: 'var(--jp-success-color0)'
    },
    failedIcon: {
        extend: 'statusIcon',
        color: 'var(--jp-error-color1)'
    },
    resolvingIcon: {
        extend: 'statusIcon',
        color: 'var(--jp-rucio-yellow-color)'
    },
    pendingInjectionIcon: {
        extend: 'statusIcon',
        color: '#c0ca33'
    }
});
const Panel = ({ notebookPanel, onClick }) => {
    const classes = useStyles();
    const notebookResolveStatusStore = useNotebookResolveStatusStore();
    const notebookResolveStatus = notebookResolveStatusStore[notebookPanel.id];
    const statuses = notebookResolveStatus ? Object.keys(notebookResolveStatus).map(k => notebookResolveStatus[k]) : null;
    const computeSummarizedStatus = (statuses) => {
        if (!statuses) {
            return null;
        }
        else if (statuses.length === 0) {
            return 'NOT_RESOLVED';
        }
        else if (statuses.includes('FAILED')) {
            return 'FAILED';
        }
        else if (statuses.includes('NOT_RESOLVED')) {
            return 'NOT_RESOLVED';
        }
        else if (statuses.includes('RESOLVING')) {
            return 'RESOLVING';
        }
        else if (statuses.includes('PENDING_INJECTION')) {
            return 'PENDING_INJECTION';
        }
        else {
            return 'READY';
        }
    };
    const summmarizedStatus = useMemo(() => computeSummarizedStatus(statuses), [notebookResolveStatus]);
    return (React.createElement(React.Fragment, null, !!summmarizedStatus && (React.createElement("div", { className: classes.main, onClick: onClick },
        React.createElement(ResolverStatusIcon, { status: summmarizedStatus }),
        summmarizedStatus === 'NOT_RESOLVED' && React.createElement("span", null, "Attach Variables"),
        summmarizedStatus === 'FAILED' && React.createElement("span", { className: classes.failed }, "Failed to Attach"),
        summmarizedStatus === 'RESOLVING' && React.createElement("span", { className: classes.resolving }, "Resolving"),
        summmarizedStatus === 'PENDING_INJECTION' && React.createElement("span", { className: classes.injecting }, "Attaching"),
        summmarizedStatus === 'READY' && React.createElement("span", { className: classes.ready }, "Ready")))));
};
const ResolverStatusIcon = ({ status }) => {
    const classes = useStyles();
    switch (status) {
        case 'RESOLVING':
            return React.createElement(Spinning, { className: `${classes.resolvingIcon} material-icons` }, "hourglass_top");
        case 'PENDING_INJECTION':
            return React.createElement(Spinning, { className: `${classes.pendingInjectionIcon} material-icons` }, "hourglass_top");
        case 'READY':
            return React.createElement("i", { className: `${classes.readyIcon} material-icons` }, "check_circle");
        case 'FAILED':
            return React.createElement("i", { className: `${classes.failedIcon} material-icons` }, "cancel");
        default:
            return React.createElement(rucioIcon.react, { tag: "span", className: classes.rucioIcon, width: "16px", height: "16px" });
    }
};
const PANEL_CLASS = 'jp-RucioExtensionInjectToolbar';
export class InjectNotebookToolbar extends VDomRenderer {
    constructor(options) {
        super();
        super.addClass(PANEL_CLASS);
        this.options = options;
    }
    render() {
        const { notebookPanel, onClick } = this.options;
        return React.createElement(Panel, { notebookPanel: notebookPanel, onClick: onClick });
    }
}
