from .....Internal.Core import Core
from .....Internal.CommandsGroup import CommandsGroup
from .....Internal.StructBase import StructBase
from .....Internal.ArgStruct import ArgStruct
from .....Internal.RepeatedCapability import RepeatedCapability
from ..... import repcap


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Path:
	"""Path commands group definition. 1 total commands, 0 Sub-groups, 1 group commands
	Repeated Capability: Path, default value after init: Path.Path1"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._base = CommandsGroup("path", core, parent)
		self._base.rep_cap = RepeatedCapability(self._base.group_name, 'repcap_path_get', 'repcap_path_set', repcap.Path.Path1)

	def repcap_path_set(self, enum_value: repcap.Path) -> None:
		"""Repeated Capability default value numeric suffix.
		This value is used, if you do not explicitely set it in the child set/get methods, or if you leave it to Path.Default
		Default value after init: Path.Path1"""
		self._base.set_repcap_enum_value(enum_value)

	def repcap_path_get(self) -> repcap.Path:
		"""Returns the current default repeated capability for the child set/get methods"""
		# noinspection PyTypeChecker
		return self._base.get_repcap_enum_value()

	# noinspection PyTypeChecker
	class PathStruct(StructBase):
		"""Structure for setting input parameters. Fields: \n
			- Pep: float: No parameter help available
			- Level: float: No parameter help available"""
		__meta_args_list = [
			ArgStruct.scalar_float('Pep'),
			ArgStruct.scalar_float('Level')]

		def __init__(self):
			StructBase.__init__(self, self)
			self.Pep: float = None
			self.Level: float = None

	def set(self, structure: PathStruct, secondaryCompCarrier=repcap.SecondaryCompCarrier.Default, path=repcap.Path.Default) -> None:
		"""SCPI: CONFigure:LTE:SIGNaling<instance>:IQIN:SCC<Carrier>:PATH<n> \n
		Snippet: driver.configure.iqIn.scc.path.set(value = [PROPERTY_STRUCT_NAME](), secondaryCompCarrier = repcap.SecondaryCompCarrier.Default, path = repcap.Path.Default) \n
		No command help available \n
			:param structure: for set value, see the help for PathStruct structure arguments.
			:param secondaryCompCarrier: optional repeated capability selector. Default value: CC1 (settable in the interface 'Scc')
			:param path: optional repeated capability selector. Default value: Path1 (settable in the interface 'Path')"""
		secondaryCompCarrier_cmd_val = self._base.get_repcap_cmd_value(secondaryCompCarrier, repcap.SecondaryCompCarrier)
		path_cmd_val = self._base.get_repcap_cmd_value(path, repcap.Path)
		self._core.io.write_struct(f'CONFigure:LTE:SIGNaling<Instance>:IQIN:SCC{secondaryCompCarrier_cmd_val}:PATH{path_cmd_val}', structure)

	def get(self, secondaryCompCarrier=repcap.SecondaryCompCarrier.Default, path=repcap.Path.Default) -> PathStruct:
		"""SCPI: CONFigure:LTE:SIGNaling<instance>:IQIN:SCC<Carrier>:PATH<n> \n
		Snippet: value: PathStruct = driver.configure.iqIn.scc.path.get(secondaryCompCarrier = repcap.SecondaryCompCarrier.Default, path = repcap.Path.Default) \n
		No command help available \n
			:param secondaryCompCarrier: optional repeated capability selector. Default value: CC1 (settable in the interface 'Scc')
			:param path: optional repeated capability selector. Default value: Path1 (settable in the interface 'Path')
			:return: structure: for return value, see the help for PathStruct structure arguments."""
		secondaryCompCarrier_cmd_val = self._base.get_repcap_cmd_value(secondaryCompCarrier, repcap.SecondaryCompCarrier)
		path_cmd_val = self._base.get_repcap_cmd_value(path, repcap.Path)
		return self._core.io.query_struct(f'CONFigure:LTE:SIGNaling<Instance>:IQIN:SCC{secondaryCompCarrier_cmd_val}:PATH{path_cmd_val}?', self.__class__.PathStruct())

	def clone(self) -> 'Path':
		"""Clones the group by creating new object from it and its whole existing sub-groups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = Path(self._core, self._base.parent)
		self._base.synchronize_repcaps(new_group)
		return new_group
