"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeFunction = exports.InvocationType = exports.LogType = exports.AwsApiCall = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const assertions_1 = require("./assertions");
const common_1 = require("./common");
const providers_1 = require("./providers");
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
class AwsApiCall extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.flattenResponse = 'false';
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_AwsApiCallProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AwsApiCall);
            }
            throw error;
        }
        this.provider = new providers_1.AssertionsProvider(this, 'SdkProvider');
        this.provider.addPolicyStatementFromSdkCall(props.service, props.api);
        this.name = `${props.service}${props.api}`;
        this.sdkCallResource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: this.provider.serviceToken,
            properties: {
                service: props.service,
                api: props.api,
                parameters: this.provider.encode(props.parameters),
                flattenResponse: aws_cdk_lib_1.Lazy.string({ produce: () => this.flattenResponse }),
                salt: Date.now().toString(),
            },
            resourceType: `${providers_1.SDK_RESOURCE_TYPE_PREFIX}${this.name}`,
        });
        // Needed so that all the policies set up by the provider should be available before the custom resource is provisioned.
        this.sdkCallResource.node.addDependency(this.provider);
    }
    getAtt(attributeName) {
        this.flattenResponse = 'true';
        return this.sdkCallResource.getAtt(`apiCallResponse.${attributeName}`);
    }
    getAttString(attributeName) {
        this.flattenResponse = 'true';
        return this.sdkCallResource.getAttString(`apiCallResponse.${attributeName}`);
    }
    expect(expected) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_ExpectedResult(expected);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.expect);
            }
            throw error;
        }
        new assertions_1.EqualsAssertion(this, `AssertEquals${this.name}`, {
            expected,
            actual: common_1.ActualResult.fromCustomResource(this.sdkCallResource, 'apiCallResponse'),
        });
    }
    assertAtPath(path, expected) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_ExpectedResult(expected);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.assertAtPath);
            }
            throw error;
        }
        new assertions_1.EqualsAssertion(this, `AssertEquals${this.name}`, {
            expected,
            actual: common_1.ActualResult.fromAwsApiCall(this, path),
        });
    }
}
exports.AwsApiCall = AwsApiCall;
_a = JSII_RTTI_SYMBOL_1;
AwsApiCall[_a] = { fqn: "@aws-cdk/integ-tests-alpha.AwsApiCall", version: "2.36.0-alpha.0" };
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
var LogType;
(function (LogType) {
    /**
     * The log messages are not returned in the response
     */
    LogType["NONE"] = "None";
    /**
     * The log messages are returned in the response
     */
    LogType["TAIL"] = "Tail";
})(LogType = exports.LogType || (exports.LogType = {}));
/**
 * The type of invocation. Default is REQUEST_RESPONE
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONE"] = "RequestResponse";
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
/**
 * An AWS Lambda Invoke function API call.
 * Use this istead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
class LambdaInvokeFunction extends AwsApiCall {
    constructor(scope, id, props) {
        super(scope, id, {
            api: 'invoke',
            service: 'Lambda',
            parameters: {
                FunctionName: props.functionName,
                InvocationType: props.invocationType,
                LogType: props.logType,
                Payload: props.payload,
            },
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_LambdaInvokeFunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LambdaInvokeFunction);
            }
            throw error;
        }
        const stack = aws_cdk_lib_1.Stack.of(this);
        // need to give the assertion lambda permission to invoke
        new aws_cdk_lib_1.CfnResource(this, 'Invoke', {
            type: 'AWS::Lambda::Permission',
            properties: {
                Action: 'lambda:InvokeFunction',
                FunctionName: props.functionName,
                Principal: this.provider.handlerRoleArn,
            },
        });
        // the api call is 'invoke', but the permission is 'invokeFunction'
        // so need to handle it specially
        this.provider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                service: 'lambda',
                resource: 'function',
                arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                resourceName: props.functionName,
            })]);
    }
}
exports.LambdaInvokeFunction = LambdaInvokeFunction;
_b = JSII_RTTI_SYMBOL_1;
LambdaInvokeFunction[_b] = { fqn: "@aws-cdk/integ-tests-alpha.LambdaInvokeFunction", version: "2.36.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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