import { Reference } from 'aws-cdk-lib';
import { Construct, IConstruct } from 'constructs';
import { ExpectedResult } from './common';
import { AssertionsProvider } from './providers';
/**
 * Interface for creating a custom resource that will perform
 * an API call using the AWS SDK
 */
export interface IAwsApiCall extends IConstruct {
    /**
     * access the AssertionsProvider. This can be used to add additional IAM policies
     * the the provider role policy
     *
     * @example
     * declare const apiCall: AwsApiCall;
     * apiCall.provider.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: ['s3:GetObject'],
     *   Resource: ['*'],
     * });
     */
    readonly provider: AssertionsProvider;
    /**
     * Returns the value of an attribute of the custom resource of an arbitrary
     * type. Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt`. Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getAttString` for string attributes.
     */
    getAtt(attributeName: string): Reference;
    /**
     * Returns the value of an attribute of the custom resource of type string.
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt` encoded as a string.
     */
    getAttString(attributeName: string): string;
    /**
     * Assert that the ExpectedResult is equal
     * to the result of the AwsApiCall
     *
     * @example
     * declare const integ: IntegTest;
     * const invoke = integ.assertions.invokeFunction({
     *   functionName: 'my-func',
     * });
     * invoke.expect(ExpectedResult.objectLike({ Payload: 'OK' }));
     */
    expect(expected: ExpectedResult): void;
    /**
     * Assert that the ExpectedResult is equal
     * to the result of the AwsApiCall at the given path.
     *
     * For example the SQS.receiveMessage api response would look
     * like:
     *
     * If you wanted to assert the value of `Body` you could do
     *
     * @example
     * const actual = {
     *   Messages: [{
     *     MessageId: '',
     *     ReceiptHandle: '',
     *     MD5OfBody: '',
     *     Body: 'hello',
     *     Attributes: {},
     *     MD5OfMessageAttributes: {},
     *     MessageAttributes: {}
     *   }]
     * };
     *
     *
     * declare const integ: IntegTest;
     * const message = integ.assertions.awsApiCall('SQS', 'receiveMessage');
     *
     * message.assertAtPath('Messages.0.Body', ExpectedResult.stringLikeRegexp('hello'));
     */
    assertAtPath(path: string, expected: ExpectedResult): void;
}
/**
 * Options to perform an AWS JavaScript V2 API call
 */
export interface AwsApiCallOptions {
    /**
     * The AWS service, i.e. S3
     */
    readonly service: string;
    /**
     * The api call to make, i.e. getBucketLifecycle
     */
    readonly api: string;
    /**
     * Any parameters to pass to the api call
     *
     * @default - no parameters
     */
    readonly parameters?: any;
}
/**
 * Options for creating an SDKQuery provider
 */
export interface AwsApiCallProps extends AwsApiCallOptions {
}
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
export declare class AwsApiCall extends Construct implements IAwsApiCall {
    private readonly sdkCallResource;
    private flattenResponse;
    private readonly name;
    readonly provider: AssertionsProvider;
    constructor(scope: Construct, id: string, props: AwsApiCallProps);
    getAtt(attributeName: string): Reference;
    getAttString(attributeName: string): string;
    expect(expected: ExpectedResult): void;
    assertAtPath(path: string, expected: ExpectedResult): void;
}
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
export declare enum LogType {
    /**
     * The log messages are not returned in the response
     */
    NONE = "None",
    /**
     * The log messages are returned in the response
     */
    TAIL = "Tail"
}
/**
 * The type of invocation. Default is REQUEST_RESPONE
 */
export declare enum InvocationType {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    EVENT = "Event",
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    REQUEST_RESPONE = "RequestResponse",
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    DRY_RUN = "DryRun"
}
/**
 * Options to pass to the Lambda invokeFunction API call
 */
export interface LambdaInvokeFunctionProps {
    /**
     * The name of the function to invoke
     */
    readonly functionName: string;
    /**
     * The type of invocation to use
     *
     * @default InvocationType.REQUEST_RESPONE
     */
    readonly invocationType?: InvocationType;
    /**
     * Whether to return the logs as part of the response
     *
     * @default LogType.NONE
     */
    readonly logType?: LogType;
    /**
     * Payload to send as part of the invoke
     *
     * @default - no payload
     */
    readonly payload?: string;
}
/**
 * An AWS Lambda Invoke function API call.
 * Use this istead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
export declare class LambdaInvokeFunction extends AwsApiCall {
    constructor(scope: Construct, id: string, props: LambdaInvokeFunctionProps);
}
