"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchemaTransformer = void 0;
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_dynamodb_transformer_1 = require("graphql-dynamodb-transformer");
const graphql_connection_transformer_1 = require("graphql-connection-transformer");
const graphql_key_transformer_1 = require("graphql-key-transformer");
const graphql_versioned_transformer_1 = require("graphql-versioned-transformer");
const graphql_auth_transformer_1 = require("graphql-auth-transformer");
// Import this way because FunctionTransformer.d.ts types were throwing an eror. And we didn't write this package so hope for the best :P
const { FunctionTransformer } = require('graphql-function-transformer');
const cdk_transformer_1 = require("./cdk-transformer");
const path_1 = require("path");
const fs = require("fs");
class SchemaTransformer {
    constructor(props) {
        this.resolvers = {};
        this.schemaPath = props.schemaPath || './schema.graphql';
        this.outputPath = props.outputPath || './appsync';
        this.isSyncEnabled = props.syncEnabled || false;
        // TODO: Make this mo betta
        this.authTransformerConfig = {
            authConfig: {
                defaultAuthentication: {
                    authenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    userPoolConfig: {
                        userPoolId: '12345xyz'
                    }
                },
                additionalAuthenticationProviders: [
                    {
                        authenticationType: 'API_KEY',
                        apiKeyConfig: {
                            description: 'Testing',
                            apiKeyExpirationDays: 100
                        }
                    },
                    {
                        authenticationType: 'AWS_IAM'
                    }
                ]
            }
        };
    }
    transform() {
        var _a, _b;
        let transformConfig = this.isSyncEnabled ? this.loadConfigSync() : {};
        // Note: This is not exact as we are omitting the @searchable transformer.
        const transformer = new graphql_transformer_core_1.GraphQLTransform({
            transformConfig: transformConfig,
            transformers: [
                new graphql_dynamodb_transformer_1.DynamoDBModelTransformer(),
                new graphql_versioned_transformer_1.VersionedModelTransformer(),
                new FunctionTransformer(),
                new graphql_key_transformer_1.KeyTransformer(),
                new graphql_connection_transformer_1.ModelConnectionTransformer(),
                new graphql_auth_transformer_1.ModelAuthTransformer(this.authTransformerConfig),
                new cdk_transformer_1.default(),
            ]
        });
        const schema = fs.readFileSync(this.schemaPath);
        const cfdoc = transformer.transform(schema.toString());
        // TODO: Get Unauth Role and Auth Role policies for authorization stuff
        this.authRolePolicy = ((_a = cfdoc.rootStack.Resources) === null || _a === void 0 ? void 0 : _a.AuthRolePolicy01) || undefined;
        this.unauthRolePolicy = ((_b = cfdoc.rootStack.Resources) === null || _b === void 0 ? void 0 : _b.UnauthRolePolicy01) || undefined;
        this.writeSchema(cfdoc.schema);
        this.writeResolversToFile(cfdoc.resolvers);
        this.outputs = cfdoc.rootStack.Outputs;
        return this.outputs;
    }
    getResolvers() {
        const statements = ['Query', 'Mutation', 'Subscription'];
        const resolversDirPath = path_1.normalize('./appsync/resolvers');
        if (fs.existsSync(resolversDirPath)) {
            const files = fs.readdirSync(resolversDirPath);
            files.forEach(file => {
                // Example: Mutation.createChannel.response
                let args = file.split('.');
                let typeName = args[0];
                let name = args[1];
                let templateType = args[2]; // request or response
                let filepath = path_1.normalize(`${resolversDirPath}/${file}`);
                if (statements.indexOf(typeName) >= 0 || (this.outputs.NONE && this.outputs.NONE[name])) {
                    if (!this.resolvers[name]) {
                        this.resolvers[name] = {
                            typeName: typeName,
                            fieldName: name,
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers[name]['requestMappingTemplate'] = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers[name]['responseMappingTemplate'] = filepath;
                    }
                }
                else { // This is a GSI
                    if (!this.resolvers['gsi']) {
                        this.resolvers['gsi'] = {};
                    }
                    let mapName = `${typeName}${name}`;
                    if (!this.resolvers['gsi'][mapName]) {
                        this.resolvers['gsi'][mapName] = {
                            typeName: typeName,
                            fieldName: name,
                            tableName: name.charAt(0).toUpperCase() + name.slice(1)
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers['gsi'][mapName]['requestMappingTemplate'] = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers['gsi'][mapName]['responseMappingTemplate'] = filepath;
                    }
                }
            });
        }
        return this.resolvers;
    }
    writeSchema(schema) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        fs.writeFileSync(`${this.outputPath}/schema.graphql`, schema);
    }
    writeResolversToFile(resolvers) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        const resolverFolderPath = path_1.normalize(this.outputPath + '/resolvers');
        if (fs.existsSync(resolverFolderPath)) {
            const files = fs.readdirSync(resolverFolderPath);
            files.forEach(file => fs.unlinkSync(resolverFolderPath + '/' + file));
            fs.rmdirSync(resolverFolderPath);
        }
        if (!fs.existsSync(resolverFolderPath)) {
            fs.mkdirSync(resolverFolderPath);
        }
        Object.keys(resolvers).forEach((key) => {
            const resolver = resolvers[key];
            const fileName = key.replace('.vtl', '');
            const resolverFilePath = path_1.normalize(`${resolverFolderPath}/${fileName}`);
            fs.writeFileSync(resolverFilePath, resolver);
        });
    }
    /**
     * @returns {@link TransformConfig}
    */
    loadConfigSync(projectDir = 'resources') {
        // Initialize the config always with the latest version, other members are optional for now.
        let config = {
            Version: graphql_transformer_core_1.TRANSFORM_CURRENT_VERSION,
            ResolverConfig: {
                project: {
                    ConflictHandler: "OPTIMISTIC_CONCURRENCY" /* OPTIMISTIC */,
                    ConflictDetection: "VERSION"
                }
            }
        };
        const configDir = path_1.join(__dirname, '..', '..', projectDir);
        try {
            const configPath = path_1.join(configDir, graphql_transformer_core_1.TRANSFORM_CONFIG_FILE_NAME);
            const configExists = fs.existsSync(configPath);
            if (configExists) {
                const configStr = fs.readFileSync(configPath);
                config = JSON.parse(configStr.toString());
            }
            return config;
        }
        catch (err) {
            return config;
        }
    }
}
exports.SchemaTransformer = SchemaTransformer;
//# sourceMappingURL=data:application/json;base64,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