import psycopg2, os, json
from secret_manager import get_secret, delete_secret

def handler(event, context):
    '''
    Event:
    {
        'function': 'create',
        'database': 'dvl_myapp',
        'env': 'dvl',
        'appName': 'myapp',
        'schemas': ['schemas'],
        'rdsInstance': "rds-instance-2"
    }
    '''

    # Parse login info
    rds_instance_name = event['rdsInstance']
    words = rds_instance_name.split('-')
    rds_resource_name = ''.join(word.title() for word in words)
    rds_secret_name = rds_resource_name + "-master-credentials"
    rds_secret = json.loads(get_secret(rds_secret_name))
    rds_host = rds_secret['host']
    rds_user = rds_secret['username']
    rds_password = rds_secret['password']


    # Connect to instance
    conn = psycopg2.connect(f'host={rds_host} user={rds_user} password={rds_password}')
    conn.set_isolation_level(psycopg2.extensions.ISOLATION_LEVEL_AUTOCOMMIT) # run outside transaction
    cursor = conn.cursor()

    # Parse function and arguments
    function = event['function']
    cluster_name = event['clusterName']
    database = event['database']
    env = event['env']
    app_name = event['appName']
    schemas = event.get('schemas')
    
    # Get app database owner user information
    master_user = env + "-" + app_name + "-master"
    master_user_secret_name = f'{cluster_name}-{master_user}'
    master_user_secret = json.loads(get_secret(master_user_secret_name))
    master_user_password = master_user_secret['password']

    # Execute function
    if function == 'create':
        cursor.execute(f"SELECT EXISTS(SELECT datname FROM pg_catalog.pg_database WHERE datname = '{database}')")
        result = cursor.fetchone()
        db_exists = result[0]
        if db_exists:
            conn = connect_to_database(conn=conn, database=database, host=rds_host, user=master_user, password=master_user_password)
            cursor = conn.cursor()
            alter_schemas(cursor, database, schemas, cluster_name)
            return {
                "result": 'Success'
            }
        try:
            cursor.execute(f'CREATE DATABASE {database}')
        except Exception as e:
            print(f'An exception was thrown when attempting to create database with name {database}')
            raise e
        create_user(cursor, database=database, username=master_user, password=master_user_password)
        cursor.execute(f'ALTER USER "{master_user}" CREATEROLE')
        try:
            cursor.execute(f'ALTER DATABASE {database} OWNER TO "{master_user}"')
        except Exception as e:
            cursor.execute(f'DROP DATABASE {database}')
            cursor.execute(f'DROP USER "{master_user}"')
            print(f'An exception was thrown when attempting to change the the owner to the user with name {master_user} for the database with name {database}')
            raise e
        conn = connect_to_database(conn=conn, database=database, host=rds_host, user=master_user, password=master_user_password)
        cursor = conn.cursor()
        alter_schemas(cursor, database, schemas, cluster_name)

    elif function == 'delete':
        conn = connect_to_database(conn=conn, database=database, host=rds_host, user=master_user, password=master_user_password)
        cursor = conn.cursor()
        cursor.execute("SELECT schema_name FROM information_schema.schemata")
        result = [r[0] for r in cursor.fetchall()]
        conn = connect_to_database(conn=conn, database='postgres', host=rds_host, user=master_user, password=master_user_password)
        cursor = conn.cursor()
        try:
            cursor.execute(f'DROP DATABASE {database}')
        except Exception as e:
            if 'does not exist' not in str(e):
                raise e
        conn = connect_to_database(conn=conn, host=rds_host, user=rds_user, password=rds_password)
        cursor = conn.cursor()
        try:
            cursor.execute(f'DROP USER "{master_user}"')
            delete_secret(master_user_secret_name)
        except Exception as e:
            if 'does not exist' not in str(e):
                raise e
        for s in result:
            if s[0:3] not in ['dvl', 'sat', 'prd', 'prf', 'edu'] and s[0:4] != 'cicd':
                continue
            try:
                usr = s + '-owner'
                cursor.execute(f'DROP USER "{usr}"')
                usr_secret = f"{cluster_name}-{usr}"
                delete_secret(usr_secret)
            except Exception as e:
                if 'does not exist' not in str(e):
                    raise e
            try:
                usr = s + '-readonly'
                cursor.execute(f'DROP USER "{usr}"')
                usr_secret = f"{cluster_name}-{usr}"
                delete_secret(usr_secret)
            except Exception as e:
                if 'does not exist' not in str(e):
                    raise e

    else:
        raise ValueError('Invalid function type')

    return {
        "result": 'Success'
    }
    
def alter_schemas(cursor, database, schemas: list, cluster_name: str):
    try:
        for schema in schemas:
            cursor.execute(f"SELECT EXISTS(SELECT 1 FROM information_schema.schemata WHERE schema_name = '{schema}')")
            result = cursor.fetchone()
            schema_exists = result[0]
            if not schema_exists:
                cursor.execute(f'CREATE SCHEMA IF NOT EXISTS "{schema}"')

                owner_user = schema + "-owner"
                owner_user_secret_name = f'{cluster_name}-{owner_user}'
                owner_user_secret = json.loads(get_secret(owner_user_secret_name))
                owner_user_password = owner_user_secret['password']
                create_user(cursor, database=database, username=owner_user, password=owner_user_password)
                cursor.execute(f'GRANT CONNECT ON DATABASE {database} TO "{owner_user}"')
                cursor.execute(f'GRANT USAGE, CREATE ON SCHEMA "{schema}" TO "{owner_user}"')
                cursor.execute(f'GRANT SELECT, INSERT, UPDATE, DELETE ON ALL TABLES IN SCHEMA "{schema}" TO "{owner_user}"')
                cursor.execute(f'ALTER DEFAULT PRIVILEGES IN SCHEMA "{schema}" GRANT SELECT, INSERT, UPDATE, DELETE ON TABLES TO "{owner_user}"')
                cursor.execute(f'GRANT USAGE ON ALL SEQUENCES IN SCHEMA "{schema}" TO "{owner_user}"')
                cursor.execute(f'ALTER DEFAULT PRIVILEGES IN SCHEMA "{schema}" GRANT USAGE ON SEQUENCES TO "{owner_user}"')

                readonly_user = schema + "-readonly"
                readonly_user_secret_name = f'{cluster_name}-{readonly_user}'
                readonly_user_secret = json.loads(get_secret(readonly_user_secret_name))
                readonly_user_password = readonly_user_secret['password']
                create_user(cursor, database=database, username=readonly_user, password=readonly_user_password)
                cursor.execute(f'GRANT CONNECT ON DATABASE {database} TO "{readonly_user}"')
                cursor.execute(f'GRANT USAGE ON SCHEMA "{schema}" TO "{readonly_user}"')
                cursor.execute(f'GRANT SELECT ON ALL TABLES IN SCHEMA "{schema}" TO "{readonly_user}"')
                cursor.execute(f'ALTER DEFAULT PRIVILEGES IN SCHEMA "{schema}" GRANT SELECT ON TABLES TO "{readonly_user}"')
    except Exception as e:
        print('An exception has been raised. Not all schemas have been created.')
        print(e)
        raise e

def create_user(cursor, database: str, username: str, password: str):
    try:
        cursor.execute(f'SELECT EXISTS(SELECT 1 FROM pg_roles WHERE rolname=\'{username}\')')
        result = cursor.fetchone()
        user_exists = result[0]
        if not user_exists:
            cursor.execute(f'CREATE USER "{username}" WITH ENCRYPTED PASSWORD \'{password}\'')
    except Exception as e:
        print(e)
        cursor.execute(f'DROP DATABASE {database}')
        print(f'An exception was thrown when attempting to create a user with name {username}')
        raise e

def connect_to_database(conn, host: str, user: str, password: str, database: str = None):
    conn.close()
    if database:
        lowercase_database = database.lower()
        conn = psycopg2.connect(dbname=lowercase_database, host=host, user=user, password=password)
    else:
        conn = psycopg2.connect(f'host={host} user={user} password={password}')
    conn.set_isolation_level(psycopg2.extensions.ISOLATION_LEVEL_AUTOCOMMIT)
    return conn