#
#     setup_scmversion - Builds a pythonic version number based on scm tags
#                        and branches.
#
#     Copyright (C) 2019 Jorge M. Faleiro Jr.
#
#     This program is free software: you can redistribute it and/or modify
#     it under the terms of the GNU Affero General Public License as published
#     by the Free Software Foundation, either version 3 of the License, or
#     (at your option) any later version.
#
#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU Affero General Public License for more details.
#
#     You should have received a copy of the GNU Affero General Public License
#     along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os
import re
from abc import ABC, abstractmethod
from enum import Enum
from typing import Optional

from setuptools import find_packages

PARSERS = dict()
DEFAULT_SCM = 'git'

DEFAULT_VERSION_FILE = '_version'


def init():
    # python and circular import references...
    if len(PARSERS) == 0:
        from .scm.git import GitParser
        PARSERS['git'] = GitParser()


class Tags(Enum):
    RELEASE = 1
    RELEASE_BRANCH = 2
    FEATURE_BRANCH = 3
    OTHER = 4


class ScmParser(ABC):

    @abstractmethod
    def version(self, pre_commit: bool = False) -> str:  # pragma: no cover
        pass

    @abstractmethod
    def version_type(self) -> Tags:  # pragma: no cover
        pass


def parser_factory(scm: str = DEFAULT_SCM) -> ScmParser:
    """builds a version number based on information on the scm"""
    init()
    if scm not in PARSERS:
        raise ValueError(
            f"scm '{scm}' invalid (options: {[k for k in PARSERS.keys()]})")
    return PARSERS[scm]


def parsers():
    init()
    return str(list(PARSERS.keys()))


def _detect_package(exclude):
    packages = find_packages(exclude=exclude)
    packages = [p for p in packages if '.' not in p]  # remove sub-packages
    if len(packages) == 0:
        raise ValueError('no default package detected')
    elif len(packages) > 1:
        raise ValueError(
            f'multiple packages detected: {str(sorted(packages))}')
    else:
        return packages[0]


def tag_version(parser: ScmParser = None, package: Optional[str] = None,
                pre_commit: bool = False,
                file: Optional[str] = None, exclude=['test*']) -> str:
    """ based on
    https://stackoverflow.com/questions/458550/standard-way-to-embed-version-into-python-package"""  # noqa: E501

    init()

    def overwrite_file(filename):
        with open(filename, 'w') as f:
            f.write('\n')
            f.write('# automatically generated by setup_scmversion\n')
            f.write('\n')
            f.write(f'__version__ = "{version}"\n')
            f.write(f'__version_type__ = "{version_type.name}"\n')

    parser = parser if parser is not None else parser_factory()
    package = package if package is not None else _detect_package(exclude)
    file = file if file is not None else '_version'
    file_name = '%s.py' % file
    version = parser.version(pre_commit=pre_commit)
    version_type = parser.version_type()
    filename = os.path.join(package, file_name)
    overwrite_file(filename)
    return package, file_name


def _version_match(line):
    m = _version_match.re.match(line)
    return None if m is None else m[2] if m[2] is not None else m[3]


_version_match.re = re.compile(
    r"^__version__\s*\=\s*(\"([^\"]+)\"|\'([^\']+)\')"
)


def _version_type_match(line):
    m = _version_type_match.re.match(line)
    return None if m is None else m[2] if m[2] is not None else m[3]


_version_type_match.re = re.compile(
    r"^__version_type__\s*\=\s*(\"([^\"]+)\"|\'([^\']+)\')"
)


def show_tagged(package: Optional[str] = None,
                file: Optional[str] = None, exclude=['test*']) -> (str, str):
    package = package if package is not None else _detect_package(exclude)
    file = file if file is not None else '_version'
    filename = os.path.join(package, f'{file}.py')

    version = version_type = ''
    with open(filename) as f:
        for line in f:
            v, t = _version_match(line), _version_type_match(line)
            version = version if v is None else v
            version_type = version_type if t is None else t
    return version, version_type


def check_tag(parser: ScmParser = None, package: Optional[str] = None,
              file: Optional[str] = None, exclude=['test*']):
    parser = parser if parser is not None else parser_factory()
    package = package if package is not None else _detect_package(exclude)
    tagged = show_tagged(package=package, file=file, exclude=exclude)
    expected = (parser.version(), parser.version_type().name)
    assert tagged == expected, (
        f'{package}.{file}.py tagged={tagged}, expected={expected}')
