/********************************************************************
 * Obiview header file                                              *
 ********************************************************************/

/**
 * @file obiview.h
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 * @date 16 December 2015
 * @brief Header file for the OBIDMS view functions and structures.
 */


#ifndef OBIVIEW_H_
#define OBIVIEW_H_


#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <time.h>
#include <math.h>

#include "obidms.h"
#include "obidmscolumn.h"
#include "obierrno.h"
#include "hashtable.h"
#include "linked_list.h"
#include "obiblob.h"


#define OBIVIEW_NAME_MAX_LENGTH (249)   		   /**< The maximum length of an OBIDMS view name, without the extension.
                                	 	  	  	    */
#define VIEW_TYPE_MAX_LENGTH (1024)   			   /**< The maximum length of the type name of a view.
                                	 	  	  	    */
#define LINES_COLUMN_NAME "LINES"				   /**< The name of the column containing the line selections
 	 	 	 	 	 	 	 	 	 	 	 	    *   in all views.
                                	 	  	  	    */
#define VIEW_TYPE_NUC_SEQS "NUC_SEQS_VIEW"   	   /**< The type name of views based on nucleotide sequences
												    *   and their metadata.
                                	 	  	  	    */
#define NUC_SEQUENCE_COLUMN "NUC_SEQ"			   /**< The name of the column containing the nucleotide sequences
 	 	 	 	 	 	 	 	 	 	 	 	    *   in NUC_SEQS_VIEW views.
                                	 	  	  	    */
#define ID_COLUMN "ID"							   /**< The name of the column containing the sequence identifiers
 	 	 	 	 	 	 	 	 	 	 	 	    *   in NUC_SEQS_VIEW views.
                                	 	  	  	    */
#define DEFINITION_COLUMN "DEFINITION"			   /**< The name of the column containing the sequence definitions
 	 	 	 	 	 	 	 	 	 	 	 	    *   in NUC_SEQS_VIEW views.
                                	 	  	  	    */
#define QUALITY_COLUMN "QUALITY"				   /**< The name of the column containing the sequence qualities
 	 	 	 	 	 	 	 	 	 	 	 	    *   in NUC_SEQS_VIEW views.
                                	 	  	   	    */
#define REVERSE_QUALITY_COLUMN "REVERSE_QUALITY"   /**< The name of the column containing the sequence qualities
 	 	 	 	 	 	 	 	 	 	 	 	    *    of the reverse read (generated by ngsfilter, used by alignpairedend).
                                	 	  	   	    */
#define REVERSE_SEQUENCE_COLUMN "REVERSE_SEQUENCE" /**< The name of the column containing the sequence
 	 	 	 	 	 	 	 	 	 	 	 	    *    of the reverse read (generated by ngsfilter, used by alignpairedend).
                                	 	  	   	    */
#define QUALITY_COLUMN "QUALITY"				   /**< The name of the column containing the sequence qualities
 	 	 	 	 	 	 	 	 	 	 	 	    *   in NUC_SEQS_VIEW views.
                                	 	  	   	    */
#define COUNT_COLUMN "COUNT"				       /**< The name of the column containing the sequence counts
 	 	 	 	 	 	 	 	 	 	 	 	    *   in NUC_SEQS_VIEW views.
                                	 	  	  	    */
#define SCIENTIFIC_NAME_COLUMN "SCIENTIFIC_NAME"   /**< The name of the column containing the taxon scientific name.
                                	                */
#define TAXID_COLUMN "TAXID"				       /**< The name of the column containing the taxids.       TODO subtype of INT column?
                                	                */
#define MERGED_TAXID_COLUMN "MERGED_TAXID"		   /**< The name of the column containing the merged taxids information.
                                	                */
#define MERGED_PREFIX "MERGED_"		               /**< The prefix to prepend to column names when merging informations during obi uniq.
                                	                */
#define TAXID_DIST_COLUMN "TAXID_DIST"			   /**< The name of the column containing a dictionary of taxid:[list of ids] when merging informations during obi uniq.
                                	                */
#define MERGED_COLUMN "MERGED"					   /**< The name of the column containing a list of ids when merging informations during obi uniq.
                                	                */
#define ID_PREFIX "seq"						       /**< The default prefix of sequence identifiers in automatic ID columns.
                                	 	  	  	    */
#define PREDICATE_KEY "predicates"		           /**< The key used in the json-formatted view comments to store predicates.
                                	 	  	  	    */


/**
 * @brief Structure for column aliases.
 * 		  Column aliases are alternative names used to identify a column in the context of a view.
 */
typedef struct Alias_column_pair {
	Column_reference_t	column_refs;   						/**< References (name and version) of the column.
		 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	char 				alias[OBIDMS_COLUMN_MAX_NAME+1];	/**< Alias of the column in the context of a view.
	 	 	 								 	 	 	 	 */
} Alias_column_pair_t, *Alias_column_pair_p;


/**
 * @brief Structure for a closed view stored in the view file.
 * 		  Views are identified by their name.
 * 		  Once a view has been written in the view file, it can not be modified and can only be read.
 */
typedef struct Obiview_infos {
	size_t				file_size;    								/**< View file size in bytes.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	size_t				used_size;    								/**< Used size in bytes.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	time_t 		 		creation_date;    							/**< Time at which the view was created.
		 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	char   				name[OBIVIEW_NAME_MAX_LENGTH+1];    		/**< Name of the view, used to identify it.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	char				created_from[OBIVIEW_NAME_MAX_LENGTH+1];    /**< Name of the view from which that view was cloned, if it was cloned.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	char				view_type[VIEW_TYPE_MAX_LENGTH+1];			/**< Type of the view if there is one.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 *   Types existing: NUC_SEQS_VIEW.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	bool				all_lines;									/**< Whether there is a line selection associated with the view.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	Column_reference_t	line_selection;								/**< The references of the line selection associated with the view if there is one.
 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	index_t             line_count;    								/**< The number of lines in the view.
																	 */
	int 				column_count;    							/**< The number of columns in the view.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	Alias_column_pair_t column_references[MAX_NB_OPENED_COLUMNS];   /**< References (name, version and alias) for all the columns in the view.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	bool				finished;									/** Whether the view is finished and can be read.
																	 */
	char 				comments[];									/**< Comments, additional informations on the view.
 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
} Obiview_infos_t, *Obiview_infos_p;


/**
 * @brief Structure for an opened view.
 */
typedef struct Obiview {
	Obiview_infos_p     infos;											/**< A pointer on the mapped view informations.
		 	 	 	  	 	 	 	 	 	 	 	 	 	 	 	 	 */
	OBIDMS_p            dms;    										/**< A pointer on the DMS to which the view belongs.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	bool                read_only;    									/**< Whether the view is read-only or can be modified.
																	 	 */
	OBIDMS_column_p     line_selection;    								/**< A pointer on the column containing the line selection
																	 	 *   associated with the view if there is one.
																	 	 *   This line selection is read-only, and when a line from the view is read,
																	 	 *   it is this line selection that is used.
																	 	 */
	Linked_list_node_p  columns;					    				/**< Double linked chain containing the pointers on all the columns of the view.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	hashtable_p		    column_dict;									/**< Hash table storing the pairs of column names or aliases with the associated
																		 *   pointers on column pointers (OBIDMS_column_p*).
																		 */
	int			        nb_predicates; 									/**< Number of predicates to test when closing the view.
 	 	 	 	  	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
	char* 			    (**predicate_functions)(struct Obiview* view);     /**< Array of pointers on all predicate functions to test when closing the view.
	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 	 */
} Obiview_t, *Obiview_p;


/**
 * Function checking if a view (either finished or unfinished) with a given name already exists in a DMS.
 *
 * @param dms The DMS.
 * @param view_name The name of the view.
 *
 * @returns A boolean value indicating whether the view already exists or not.
 *
 * @since September 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
bool obi_view_exists(OBIDMS_p dms, const char* view_name);


/**
 * @brief Creates a new view.
 *
 * Fails if a view with the same name already exists.
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_name The unique name of the view. Can not be 'taxonomy' (used for URIs).
 * @param view_to_clone Eventually a pointer on the opened view to clone to create the new one, if there is one. NULL if not.
 * @param line_selection Eventually a pointer on a list of indexes corresponding to a line selection to use with the view to clone
 * 						 if there is one. NULL if there is no line selection or no view to clone.
 * 						 The end of the index array is flagged by a -1 value.
 * @param comments Eventually, comments to associate with the view. NULL if not.
 *
 * @returns A pointer to the newly created view structure.
 * @retval NULL if an error occurred.
 *
 * @since December 2015
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obiview_p obi_new_view(OBIDMS_p dms, const char* view_name, Obiview_p view_to_clone, index_t* line_selection, const char* comments);


/**
 * @brief Creates a new view by cloning another view written in the view file and identified by its name.
 *
 * Note : obi_new_view can clone from a pointer on an opened view, while this function will read the view to clone
 * 		  from the view file.
 *
 * Fails if a view with the same name already exists.
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_name The unique name of the new view. Can not be 'taxonomy' (used for URIs).
 * @param view_to_clone_name The name of the view to clone stored in the view file of the OBIDMS.
 * @param line_selection Eventually a pointer on a list of indexes corresponding to a line selection to use with the view to clone
 * 						 if there is one. NULL if there is no line selection or no view to clone.
 * @param comments Eventually, comments to associate with the view. NULL if not.
 *
 * @returns A pointer to the newly created view structure.
 * @retval NULL if an error occurred.
 *
 * @since December 2015
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obiview_p obi_new_view_cloned_from_name(OBIDMS_p dms, const char* view_name, const char* view_to_clone_name, index_t* line_selection, const char* comments);


/**
 * @brief Creates a new view with the type NUC_SEQ_VIEW, based on nucleotide sequences and their metadata.
 *
 * Fails if a view with the same name already exists.
 *
 * The obligatory columns specific to the view type are created too and opened:
 * 	- NUC_SEQUENCE_COLUMN where nucleotide sequences are stored
 * 	- ID_COLUMN			  where sequence identifiers are stored
 * 	- DEFINITION_COLUMN   where sequence definitions are stored
 * 	And, if quality_column is True:
 * 	- QUALITY_COLUMN   	  where sequence qualities are stored
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_name The unique name of the view. Can not be 'taxonomy' (used for URIs).
 * @param view_to_clone Eventually a pointer on the opened view to clone to create the new one, if there is one. NULL if not.
 * @param line_selection Eventually a pointer on a list of indexes corresponding to a line selection to use with the view to clone
 * 						 if there is one. NULL if there is no line selection or no view to clone.
 * @param comments Eventually, comments to associate with the view. NULL if not.
 * @param quality_column Whether or not a sequence quality column is associated with the view.
 * @param create_default_columns Whether to create the default columns associated with the view type (false when importing all columns from a view for example).
 *
 * @returns A pointer to the newly created view structure.
 * @retval NULL if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obiview_p obi_new_view_nuc_seqs(OBIDMS_p dms, const char* view_name, Obiview_p view_to_clone, index_t* line_selection, const char* comments, bool quality_column, bool create_default_columns);


/**
 * @brief Creates a new view with the type NUC_SEQ_VIEW, based on nucleotide sequences and their metadata,
 *        by cloning another NUC_SEQ_VIEW view written in the view file and identified by its name.
 *
 * Note : obi_new_view_nuc_seqs can clone from a pointer on an opened view, while this function will read the view to clone
 * 		  from the view file.
 *
 * Fails if a view with the same name already exists.
 * Fails if the view to clone doesn't have the type NUC_SEQ_VIEW.
 *
 * The obligatory columns specific to the view type are created too and opened:
 * 	- NUC_SEQUENCE_COLUMN where nucleotide sequences are stored
 * 	- ID_COLUMN			  where sequence identifiers are stored
 * 	- DEFINITION_COLUMN   where sequence definitions are stored
 * 	And, if quality_column is True:
 * 	- QUALITY_COLUMN   	  where sequence qualities are stored
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_name The unique name of the new view. Can not be 'taxonomy' (used for URIs).
 * @param view_to_clone_name The name of the view to clone stored in the view file of the OBIDMS.
 * @param line_selection Eventually a pointer on a list of indexes corresponding to a line selection to use with the view to clone
 * 						 if there is one. NULL if there is no line selection or no view to clone.
 * @param comments Eventually, comments to associate with the view. NULL if not.
 *
 * @returns A pointer to the newly created view structure.
 * @retval NULL if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obiview_p obi_new_view_nuc_seqs_cloned_from_name(OBIDMS_p dms, const char* view_name, const char* view_to_clone_name, index_t* line_selection, const char* comments);


/**
 * @brief Clones a view using a pointer on the view to clone.
 *
 * The new view has the same type as the cloned one.
 * Fails if a view with the same name already exists.
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_to_clone A pointer on the opened view to clone to create the new one.
 * @param view_name The unique name of the new view. Can not be 'taxonomy' (used for URIs).
 * @param line_selection Eventually a pointer on a list of indexes corresponding to a line selection to use with the view to clone
 * 						 if there is one. NULL if there is no line selection or no view to clone.
 * @param comments Eventually, comments to associate with the view. NULL if not.
 *
 * @returns A pointer to the newly created view structure.
 * @retval NULL if an error occurred.
 *
 * @since April 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obiview_p obi_clone_view(OBIDMS_p dms, Obiview_p view_to_clone, const char* view_name, index_t* line_selection, const char* comments);


/**
 * @brief Clones a view using the name of the view to clone.
 *
 * The new view has the same type as the cloned one.
 * Fails if a view with the same name already exists.
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_to_clone_name The name of the view to clone.
 * @param view_name The unique name of the new view. Can not be 'taxonomy' (used for URIs).
 * @param line_selection Eventually a pointer on a list of indexes corresponding to a line selection to use with the view to clone
 * 						 if there is one. NULL if there is no line selection or no view to clone.
 * @param comments Eventually, comments to associate with the view. NULL if not.
 *
 * @returns A pointer to the newly created view structure.
 * @retval NULL if an error occurred.
 *
 * @since April 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obiview_p obi_clone_view_from_name(OBIDMS_p dms, const char* view_to_clone_name, const char* view_name, index_t* line_selection, const char* comments);


/**
 * @brief Maps a view file and returns the mapped structure stored in it.
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_name The unique name identifying the view.
 * @param finished Whether the view is finished or not.
 *
 * @returns A pointer on the mapped view infos structure.
 * @retval NULL if an error occurred.
 *
 * @since June 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obiview_infos_p obi_view_map_file(OBIDMS_p dms, const char* view_name, bool finished);


/**
 * @brief Unmaps a view file.
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_infos A pointer on the mapped view infos structure.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since June 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_view_unmap_file(OBIDMS_p dms, Obiview_infos_p view_infos);


/**
 * @brief Opens a view identified by its name stored in the view file.
 *
 * When opening a view, all the columns and eventually the line selection belonging to it are opened with it.
 *
 * @warning The opened view is read-only.
 *
 * @param dms A pointer on the OBIDMS.
 * @param view_name The unique name identifying the view.
 *
 * @returns A pointer on the opened view structure.
 * @retval NULL if an error occurred.
 *
 * @since December 2015
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obiview_p obi_open_view(OBIDMS_p dms, const char* view_name);


/**
 * @brief Adds a column to a view.
 *
 * - If there is a line selection:
 * 		- The column's lines_used attribute must be at least the view's line count
 * - If there is no line selection:
 *    	- If it's the first column in the view:
 *		 	-  The view's line count is set to the column's lines_used attribute
 *  	- If it's not the first column in the view:
 *  		- The column's lines_used attribute must be equal to the view's line count
 *
 * @warning The view must be writable.
 *
 * @param view A pointer on the view.
 * @param column_name The name of the column.
 * @param version_number The version of the column if it should be opened and not created
 * 						 (if -1, the latest version is retrieved).
 * @param alias The unique name used to identify the column in the context of this view.
 * 			    If NULL or "", column_name is used.
 * @param data_type The OBIType code of the data, if the column is created.
 * @param nb_lines The number of lines to be stored (can be 0 if not known), if the column is created.
 * @param nb_elements_per_line The number of elements per line, if the column is created.
 * @param elements_names The names of the elements with ';' as separator (no terminal ';'),
 * 						 if the column is created; NULL or "" if the default names are to be used ("0\01\02\0...\0n").
 * @param elt_names_formatted Whether the separator for the elements names is ';' (false), or '\0' (true, as formatted by format_elements_names()).
 * @param tuples A boolean indicating whether the column should contain indices referring to indexed tuples.
 * @param to_eval A boolean indicating whether the column contains expressions that should be evaluated
 *                (typically OBI_STR columns containing character strings to be evaluated by Python).
 * @param indexer_name The name of the indexer if there is one associated with the column, if the column is created.
 *                     If NULL or "", the indexer name is set as the column name.
 * @param associated_column_name The name of the associated column if there is one (otherwise NULL or ""), if the column is created.
 * @param associated_column_version The version of the associated column if there is one (otherwise -1), if the column is created.
 * @param comments Optional comments associated with the column if it is created (NULL or "" if no comments associated).
 * @param create Whether the column should be created (create == true) or already exists (create == false).
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_view_add_column(Obiview_p    view,
						char*        column_name,
						obiversion_t version_number,
						const char*  alias,
						OBIType_t    data_type,
						index_t      nb_lines,
						index_t      nb_elements_per_line,
						char*        elements_names,
						bool         elt_names_formatted,
						bool         tuples,
						bool         to_eval,
						const char*  indexer_name,
						const char*  associated_column_name,
						obiversion_t associated_column_version,
						const char*  comments,
						bool         create);


/**
 * @brief Deletes a column from a view.
 *
 * @warning The view must be writable.
 *
 * @param view A pointer on the view.
 * @param column_name The name of the column that should be deleted from the view.
 * @param delete_file Whether the column file should be deleted. Use carefully re: dependencies.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_view_delete_column(Obiview_p view, const char* column_name, bool delete_file);


/**
 * @brief Gets the pointer on a column from its name in a view.
 *
 * @param view A pointer on the view.
 * @param column_name The name of the column in the view.
 *
 * @returns A pointer on the column.
 * @retval NULL if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
OBIDMS_column_p obi_view_get_column(Obiview_p view, const char* column_name);


/**
 * @brief Checks if a column exists in a view.
 *
 * @param view A pointer on the view.
 * @param column_name The name of the column in the view.
 *
 * @returns A boolean indicating whether or not the column exists in the view.
 *
 * @since July 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
bool obi_view_column_exists(Obiview_p view, const char* column_name);


/**
 * @brief Gets the pointer on the pointer on a column from its name in a view.
 *
 * Note: This is used to replace old cloned columns with new ones in views in layers above the C layer.
 *
 * @param view A pointer on the view.
 * @param column_name The name of the column in the view.
 *
 * @returns A pointer on the pointer on the column.
 * @retval NULL if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
OBIDMS_column_p* obi_view_get_pointer_on_column_in_view(Obiview_p view, const char* column_name);


/**
 * @brief Changes the name that identifies a column in the context of a view.
 *
 * In the context of a view, each column is identified by a name that is unique in this view.
 *
 * @warning The view must be writable.
 *
 * @param view A pointer on the view.
 * @param current_name The current name that identifies the column in this view.
 * @param alias The new name that should be used to identify the column in this view.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since July 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_view_create_column_alias(Obiview_p view, const char* current_name, const char* alias);


/**
 * @brief Returns the informations of a view with a human readable format (view name, date created, line count, column informations, comments).
 *
 * @warning The returned pointer has to be freed by the caller.
 *
 * @param column A pointer on a view.
 * @param detailed Whether the informations should contain view comments.
 *
 * @returns A pointer on a character array where the formatted view informations are stored.
 * @retval NULL if an error occurred.
 *
 * @since September 2020
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_view_formatted_infos(Obiview_p view, bool detailed);


/**
 * @brief Returns the informations of a view with a human readable format on one line (view name, date created, line count).
 *
 * @warning The returned pointer has to be freed by the caller.
 *
 * @param column A pointer on a view.
 *
 * @returns A pointer on a character array where the formatted view informations are stored.
 * @retval NULL if an error occurred.
 *
 * @since September 2020
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_view_formatted_infos_one_line(Obiview_p view);


/**
 * @brief Internal function writing new comments in a view file.
 *
 * The new comments replace the pre-existing ones.
 * The view file is enlarged if necessary.
 *
 * @param view A pointer on the view.
 * @param comments The new comments that should be written.
 *
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_view_write_comments(Obiview_p view, const char* comments);


/**
 * @brief Adds comments to a view file.
 *
 * This reads the comments in the JSON format and adds the key value pair.
 * If the key already exists, the value format is turned into an array and the new value is appended
 * if it is not already in the array.
 *
 * @param column A pointer on an OBIDMS column.
 * @param key The key.
 * @param value The value associated with the key.
 *
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since August 2018
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_view_add_comment(Obiview_p view, const char* key, const char* value);


/**
 * @brief Closes an opened view, and saves it if it is not read-only (meaning it is not already saved in the view file).
 *
 * The predicates associated with the view are checked before saving it, and -1 is returned if at
 * least one predicate is not respected.
 *
 * @param view A pointer on the view.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_save_and_close_view(Obiview_p view);


/**
 * @brief Goes through all the view files of a DMS and checks for views that have
 *        not been flagged as finished (file extension renamed from '.obiview_unfinished'
 *        to '.obiview' and finished boolean set to true in the file, done by finish_view()).
 *
 * @param dms A pointer on an OBIDMS.
 *
 * @returns A boolean indicating whether unfinished views were found.
 *
 * @since September 2019
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_dms_has_unfinished_views(OBIDMS_p dms);


/**
 * @brief Goes through all the view files of a DMS and deletes views that have
 *        not been flagged as finished (file extension renamed from '.obiview_unfinished'
 *        to '.obiview' and finished boolean set to true in the file, done by finish_view()).
 *
 * @param dms A pointer on an OBIDMS.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since October 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_clean_unfinished_views(OBIDMS_p dms);


/**
 * @brief Frees and deletes an opened, writable view and the columns it created.
 *
 * The view and column files are deleted and the structures are freed.
 *
 * @warning The view pointer becomes invalid, as well as the pointers on
 *          the columns created by the view.
 *
 * @param view A pointer on the writable view to rollback.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since October 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_rollback_view(Obiview_p view);


/**
 * @brief Deletes a finished view file.
 *
 * Only the view informations file is deleted.
 * This function should only be used to delete 'temporary views' not meant to appear in the DMS history.
 *
 * @warning The view needs to be closed. The function checks that the view exists and is finished, but not that it is closed.
 *          If it was not closed, memory errors will be generated when the system tries to read the deleted file.
 *          // TODO maybe take opened view as argument in order to close it? but doesn't guarantee that it isn't open elsewhere.
 *
 * @param dms A pointer on the DMS to which the view belongs.
 * @param view_name The name of the view to delete.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since October 2018
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_delete_view(OBIDMS_p dms, const char* view_name);


/**
 * @brief Creates an OBI_INT column with the line count of the view it belongs to, and sets all lines to 1.
 *
 * @warning The number of lines set corresponds to the line count of the view.
 *
 * @param view A pointer on the view.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since July 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_create_auto_count_column(Obiview_p view);


/**
 * @brief Creates an OBI_STR column with the line count of the view it belongs to, and sets all lines with automatic identifiers
 *        with the form prefix_index.
 *
 * @warning The number of lines set corresponds to the line count of the view.
 *
 * @param view A pointer on the view.
 * @param prefix The prefix of automatic ids. The ids have the form prefix_index. If NULL, the default ID_PREFIX is used.
 *
 * @returns A value indicating the success of the operation.
 * @retval 0 if the operation was successfully completed.
 * @retval -1 if an error occurred.
 *
 * @since July 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_create_auto_id_column(Obiview_p view, const char* prefix);


/**
 * @brief Recovers an obiblob from an OBIDMS column containing indices referring to obiblobs,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered obiblob.
 * @retval OBIBlob_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obi_blob_p obi_get_blob_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers an obiblob from an OBIDMS column containing indices referring to obiblobs,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered obiblob.
 * @retval OBIBlob_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obi_blob_p obi_get_blob_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers an obiblob from an OBIDMS column containing indices referring to obiblobs,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered obiblob.
 * @retval OBIBlob_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obi_blob_p obi_get_blob_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name);


/**
 * @brief Recovers an obiblob from an OBIDMS column containing indices referring to obiblobs,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered obiblob.
 * @retval OBIBlob_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
Obi_blob_p obi_get_blob_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_BOOL,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_bool_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, obibool_t value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_BOOL,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_bool_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, obibool_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_BOOL,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIBool_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obibool_t obi_get_bool_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_BOOL,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIBool_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obibool_t obi_get_bool_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_BOOL,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_bool_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, obibool_t value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_BOOL,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_bool_with_elt_name_and_column_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, obibool_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_BOOL,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIBool_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obibool_t obi_get_bool_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_BOOL,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIBool_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obibool_t obi_get_bool_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_CHAR,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_char_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, obichar_t value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_CHAR,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_char_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, obichar_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_CHAR,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIChar_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obichar_t obi_get_char_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_CHAR,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIChar_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obichar_t obi_get_char_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_CHAR,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_char_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, obichar_t value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_CHAR,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_char_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, obichar_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_CHAR,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIChar_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obichar_t obi_get_char_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_CHAR,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIChar_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obichar_t obi_get_char_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_FLOAT,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_float_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, obifloat_t value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_FLOAT,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_float_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, obifloat_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_FLOAT,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIFloat_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obifloat_t obi_get_float_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_FLOAT,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIFloat_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obifloat_t obi_get_float_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_FLOAT,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_float_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, obifloat_t value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_FLOAT,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_float_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, obifloat_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_FLOAT,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIFloat_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obifloat_t obi_get_float_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_FLOAT,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIFloat_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obifloat_t obi_get_float_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_INT,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_int_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, obiint_t value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_INT,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_int_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, obiint_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_INT,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIInt_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obiint_t obi_get_int_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_INT,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIInt_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obiint_t obi_get_int_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_INT,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_int_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, obiint_t value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_INT,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_int_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, obiint_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_INT,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIInt_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obiint_t obi_get_int_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_INT,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIInt_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
obiint_t obi_get_int_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name);


/**
 * @brief Sets a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * This function is for qualities in the character string format.
 *
 * @warning Pointers returned by obi_open_column() don't allow writing.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer as returned by obi_create_column() or obi_clone_column().
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set, in the character string format.
 * @param offset The ASCII base of sequence quality, used to convert sequence qualities from characters to integers
 *				 and the other way around.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since May 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_qual_char_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, const char* value, int offset);


/**
 * @brief Sets a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * This function is for qualities in the character string format.
 *
 * @warning Pointers returned by obi_open_column() don't allow writing.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set, in the character string format.
 * @param offset The ASCII base of sequence quality, used to convert sequence qualities from characters to integers
 *				 and the other way around.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_qual_char_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, const char* value, int offset);


/**
 * @brief Sets a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * This function is for qualities in the integer array format.
 *
 * @warning Pointers returned by obi_open_column() don't allow writing.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer as returned by obi_create_column() or obi_clone_column().
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set, in the integer array format.
 * @param value_length The length of the integer array.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since May 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_qual_int_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, const uint8_t* value, int value_length);


/**
 * @brief Sets a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * This function is for qualities in the integer array format.
 *
 * @warning Pointers returned by obi_open_column() don't allow writing.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set, in the integer array format.
 * @param value_length The length of the integer array.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_qual_int_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, const uint8_t* value, int value_length);


/**
 * @brief Recovers a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * This function returns quality scores in the character string format.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer as returned by obi_create_column().
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 * @param offset The ASCII base of sequence quality, used to convert sequence qualities from characters to integers
 *				 and the other way around.
 *
 * @returns The recovered value, in the character string format.
 * @retval OBIQual_char_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since May 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_get_qual_char_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, int offset);


/**
 * @brief Recovers a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * This function returns quality scores in the character string format.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 * @param offset The ASCII base of sequence quality, used to convert sequence qualities from characters to integers
 *				 and the other way around.
 *
 * @returns The recovered value, in the character string format.
 * @retval OBIQual_char_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_get_qual_char_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, int offset);


/**
 * @brief Recovers a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * This function returns quality scores in the integer array format.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer as returned by obi_create_column().
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 * @param value_length A pointer on an integer to store the length of the integer array recovered.
 *
 * @returns The recovered value, in the integer array format.
 * @retval OBIQual_int_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since May 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const uint8_t* obi_get_qual_int_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, int* value_length);


/**
 * @brief Recovers a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * This function returns quality scores in the integer array format.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 * @param value_length A pointer on an integer to store the length of the integer array recovered.
 *
 * @returns The recovered value, in the integer array format.
 * @retval OBIQual_int_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const uint8_t* obi_get_qual_int_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, int* value_length);


/**
 * @brief Sets a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * This function is for quality scores in the character string format.
 *
 * @warning Pointers returned by obi_open_column() don't allow writing.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer as returned by obi_create_column() or obi_clone_column().
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set, in the character string format.
 * @param offset The ASCII base of sequence quality, used to convert sequence qualities from characters to integers
 *				 and the other way around.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since May 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_qual_char_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, const char* value, int offset);


/**
 * @brief Sets a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * This function is for quality scores in the character string format.
 *
 * @warning Pointers returned by obi_open_column() don't allow writing.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set, in the character string format.
 * @param offset The ASCII base of sequence quality, used to convert sequence qualities from characters to integers
 *				 and the other way around.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_qual_char_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, const char* value, int offset);


/**
 * @brief Sets a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * This function is for quality scores in the integer array format.
 *
 * @warning Pointers returned by obi_open_column() don't allow writing.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer as returned by obi_create_column() or obi_clone_column().
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set, in the integer format.
 * @param value_length The length of the integer array.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since May 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_qual_int_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, const uint8_t* value, int value_length);


/**
 * @brief Sets a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * This function is for quality scores in the integer array format.
 *
 * @warning Pointers returned by obi_open_column() don't allow writing.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set, in the integer format.
 * @param value_length The length of the integer array.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_qual_int_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, const uint8_t* value, int value_length);


/**
 * @brief Recovers a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * This function returns quality scores in the character string format.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer as returned by obi_create_column() or obi_clone_column().
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 * @param offset The ASCII base of sequence quality, used to convert sequence qualities from characters to integers
 *				 and the other way around.
 *
 * @returns The recovered value, in the character string format.
 * @retval OBIQual_char_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since May 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_get_qual_char_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, int offset);


/**
 * @brief Recovers a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * This function returns quality scores in the character string format.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 * @param offset The ASCII base of sequence quality, used to convert sequence qualities from characters to integers
 *				 and the other way around.
 *
 * @returns The recovered value, in the character string format.
 * @retval OBIQual_char_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_get_qual_char_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, int offset);


/**
 * @brief Recovers a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * This function returns quality scores in the integer array format.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer as returned by obi_create_column() or obi_clone_column().
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 * @param value_length A pointer on an integer to store the length of the integer array recovered.
 *
 * @returns The recovered value, in the integer format.
 * @retval OBIQual_int_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since May 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const uint8_t* obi_get_qual_int_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, int* value_length);


/**
 * @brief Recovers a value in an OBIDMS column containing data in the form of indices referring
 * 		  to sequence qualities handled by an indexer (OBI_QUAL data type),
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * This function returns quality scores in the integer array format.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 * @param value_length A pointer on an integer to store the length of the integer array recovered.
 *
 * @returns The recovered value, in the integer format.
 * @retval OBIQual_int_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const uint8_t* obi_get_qual_int_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, int* value_length);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_SEQ,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_seq_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, const char* value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_SEQ,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_seq_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, const char* value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_SEQ,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBISeq_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_get_seq_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_SEQ,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBISeq_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_get_seq_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_SEQ,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_seq_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, const char* value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_SEQ,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_seq_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, const char* value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_SEQ,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBISeq_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_get_seq_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_SEQ,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBISeq_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
char* obi_get_seq_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_STR,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_str_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, const char* value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_STR,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_str_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, const char* value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_STR,
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIStr_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const char* obi_get_str_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_STR,
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIStr_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const char* obi_get_str_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_STR,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_str_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, const char* value);


/**
 * @brief Sets a value in an OBIDMS column containing data with the type OBI_STR,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_str_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, const char* value);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_STR,
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIStr_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since February 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const char* obi_get_str_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name);


/**
 * @brief Recovers a value in an OBIDMS column containing data with the type OBI_STR,
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIStr_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since August 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const char* obi_get_str_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name);


/**
 * @brief Sets a value in an OBIDMS column containing indices (stored data type: OBI_IDX),
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 *	Note: Columns containing indices refer to data stored elsewhere, for example lines in other columns,
 *	      or data stored in indexers.
 *
 * In the case of columns referring to values stored in indexers, this allows to directly set already-known
 * indices without going through the time-consuming step of indexing the value.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_index_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx, index_t value);


/**
 * @brief Sets a value in an OBIDMS column containing indices (stored data type: OBI_IDX),
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 *	Note: Columns containing indices refer to data stored elsewhere, for example lines in other columns,
 *	      or data stored in indexers.
 *
 * In the case of columns referring to values stored in indexers, this allows to directly set already-known
 * indices without going through the time-consuming step of indexing the value.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_idx The index of the element that should be set in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_index_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx, index_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing indices (stored data type: OBI_IDX),
 *        using the index of the element in the line, and the column pointer, in the context of a view.
 *
 * This enables to recover the indexes in columns containing any type of indexed data (OBI_SEQ, OBI_STR...), for example to quickly test if
 * the values are identical (same index in the same indexer).
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIIdx_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
index_t obi_get_index_with_elt_idx_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, index_t element_idx);


/**
 * @brief Recovers a value in an OBIDMS column containing indices (stored data type: OBI_IDX),
 *        using the index of the element in the line, and the column name, in the context of a view.
 *
 * This enables to recover the indexes in columns containing any type of indexed data (OBI_SEQ, OBI_STR...), for example to quickly test if
 * the values are identical (same index in the same indexer).
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_idx The index of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIIdx_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
index_t obi_get_index_with_elt_idx_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, index_t element_idx);


/**
 * @brief Sets a value in an OBIDMS column containing indices (stored data type: OBI_IDX),
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 *	Note: Columns containing indices refer to data stored elsewhere, for example lines in other columns,
 *	      or data stored in indexers.
 *
 * In the case of columns referring to values stored in indexers, this allows to directly set already-known
 * indices without going through the time-consuming step of indexing the value.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be recovered in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_index_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name, index_t value);


/**
 * @brief Sets a value in an OBIDMS column containing indices (stored data type: OBI_IDX),
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 *	Note: Columns containing indices refer to data stored elsewhere, for example lines in other columns,
 *	      or data stored in indexers.
 *
 * In the case of columns referring to values stored in indexers, this allows to directly set already-known
 * indices without going through the time-consuming step of indexing the value.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be set.
 * @param element_name The name of the element that should be recovered in the line.
 * @param value The value that should be set.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_index_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name, index_t value);


/**
 * @brief Recovers a value in an OBIDMS column containing indices (stored data type: OBI_IDX),
 *        using the name of the element in the line, and the column pointer, in the context of a view.
 *
 * This enables to recover the indexes in columns containing any type of indexed data (OBI_SEQ, OBI_STR...), for example to quickly test if
 * the values are identical (same index in the same indexer).
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIIdx_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
index_t obi_get_index_with_elt_name_and_col_p_in_view(Obiview_p view, OBIDMS_column_p column_p, index_t line_nb, const char* element_name);


/**
 * @brief Recovers a value in an OBIDMS column containing indices (stored data type: OBI_IDX),
 *        using the name of the element in the line, and the column name, in the context of a view.
 *
 * This enables to recover the indexes in columns containing any type of indexed data (OBI_SEQ, OBI_STR...), for example to quickly test if
 * the values are identical (same index in the same indexer).
 *
 * @param view A pointer on the opened view.
 * @param column_name The alias of the column in the view.
 * @param line_nb The number of the line where the value should be recovered.
 * @param element_name The name of the element that should be recovered in the line.
 *
 * @returns The recovered value.
 * @retval OBIIdx_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since November 2016
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
index_t obi_get_index_with_elt_name_and_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const char* element_name);


/*********** FOR ARRAY COLUMNS ***********/

/**
 * @brief Sets a value in an OBIDMS column containing indices referring to indexed arrays,
 *        using the column pointer, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be set.
 * @param value The value that should be set.
 * @param elt_size The size in bits of one element.
 * @param value_length The length (number of elements) of the array to index.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since October 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_array_with_col_p_in_view(Obiview_p view, OBIDMS_column_p column, index_t line_nb, const void* value, uint8_t elt_size, int32_t value_length);


/**
 * @brief Recovers a value in an OBIDMS column containing indices referring to indexed arrays,
 *        using the column pointer, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_p A pointer on the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param value_length_p A pointer on an int where the length of the value (number of elements in the array) will be stored.
 *
 * @returns The recovered value.
 * @retval OBITuple_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since October 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const void* obi_get_array_with_col_p_in_view(Obiview_p view, OBIDMS_column_p column, index_t line_nb, int32_t* value_length_p);


/**
 * @brief Sets a value in an OBIDMS column containing indices referring to indexed arrays,
 *        using the column name, in the context of a view.
 *
 *  Note: If the column is read-only or if there is a line selection associated with the view (making columns non-writable), it is cloned.
 *
 * @param view A pointer on the opened writable view.
 * @param column_name The name of the column.
 * @param line_nb The number of the line where the value should be set.
 * @param value The value that should be set.
 * @param elt_size The size in bits of one element.
 * @param value_length The length (number of elements) of the array to index.
 *
 * @returns An integer value indicating the success of the operation.
 * @retval 0 on success.
 * @retval -1 if an error occurred.
 *
 * @since October 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
int obi_set_array_with_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, const void* value, uint8_t elt_size, int32_t value_length);


/**
 * @brief Recovers a value in an OBIDMS column containing indices referring to indexed arrays,
 *        using the column name, in the context of a view.
 *
 * @param view A pointer on the opened view.
 * @param column_name The name of the column.
 * @param line_nb The number of the line where the value should be recovered.
 * @param value_length_p A pointer on an int where the length of the value (number of elements in the array) will be stored.
 *
 * @returns The recovered value.
 * @retval OBITuple_NA the NA value of the type if an error occurred and obi_errno is set.
 *
 * @since October 2017
 * @author Celine Mercier (celine.mercier@metabarcoding.org)
 */
const void* obi_get_array_with_col_name_in_view(Obiview_p view, const char* column_name, index_t line_nb, int32_t* value_length_p);


#endif /* OBIVIEW_H_ */
