# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tracers', 'tracers.function']

package_data = \
{'': ['*']}

install_requires = \
['aioboto3==8.0.3', 'aiogqlc==1.0.5', 'more-itertools==8.4.0']

setup_kwargs = {
    'name': 'tracers',
    'version': '20.6.40129',
    'description': 'Open-Source APM (Application monitoring) project that offers you zero overhead wrappers for profiling your code execution flow',
    'long_description': "# Table of contents\n\n- [Table of contents](#table-of-contents)\n- [Project description](#project-description)\n  * [Key Features](#key-features)\n- [Quick Introduction](#quick-introduction)\n- [Installation](#installation)\n- [Examples](#examples)\n\n# Project description\n\n**Tracers** is an Open-Source **APM** (Application monitoring) project\nthat offers you minimal overhead wrappers for profiling your code execution flow\n\n```\n🛈  Finished transaction: 3.81 seconds\n\n     # Timestamp                Net              Total    Call Chain\n\n     1     0.00s     0.00s [  0.0%]     3.81s [100.0%]    ✓ async function_a\n     2     0.00s     0.10s [  2.6%]     0.10s [  2.6%]    ¦   ✓ async asyncio.tasks.sleep\n     3     0.10s     0.50s [ 13.1%]     0.50s [ 13.1%]    ¦   ✓ time.sleep\n     4     0.60s     0.00s [  0.0%]     3.21s [ 84.2%]    ¦   ✓ async function_b\n     5     0.60s     0.10s [  2.6%]     0.10s [  2.6%]    ¦   ¦   ✓ async asyncio.tasks.sleep\n     6     0.70s     0.00s [  0.0%]     0.70s [ 18.4%]    ¦   ¦   ✓ async function_c\n     7     0.70s     0.10s [  2.6%]     0.10s [  2.6%]    ¦   ¦   ¦   ✓ async asyncio.tasks.sleep\n     8     0.80s     0.50s [ 13.1%]     0.50s [ 13.1%]    ¦   ¦   ¦   ✓ time.sleep\n     9     1.31s     0.00s [  0.0%]     0.10s [  2.7%]    ¦   ¦   ¦   ✓ async function_d\n    10     1.31s     0.10s [  2.6%]     0.10s [  2.6%]    ¦   ¦   ¦   ¦   ✓ async asyncio.tasks.sleep\n    11     1.41s     2.00s [ 52.5%]     2.00s [ 52.5%]    ¦   ¦   ✓ time.sleep\n    12     3.41s     0.10s [  2.7%]     0.10s [  2.7%]    ¦   ¦   ✓ async asyncio.tasks.sleep\n    13     3.51s     0.00s [  0.0%]     0.10s [  2.7%]    ¦   ¦   ✓ async function_d\n    14     3.51s     0.10s [  2.7%]     0.10s [  2.7%]    ¦   ¦   ¦   ✓ async asyncio.tasks.sleep\n    15     3.61s     0.10s [  2.7%]     0.10s [  2.7%]    ¦   ¦   ✓ async asyncio.tasks.sleep\n    16     3.71s     0.00s [  0.0%]     0.10s [  2.7%]    ¦   ¦   ✓ async function_e\n    17     3.71s     0.10s [  2.7%]     0.10s [  2.7%]    ¦   ¦   ¦   ✓ async asyncio.tasks.sleep\n\n           Count                Net              Total    Function\n\n               3     3.00s [ 78.7%]     3.00s [ 78.7%]    ✓ time.sleep\n               8     0.81s [ 21.2%]     0.81s [ 21.2%]    ✓ async asyncio.tasks.sleep\n               1     0.00s [  0.0%]     3.21s [ 84.2%]    ✓ async function_b\n               1     0.00s [  0.0%]     0.70s [ 18.4%]    ✓ async function_c\n               1     0.00s [  0.0%]     3.81s [100.0%]    ✓ async function_a\n               2     0.00s [  0.0%]     0.20s [  5.3%]    ✓ async function_d\n               1     0.00s [  0.0%]     0.10s [  2.7%]    ✓ async function_e\n\n  Some blocks (skews) occurred in the event loop ¹\n\n  #    Timestamp     Delay\n\n     0     1.40s     2.00s\n     1     0.09s     0.50s\n     2     0.80s     0.50s\n\n  ¹ Consider reviewing them carefully to improve the overall system throughput\n```\n\n## Key Features\n\n- Allows you to instrument any **function** or **statement**\n  on **your own code** or any **third party library**\n- Handles [**asynchronous**](https://docs.python.org/3/library/asyncio.html) code\n  **out-of-the box**, no config required\n- Exposes a **high-level API**:\n  - A decorator: `@trace`, to instrument functions\n    - Example:\n      ```\n      @trace\n      def any_function(*args, **kwargs):\n          pass\n\n      @trace\n      async def other_function(*args, **kwargs):\n          pass\n      ```\n  - A function: `call` to instrument code in-line\n    - Example:\n      ```\n      call(any_function, *args, **kwargs)\n      await call(other_function, *args, **kwargs)\n      ```\n  - Sync/async cases are handled internally\n  - `call` and `@trace` are equivalent, you can choose the one that fits you best\n- It's **Thread-safe**, **Async-safe**, **Process-safe** and **Context-safe**\n  - You'll get accurate results in any scenario\n- Introduces **minimal overhead** and it's **easy to deploy**!\n  - It's a pure python implementation based on\n   [contextvars](https://docs.python.org/3/library/contextvars.html) and\n   [decorators](https://docs.python.org/3/glossary.html#term-decorator).\n   Your code runs as fast as if you were not profiling it\n- It's easy to pin-point performance problems:\n  - Gives you the total execution time in seconds and **%**\n  - Allows you to identify points in time where your **async** event loop got blocked\n- Made with love by nerds, for humans :heart:\n\n# Quick Introduction\n\nLet's start with a very basic example:\n\n```py\nimport time\nfrom dateutil.parser import parse\n\n\ndef example():\n    time.sleep(2.0)\n    your_business_logic('Sat Oct 11')\n\n\ndef your_business_logic(date: str):\n    parse(date)\n    time.sleep(1.0)\n\n\nexample()\n```\n\nTracing its flow and gathering profiling information is a matter of\ndecorating your functions:\n\n```diff\n--- a/examples/without_tracers.py\n+++ b/examples/with_tracers.py\n@@ -1,15 +1,18 @@\n import time\n from dateutil.parser import parse\n+from tracers.function import trace\n\n\n+@trace()\n def example():\n     time.sleep(2.0)\n     your_business_logic('Sat Oct 11')\n\n\n+@trace()\n def your_business_logic(date: str):\n     parse(date)\n     time.sleep(1.0)\n\n\n example()\n```\n\nIf you run it, all the functions you decorated will be traced\nand you'll have metrics of the execution flow:\n\n```\n🛈  Finished transaction: 3.00 seconds\n\n     # Timestamp                Net              Total    Call Chain\n\n     1     0.00s     2.00s [ 66.7%]     3.00s [100.0%]    ✓ example\n     2     2.00s     1.00s [ 33.3%]     1.00s [ 33.3%]    ¦   ✓ your_business_logic\n\n           Count                Net              Total    Function\n\n               1     2.00s [ 66.7%]     3.00s [100.0%]    ✓ example\n               1     1.00s [ 33.3%]     1.00s [ 33.3%]    ✓ your_business_logic\n```\n\nFrom the output you can conclude:\n- executing function *example* took a total of *3.0* seconds to complete\n- function *example* takes *66.7%* of the execution time\n- function *example encloses *100%* of your code time\n- function *example* called function: *your_business_logic*\n- function *your_business_logic* took *1.0* seconds out of the *3.0* seconds\n  the function *example* needed to complete\n- function *your_business_logic* represents *33.3%* of your execution time\n\nTracing code is not limited to your own code.\nYou can trace any **callable object** including **third party packages**,\n**Python's standard library**, and almost anything\n\nThe level of detail is up to you!\n\n\n```diff\n--- a/examples/with_tracers.py\n+++ b/examples/with_detailed_tracers.py\n@@ -1,18 +1,18 @@\n import time\n from dateutil.parser import parse\n-from tracers.function import trace\n+from tracers.function import call, trace\n\n\n @trace()\n def example():\n-    time.sleep(2.0)\n+    call(time.sleep, 2.0)\n     your_business_logic('Sat Oct 11')\n\n\n @trace()\n def your_business_logic(date: str):\n-    parse(date)\n-    time.sleep(1.0)\n+    call(parse, date)\n+    call(time.sleep, 1.0)\n\n\n example()\n```\n\n```\n\n🛈  Finished transaction: 3.00 seconds\n\n     # Timestamp                Net              Total    Call Chain\n\n     1     0.00s     0.00s [  0.0%]     3.00s [100.0%]    ✓ example\n     2     0.00s     2.00s [ 66.6%]     2.00s [ 66.6%]    ¦   ✓ time.sleep\n     3     2.00s     0.00s [  0.0%]     1.00s [ 33.4%]    ¦   ✓ your_business_logic\n     4     2.00s     0.00s [  0.0%]     0.00s [  0.0%]    ¦   ¦   ✓ dateutil.parser._parser.parse\n     5     2.00s     1.00s [ 33.3%]     1.00s [ 33.3%]    ¦   ¦   ✓ time.sleep\n\n           Count                Net              Total    Function\n\n               2     3.00s [100.0%]     3.00s [100.0%]    ✓ time.sleep\n               1     0.00s [  0.0%]     0.00s [  0.0%]    ✓ dateutil.parser._parser.parse\n               1     0.00s [  0.0%]     1.00s [ 33.4%]    ✓ your_business_logic\n               1     0.00s [  0.0%]     3.00s [100.0%]    ✓ example\n\n```\n\n# Installation\n\nWe are hosted on **PyPI**: https://pypi.org/project/tracers\n\nJust run: `pip install tracers`\nor use the package manager you like the most\n\n# Examples\n\nCheck them out in the\n[examples](https://github.com/kamadorueda/tracers/tree/master/examples)\nfolder\n",
    'author': 'Kevin Amado',
    'author_email': 'kamadorueda@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kamadorueda/tracers',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
